package ru.yandex.qe.dispenser.quartz.trigger;

import java.util.UUID;

import com.fasterxml.jackson.core.JsonProcessingException;
import org.jetbrains.annotations.Nullable;
import org.quartz.JobBuilder;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.SimpleScheduleBuilder;
import org.quartz.Trigger;
import org.quartz.TriggerBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.quartz.job.TicketUpdateJob;
import ru.yandex.qe.dispenser.quartz.job.TicketUpdateSerializationUtils;
import ru.yandex.startrek.client.model.IssueUpdate;

public class TrackerJobRunner {
    private static final Logger LOG = LoggerFactory.getLogger(TrackerJobRunner.class);

    private final Scheduler scheduler;
    private final TicketUpdateSerializationUtils ticketUpdateSerializationUtils;

    public TrackerJobRunner(final Scheduler scheduler, final TicketUpdateSerializationUtils ticketUpdateSerializationUtils) {
        this.scheduler = scheduler;
        this.ticketUpdateSerializationUtils = ticketUpdateSerializationUtils;
    }

    public Trigger createUpdateIssueTrigger(final String issueKey, @Nullable final String transition, final IssueUpdate issueUpdate) {

        final String body;
        try {
            body = ticketUpdateSerializationUtils.writeValueAsString(issueUpdate);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid issueUpdate", e);
        }

        final String name = issueKey + UUID.randomUUID().toString();
        return TriggerBuilder.newTrigger()
                .withIdentity(name, TicketUpdateJob.TRIGGER_GROUP)
                .withSchedule(SimpleScheduleBuilder.simpleSchedule()
                        .withRepeatCount(4)
                        .withIntervalInMinutes(15)
                        .withMisfireHandlingInstructionNextWithExistingCount())
                .startNow()
                .usingJobData("issueKey", issueKey)
                .usingJobData("transition", transition)
                .usingJobData("body", body)
                .build();
    }


    @Transactional(propagation = Propagation.REQUIRED)
    public boolean scheduleUpdateIssue(final String issueKey, final @Nullable String transition, final IssueUpdate bodyUpdate) {

        final Trigger trigger = createUpdateIssueTrigger(issueKey, transition, bodyUpdate);

        final JobDetail job = JobBuilder.newJob(TicketUpdateJob.class)
                .withIdentity(trigger.getJobKey())
                .requestRecovery(true)
                .build();

        try {
            scheduler.scheduleJob(job, trigger);
        } catch (SchedulerException e) {
            LOG.error("Error with scheduling QuartzTicketUpdateTrigger", e);
            return false;
        }

        return true;
    }

}
