package ru.yandex.qe.dispenser.tvm;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UncheckedIOException;
import java.net.HttpURLConnection;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.kotlin.KotlinModule;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.hitman.tvm.TvmConstants;
import ru.yandex.qe.hitman.tvm.qloud.QloudTvmService;
import ru.yandex.qe.hitman.tvm.qloud.QloudTvmServiceImpl;
import ru.yandex.qe.hitman.tvm.qloud.TvmServiceTicketInfo;
import ru.yandex.qe.hitman.tvm.qloud.TvmUserTicketInfo;

public class DeployTvmServiceImpl implements QloudTvmService {

    private static final Logger LOG = LoggerFactory.getLogger(DeployTvmServiceImpl.class);

    private final String tvmToolAddress;
    private final String tvmToolToken;
    private final String ownTvmId;

    private final URI serviceTicketCheckURI;
    private final URI userTicketCheckURI;

    private final ObjectMapper objectMapper;
    private final HttpClientBuilder httpClientBuilder;

    public DeployTvmServiceImpl(final String ownTvmId,
                                final String tvmToolAddress,
                                final String tvmToolToken) {
        this.ownTvmId = ownTvmId;
        this.tvmToolAddress = tvmToolAddress;
        this.tvmToolToken = tvmToolToken;
        this.httpClientBuilder = QloudTvmServiceImpl.getDefaultHttpClientBuilder();
        this.objectMapper = new ObjectMapper().registerModules(new KotlinModule.Builder().build(), new Jdk8Module(),
                new JavaTimeModule());

        this.serviceTicketCheckURI = getServiceTicketCheckURI(tvmToolAddress, ownTvmId);
        this.userTicketCheckURI = getUserTicketCheckURI(tvmToolAddress);
    }

    private static URI getTicketURI(final String tvmToolAddress, final String fromClientId, final String toClientId) {
        try {
            return new URIBuilder(tvmToolAddress)
                    .setPath("/tvm/tickets")
                    .addParameter("src", fromClientId)
                    .addParameter("dsts", toClientId)
                    .build();
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private static URI getServiceTicketCheckURI(final String tvmToolAddress, final String ownTvmId) {
        try {
            return new URIBuilder(tvmToolAddress)
                    .setPath("/tvm/checksrv")
                    .addParameter("dst", ownTvmId)
                    .build();
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    private static URI getUserTicketCheckURI(final String tvmToolAddress) {
        try {
            return new URIBuilder(tvmToolAddress)
                    .setPath("/tvm/checkusr")
                    .build();
        } catch (URISyntaxException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String getTicket(final String targetClientId) {
        final URI uri = getTicketURI(tvmToolAddress, ownTvmId, targetClientId);
        try (CloseableHttpClient httpClient = httpClientBuilder.build()) {
            final HttpGet method = new HttpGet(uri);
            method.setHeader("Authorization", tvmToolToken);
            try (CloseableHttpResponse response = httpClient.execute(method)) {
                final String contents = new BufferedReader(new InputStreamReader(response.getEntity().getContent()))
                        .lines()
                        .collect(Collectors.joining("\n"));
                if (response.getStatusLine().getStatusCode() != HttpURLConnection.HTTP_OK) {
                    LOG.error("Got {} with body {} from local TVM agent call {}",
                            response.getStatusLine(), contents.replaceAll("\n", ""), uri.toString());
                    throw new IllegalStateException("Got unexpected response with status '" + response.getStatusLine()
                            + "' from local TVM agent");
                }

                final Map<String, QloudTvmServiceImpl.TicketContent> result;
                try {
                    result = objectMapper.readValue(contents, new TypeReference<Map<String, QloudTvmServiceImpl.TicketContent>>() {
                    });
                } catch (IOException e) {
                    LOG.error("Failed to parse TVM agent response: " + contents.replaceAll("\n", ""), e);
                    throw new UncheckedIOException(e);
                }

                if (result.containsKey(targetClientId)) {
                    return result.get(targetClientId).getTicket();
                }

                final long targetClientIdLong = parseOrZero(targetClientId);
                for (final QloudTvmServiceImpl.TicketContent ticketContent : result.values()) {
                    if (targetClientIdLong == ticketContent.getTvmId()) {
                        return ticketContent.getTicket();
                    }
                }

                throw new IllegalArgumentException(
                        "TVM agent response does not contain ticket for destination " + targetClientId);
            }
        } catch (IOException e) {
            LOG.error(String.format("Failed to query for TVM ticket for client \"%s\"", targetClientId), e);
            throw new UncheckedIOException(e);
        }
    }

    private long parseOrZero(final String targetClientId) {
        try {
            return Long.parseLong(targetClientId);
        } catch (final NumberFormatException ignored) {
            return 0L;
        }
    }

    @Override
    public boolean isValidTarget(final String clientId) {
        return true;
    }

    @Override
    public Optional<TvmServiceTicketInfo> validateServiceTicket(final String serviceTicket) {
        final HttpGet method = new HttpGet(serviceTicketCheckURI);
        method.setHeader("Authorization", tvmToolToken);
        method.setHeader(TvmConstants.TVM_SERVICE_HEADER_NAME, serviceTicket);

        return validateTicket(method, TvmServiceTicketInfo.class);
    }

    @Override
    public Optional<TvmUserTicketInfo> validateUserTicket(final String userTicket) {
        final HttpGet method = new HttpGet(userTicketCheckURI);
        method.setHeader("Authorization", tvmToolToken);
        method.setHeader(TvmConstants.TVM_USER_HEADER_NAME, userTicket);

        return validateTicket(method, TvmUserTicketInfo.class);
    }

    private <T> Optional<T> validateTicket(final HttpGet method,
                                           final Class<T> ticketClass) {
        try (final CloseableHttpClient httpClient = httpClientBuilder.build()) {
            try (final CloseableHttpResponse response = httpClient.execute(method)) {
                final String contents = new BufferedReader(new InputStreamReader(response.getEntity().getContent()))
                        .lines()
                        .collect(Collectors.joining("\n"));
                if (response.getStatusLine().getStatusCode() != HttpURLConnection.HTTP_OK) {
                    LOG.error("Got {} with body {} from local TVM agent call {}",
                            response.getStatusLine(), contents.replaceAll("\n", ""), method.getURI().toString());
                    return Optional.empty();
                } else {
                    try {
                        return Optional.of(objectMapper.readValue(contents, ticketClass));
                    } catch (IOException e) {
                        LOG.error("Failed to parse TVM agent response: " + contents.replaceAll("\n", ""), e);
                        throw new UncheckedIOException(e);
                    }
                }
            }
        } catch (IOException e) {
            LOG.error("Failed to validate TVM ticket", e);
            throw new UncheckedIOException(e);
        }
    }
}
