package ru.yandex.qe.dispenser.ws;

import java.util.Objects;
import java.util.stream.Collectors;

import javax.ws.rs.Consumes;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.postgresql.util.PSQLException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.CannotAcquireLockException;
import org.springframework.jdbc.UncategorizedSQLException;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestBody;

import ru.yandex.qe.dispenser.domain.dao.entity.EntityDao;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaDao;
import ru.yandex.qe.dispenser.domain.distributed.Identifier;
import ru.yandex.qe.dispenser.domain.support.EntityOperation;
import ru.yandex.qe.dispenser.domain.support.OperationResult;
import ru.yandex.qe.dispenser.domain.support.QuotaChangeResponse;
import ru.yandex.qe.dispenser.domain.support.QuotaDiff;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.common.domain.exceptions.TooManyRequestsException;
import ru.yandex.qe.dispenser.ws.reqbody.QuotaChangeBody;

@Path("/v1")
@Service("batch-quota")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@Api(tags = SwaggerTags.DISPENSER_API, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class BatchQuotaServiceImpl implements BatchQuotaService, Predictable<QuotaChangeResponse, QuotaChangeBody> {

    private static final Logger LOG = LoggerFactory.getLogger(BatchQuotaServiceImpl.class);

    @Autowired
    private Identifier myIdentifier;

    @Autowired
    private QuotaDao quotaDao;

    @Autowired
    private EntityDao entityDao;

    @POST
    @Idempotent
    @Path("/{" + Access.SERVICE_KEY + "}/{" + Access.ENTITY_SPEC_KEY + "}/change-quotas")
    @Access(serviceTrustee = true)
    @Consumes(MediaType.APPLICATION_JSON)
    public QuotaChangeResponse balancedChangeQuotas(@RequestBody @NotNull final QuotaChangeBody body) {
        return changeQuotas(body);
    }

    @Override
    @POST
    @Idempotent
    @Path("/change-quotas")
    @Access(serviceTrustee = true)
    @Consumes(MediaType.APPLICATION_JSON)
    public QuotaChangeResponse changeQuotas(@RequestBody @NotNull final QuotaChangeBody body) {

        if (body.getActions(EntityOperation.class)
                .stream()
                .anyMatch(op -> !op.getEntity().getSpec().getTag().equals(myIdentifier.tag()))) {
            LOG.error("tag missmatch. expected {}", myIdentifier.tag());
            throw new IllegalStateException("tag missmatch! ask dispenser-dev@");
        }
        // TODO: merging & optimization

        //todo can get deadlock while updating entities and row resources in same request
        try {
            quotaDao.finalizeAndChangeAll(body.getActions(QuotaDiff.class));

            entityDao.doChanges(body.getActions(EntityOperation.class));
        } catch (CannotAcquireLockException e) {
            throw new TooManyRequestsException(e.getMessage(), e);
        } catch (UncategorizedSQLException e) {
            if ((e.getCause() instanceof PSQLException) &&
                    Objects.equals(((PSQLException) e.getCause()).getSQLState(), "55P03")) {
                throw new TooManyRequestsException(e.getMessage(), e);
            }
            throw e;
        }

        return new QuotaChangeResponse(body.getOperations().stream()
                .map(o -> OperationResult.success(o.getId()))
                .collect(Collectors.toList())
        );
    }

    @Override
    public QuotaChangeResponse predict(@NotNull final QuotaChangeBody body) {
        return new QuotaChangeResponse(body.getOperations().stream()
                .map(o -> OperationResult.success(o.getId()))
                .collect(Collectors.toList())
        );
    }
}
