package ru.yandex.qe.dispenser.ws;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.PATCH;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import com.google.common.collect.Sets;
import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestBody;
import ru.yandex.qe.dispenser.api.v1.DiCampaign;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.CampaignResource;
import ru.yandex.qe.dispenser.domain.CampaignUpdate;
import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.domain.dao.bot.service.reserve.BotServiceReserveDao;
import ru.yandex.qe.dispenser.domain.dao.bot.settings.BotCampaignGroupDao;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignResourceDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.bot.BigOrderManager;
import ru.yandex.qe.dispenser.ws.reqbody.CampaignBodyCreate;
import ru.yandex.qe.dispenser.ws.reqbody.CampaignBodyUpdate;

@ParametersAreNonnullByDefault
@Controller
@Path("/v1/campaigns")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@Service("campaign")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class CampaignService extends ServiceBase {
    public static final String CAMPAIGN_ID = "campaign_id";

    @Autowired
    private CampaignDao campaignDao;

    @Autowired
    private QuotaChangeRequestDao quotaChangeRequestDao;

    @Autowired
    private BotServiceReserveDao botServiceReserveDao;

    @Autowired
    private BigOrderManager bigOrderManager;

    @Autowired
    private CampaignResourceDao campaignResourceDao;

    @Autowired
    private BotCampaignGroupDao botCampaignGroupDao;

    @GET
    @Access
    @NotNull
    public DiListResponse<DiCampaign> getAll(@NotNull @QueryParam("status") final Set<Campaign.Status> statuses) {
        // TODO Add paging.
        return new DiListResponse<>(campaignDao.getAllSorted(statuses).stream()
                .map(Campaign::toView)
                .collect(Collectors.toList()));
    }

    @GET
    @Path("/{" + CAMPAIGN_ID + "}")
    @Access
    @NotNull
    public DiCampaign getById(@NotNull @PathParam(CAMPAIGN_ID) final Long id) {
        return campaignDao.read(id).toView();
    }

    @POST
    @Access(dispenserAdmin = true)
    @NotNull
    public DiCampaign create(@RequestBody @NotNull final CampaignBodyCreate body) {
        final Map<Long, BigOrder> validatedBigOrders = BigOrderUtils.validateBigOrders(bigOrderManager, body.getBigOrders());
        final Campaign campaign = body.toEntity(validatedBigOrders);
        return campaignDao.create(campaign).toView();
    }

    @PATCH
    @Path("/{" + CAMPAIGN_ID + "}")
    @Access(dispenserAdmin = true)
    @NotNull
    public DiCampaign update(@PathParam(CAMPAIGN_ID) @NotNull final Long id,
                             @RequestBody @NotNull final CampaignBodyUpdate update) {
        final Map<Long, BigOrder> validatedBigOrders = BigOrderUtils.validateBigOrders(bigOrderManager, update.getBigOrders());
        final CampaignUpdate entityUpdate = update.toEntityUpdate(validatedBigOrders);
        final Campaign campaign = campaignDao.readForUpdate(id);
        validateResourceRequests(id, entityUpdate);
        validateCampaignResourcesSettings(campaign, entityUpdate);
        final boolean hasGroup = botCampaignGroupDao.hasGroupForCampaign(campaign.getId());
        if (!hasGroup && update.getStatus() == Campaign.Status.ACTIVE) {
            throw new IllegalArgumentException("Cannot set ACTIVE status for campaign without campaign group");
        }
        campaignDao.partialUpdate(campaign, entityUpdate);
        return campaignDao.read(id).toView();
    }

    @DELETE
    @Path("/{" + CAMPAIGN_ID + "}")
    @Access(dispenserAdmin = true)
    @NotNull
    public DiCampaign delete(@PathParam(CAMPAIGN_ID) @NotNull final Long id) {
        final Campaign campaignToDelete = campaignDao.readForUpdate(id);
        if (quotaChangeRequestDao.hasRequestsInCampaign(id)) {
            throw new IllegalArgumentException("Campaign removal is forbidden when campaign already has resource requests");
        }
        if (botServiceReserveDao.hasReservesInCampaign(id)) {
            throw new IllegalArgumentException("Campaign removal is forbidden when campaign already has reserves");
        }
        if (botCampaignGroupDao.hasGroupForCampaign(id)) {
            throw new IllegalArgumentException("Campaign removal is forbidden when campaign already is in campaign group");
        }
        campaignDao.delete(campaignToDelete);
        return campaignToDelete.toView();
    }

    private void validateResourceRequests(@NotNull final Long id, @NotNull final CampaignUpdate entityUpdate) {
        if (entityUpdate.getBigOrders() != null) {
            final Set<Long> bigOrderIds = entityUpdate.getBigOrders().stream().map(Campaign.BigOrder::getBigOrderId).collect(Collectors.toSet());
            if (quotaChangeRequestDao.hasRequestsInCampaignForOrdersOtherThan(id, bigOrderIds)) {
                throw new IllegalArgumentException("Big order unbinding is forbidden when campaign already has resource requests for them");
            }
            if (botServiceReserveDao.hasReservesInCampaignForOrdersOtherThan(id, bigOrderIds)) {
                throw new IllegalArgumentException("Big order unbinding is forbidden when campaign already has reserves for them");
            }
            botCampaignGroupDao.getByCampaign(id).ifPresent(g -> {
                final Set<Long> groupOrders = CollectionUtils.ids(g.getBigOrders());
                final Set<Long> outOfGroupOrders = Sets.difference(bigOrderIds, groupOrders);
                if (!outOfGroupOrders.isEmpty()) {
                    throw new IllegalArgumentException(String.format("Orders %s not present in campaign group (%s) orders %s", outOfGroupOrders,
                            g.getKey(), groupOrders));
                }
            });
        }
    }

    private void validateCampaignResourcesSettings(@NotNull final Campaign campaign,
                                                   @NotNull final CampaignUpdate entityUpdate) {
        if (entityUpdate.getBigOrders() == null) {
            return;
        }
        final Set<Long> updatedBigOrderIds = entityUpdate.getBigOrders().stream()
                .map(Campaign.BigOrder::getBigOrderId).collect(Collectors.toSet());
        final Set<Long> removedCampaignBigOrderIds = campaign.getBigOrders().stream()
                .filter(o -> !updatedBigOrderIds.contains(o.getBigOrderId())).map(LongIndexBase::getId)
                .collect(Collectors.toSet());
        final List<CampaignResource> campaignResources = campaignResourceDao.getByCampaignId(campaign.getId());
        final boolean requiresRemovedBigOrders = campaignResources.stream().anyMatch(cr -> {
            return cr.getSettings().getBigOrders().stream().anyMatch(removedCampaignBigOrderIds::contains)
                    || cr.getSettings().getSegmentsBigOrders().stream()
                    .anyMatch(sbo -> sbo.getBigOrders().stream().anyMatch(removedCampaignBigOrderIds::contains));
        });
        if (requiresRemovedBigOrders) {
            throw new IllegalArgumentException("Some of the removed big orders are used in campaign resource settings");
        }
    }
}
