package ru.yandex.qe.dispenser.ws;


import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Inject;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;

import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.v1.DiCheck;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.hierarchy.Role;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;

import static ru.yandex.qe.dispenser.ws.ProjectService.PROJECT_KEY;

@Path("/v1/projects/{" + PROJECT_KEY + "}/checks")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("checks")
@ParametersAreNonnullByDefault
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class CheckService extends ServiceBase {
    private static final String CHECK_KEY = "check_key";
    private final JugglerChecks jugglerChecks;

    @Inject
    public CheckService(final JugglerChecks jugglerChecks) {
        this.jugglerChecks = jugglerChecks;
    }

    @GET
    @NotNull
    public DiListResponse<DiCheck> getAllChecks(@PathParam(PROJECT_KEY) final Project project) {

        final List<DiCheck> checks = jugglerChecks.getAll(project);

        return new DiListResponse<>(checks);
    }

    @GET
    @Path("{" + CHECK_KEY + "}")
    @NotNull
    public DiCheck getCheck(@PathParam(PROJECT_KEY) final Project project,
                            @PathParam(CHECK_KEY) final String checkKey) {
        return jugglerChecks.get(project, checkKey);
    }

    @POST
    @NotNull
    @Access(role = Role.RESPONSIBLE)
    public DiCheck createCheck(@PathParam(PROJECT_KEY) final Project project,
                               final DiCheck.Body checkBody) {
        return jugglerChecks.create(project, checkBody);
    }

    @PUT
    @Path("{" + CHECK_KEY + "}")
    @NotNull
    @Access(role = Role.RESPONSIBLE)
    public DiCheck updateCheck(@PathParam(PROJECT_KEY) final Project project,
                               @PathParam(CHECK_KEY) final String checkKey,
                               final DiCheck.Body checkBody) {
        return jugglerChecks.update(project, checkKey, checkBody);
    }

    @DELETE
    @Path("{" + CHECK_KEY + "}")
    @Access(role = Role.RESPONSIBLE)
    public DiCheck removeCheck(@PathParam(PROJECT_KEY) final Project project,
                               @PathParam(CHECK_KEY) final String key) {
        final DiCheck diCheck = jugglerChecks.get(project, key);
        jugglerChecks.remove(project, key);
        return diCheck;
    }
}
