package ru.yandex.qe.dispenser.ws;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.ws.rs.Consumes;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import com.google.common.collect.Table;
import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;

import ru.yandex.qe.dispenser.api.v1.request.DiEntityOwnership;
import ru.yandex.qe.dispenser.api.v1.response.DiEntityOwnershipResponse;
import ru.yandex.qe.dispenser.domain.Entity;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.dao.entity.EntityDao;
import ru.yandex.qe.dispenser.domain.dao.entity.InMemoryOnlyEntityDao;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaUtils;
import ru.yandex.qe.dispenser.domain.distributed.Identifier;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.support.EntityOwnership;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.param.EntityFilterParam;
import ru.yandex.qe.dispenser.ws.reqbody.EntityOwnershipBody;

@Path("/v1/entity-ownerships")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("entity-ownership")
@Api(tags = SwaggerTags.DISPENSER_API, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public final class EntityOwnershipService extends ServiceBase {
    @Autowired
    private EntityDao entityDao;

    @Autowired(required = false)
    private InMemoryOnlyEntityDao inmemoryEntityDao;

    /**
     * @deprecated DISPENSER-634
     */
    @Deprecated
    @GET
    @NotNull
    public DiEntityOwnershipResponse filter(@QueryParam("project") @NotNull final List<Project> projects,
                                            @QueryParam("entity") @NotNull final List<EntityFilterParam> entityParams,
                                            @QueryParam("leaf") @DefaultValue("false") final boolean leaf) {
        if (entityParams.isEmpty()) {
            throw new UnsupportedOperationException("Please, specify at least one entity!");
        }

        final List<Entity> presentEntities = entityParams.stream()
                .map(EntityFilterParam::get)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());

        return new DiEntityOwnershipResponse(getEntityOwnerships(projects, presentEntities, leaf));
    }

    @POST
    @NotNull
    @Consumes(MediaType.APPLICATION_JSON)
    public DiEntityOwnershipResponse filter(@RequestBody @NotNull final EntityOwnershipBody entityOwnershipBody) {
        return new DiEntityOwnershipResponse(getEntityOwnerships(entityOwnershipBody.getProjects(), entityOwnershipBody.getEntities(),
                entityOwnershipBody.isInLeafs()));
    }

    private List<DiEntityOwnership> getEntityOwnerships(@NotNull final Collection<Project> projects,
                                                        @NotNull final Collection<Entity> entities,
                                                        final boolean inLeafs) {
        final Table<Entity, Project, Integer> allUsages = entityDao.getUsages(entities);

        final List<EntityOwnership> ownerships = new ArrayList<>();
        allUsages.rowMap().forEach((e, usages) -> usages.forEach((p, u) -> ownerships.add(new EntityOwnership(e, p, u))));

        final Table<Entity, Project, EntityOwnership> allOwnerships = QuotaUtils.ownershipTable(ownerships);

        Set<Project> filterProjects = new HashSet<>(projects);
        if (projects.isEmpty()) {
            filterProjects = Hierarchy.get().getProjectReader().getAllReal();
        }
        return filterProjects.stream()
                .filter(p -> !inLeafs || p.isRealLeaf())
                .map(allOwnerships::column)
                .map(Map::values)
                .flatMap(Collection::stream)
                .map(EntityOwnership::toView)
                .collect(Collectors.toList());
    }

    @POST
    @Path("/byhost")
    public DiEntityOwnershipResponse getEntityOwnerShipByPath(@RequestBody @NotNull final Map<String, Long> timestamps, @QueryParam("limit") @DefaultValue("1000") final int limit) {
        Objects.requireNonNull(inmemoryEntityDao, "inmemory mode is required!");

        final Map<Identifier, Long> hostTs = new HashMap<>();
        timestamps.forEach((id, ts) -> {
            hostTs.put(Identifier.fromString(id), ts);
        });

        return new DiEntityOwnershipResponse(
                inmemoryEntityDao.getAllUsagesFrom(hostTs)
                        .limit(limit)
                        .map(EntityOwnership::toPersonalView)
                        .collect(Collectors.toList()));
    }
}
