package ru.yandex.qe.dispenser.ws

import io.github.resilience4j.core.IntervalFunction
import io.github.resilience4j.retry.Retry
import io.github.resilience4j.retry.RetryConfig
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.stereotype.Component
import ru.yandex.qe.dispenser.domain.jns.JnsApi
import ru.yandex.qe.dispenser.domain.jns.JnsMessage
import ru.yandex.qe.dispenser.domain.jns.JnsRequest
import ru.yandex.qe.dispenser.domain.jns.JnsResult
import java.net.ConnectException
import java.net.SocketTimeoutException
import java.util.*
import java.util.concurrent.Callable
import javax.ws.rs.ServerErrorException
import javax.ws.rs.WebApplicationException

@Component
class JnsApiHelper(
    private val jnsApi: JnsApi,
    @Qualifier("jns-tvm")
    private val tvmDestination: TvmDestination
) {

    private val retry: Retry = Retry.of("jnsHttpRetry", RetryConfig.custom<Any>()
        .maxAttempts(3)
        .intervalFunction(IntervalFunction.ofExponentialRandomBackoff(500, 1.5, 0.5))
        .retryOnException { it is SocketTimeoutException || it is ConnectException || it is ServerErrorException }
        .build())

    fun send(message: JnsMessage): JnsResult {
        try {
            val requestId = UUID.randomUUID().toString()
            return retry.executeSupplier {
                tvmDestination.runAuthorized( Callable {
                    val jnsRequest = JnsRequest(
                        project = message.project,
                        template = message.template,
                        channel = message.channel,
                        targetProject = message.targetProject,
                        requestId = requestId,
                        params = message.parameters
                    )
                    jnsApi.send(jnsRequest)
                    return@Callable JnsResult.success()
                })
            }
        } catch (e: WebApplicationException) {
            return JnsResult.error(e.response.status, e.response.readEntity(String::class.java))
        } catch (e: Exception) {
            return JnsResult.failure(e)
        }
    }
}
