package ru.yandex.qe.dispenser.ws;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Inject;

import com.google.common.collect.BiMap;
import com.google.common.collect.EnumHashBiMap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import ru.yandex.qe.bus.Robot;
import ru.yandex.qe.dispenser.api.v1.DiCheck;
import ru.yandex.qe.dispenser.api.v1.DiCheckNotificationType;
import ru.yandex.qe.dispenser.api.v1.DiCheckType;
import ru.yandex.qe.dispenser.client.v1.impl.DispenserConfig;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.juggler.ActiveCheckKwargs;
import ru.yandex.qe.dispenser.domain.juggler.ActiveCheckType;
import ru.yandex.qe.dispenser.domain.juggler.Check;
import ru.yandex.qe.dispenser.domain.juggler.CheckData;
import ru.yandex.qe.dispenser.domain.juggler.JugglerApi;
import ru.yandex.qe.dispenser.domain.juggler.JugglerNotification;
import ru.yandex.qe.dispenser.domain.juggler.notifications.OnChangeKwargs;
import ru.yandex.qe.dispenser.domain.juggler.notifications.StartrekKwargs;
import ru.yandex.qe.dispenser.domain.juggler.notifications.Template;
import ru.yandex.qe.dispenser.domain.juggler.notifications.TemplateKwargs;
import ru.yandex.qe.dispenser.ws.param.QuotaCheckParam;

@Component
@ParametersAreNonnullByDefault
public final class JugglerChecks {

    private final static EnumMap<DiCheckNotificationType, Template> NOTIFICATION_TYPE_TEMPLATE_MAP = new EnumMap<>(DiCheckNotificationType.class);
    private static final List<Object> NOTIFY_STATUSES = Arrays.asList("CRIT", "WARN", "OK");

    static {
        NOTIFICATION_TYPE_TEMPLATE_MAP.put(DiCheckNotificationType.MAIL, Template.ON_STATUS_CHANGE);
        NOTIFICATION_TYPE_TEMPLATE_MAP.put(DiCheckNotificationType.TELEGRAM, Template.ON_STATUS_CHANGE);
        NOTIFICATION_TYPE_TEMPLATE_MAP.put(DiCheckNotificationType.TRACKER, Template.STARTREK);
    }

    private final static BiMap<DiCheckNotificationType, String> NOTIFICATION_TYPE_METHOD_NAME_MAP = EnumHashBiMap.create(DiCheckNotificationType.class);

    static {
        NOTIFICATION_TYPE_METHOD_NAME_MAP.put(DiCheckNotificationType.MAIL, "email");
        NOTIFICATION_TYPE_METHOD_NAME_MAP.put(DiCheckNotificationType.TELEGRAM, "telegram");
    }

    @NotNull
    private final Robot robot;

    @NotNull
    private final String host;

    @NotNull
    private final String pathPrefix;

    @NotNull
    private final JugglerApi jugglerApi;

    @Inject
    public JugglerChecks(final JugglerApi jugglerApi,
                         @Qualifier("juggler-robot") final Robot robot,
                         @Value("${qe.app.environment}") final DispenserConfig.Environment environment,
                         @Value("${dispenser.cluster.prefix}") final String clusterPrefix) {
        this.jugglerApi = jugglerApi;
        this.robot = robot;
        this.host = environment.getDispenserHost();
        this.pathPrefix = clusterPrefix;
    }

    @NotNull
    private static List<String> getMethods(final Collection<DiCheckNotificationType> types) {
        return types.stream()
                .filter(NOTIFICATION_TYPE_METHOD_NAME_MAP::containsKey)
                .map(NOTIFICATION_TYPE_METHOD_NAME_MAP::get)
                .collect(Collectors.toList());
    }

    @NotNull
    private Check createCheck(final Project project, final DiCheck.Body diCheck) {
        return createCheck(project, generateCheckKey(project, diCheck), diCheck);
    }

    @NotNull
    private static String generateCheckKey(final Project project, final DiCheck.Body diCheck) {
        return UUID.randomUUID().toString();
    }

    @NotNull
    private static String getServiceKey(final Project project, final String checkKey) {
        return project.getPublicKey() + "_" + checkKey;
    }

    @NotNull
    private Check createCheck(final Project project, final String checkKey, final DiCheck.Body diCheck) {
        return createCheck(host, project, checkKey, diCheck);
    }

    @NotNull
    public Check createCheck(final String host, final Project project, final String checkKey, final DiCheck.Body diCheck) {

        final EnumSet<DiCheckNotificationType> notificationTypes = diCheck.getNotificationTypes();


        final Map<Template, List<DiCheckNotificationType>> notificationByTemplate = notificationTypes.stream()
                .filter(NOTIFICATION_TYPE_TEMPLATE_MAP::containsKey)
                .collect(Collectors.groupingBy(NOTIFICATION_TYPE_TEMPLATE_MAP::get));

        final Collection<String> persons = diCheck.getPersons();

        final List<JugglerNotification> notifications = notificationByTemplate.keySet().stream()
                .map(template -> createNotification(template, persons, notificationByTemplate.get(template)))
                .collect(Collectors.toList());


        final QuotaCheckParam.Value quotaCheckParam = QuotaCheckParam.fromCheck(project, diCheck);


        final String projectKey = project.getKey().getPublicKey();

        return new Check(host,
                getServiceKey(project, checkKey),
                "dispenser_quotas",
                Collections.singleton(getProjectTag(project)),
                ActiveCheckType.HTTPS,
                getCheckKwargs(quotaCheckParam, diCheck.getType()),
                notifications,
                String.format("Quota state for project='%s' resource='%s'",
                        projectKey,
                        quotaCheckParam.getQuotaSpecKey().get().getResource().getKey().getPublicKey()
                ),
                Collections.singletonMap("urls", Collections.singleton(
                        new ImmutableMap.Builder<String, String>()
                                .put("title", "Link to dispenser project")
                                .put("url", "https://" + host + pathPrefix + "/projects/" + projectKey)
                                .put("type", "dispenser")
                                .build()
                ))
        );
    }

    @NotNull
    private ActiveCheckKwargs getCheckKwargs(final QuotaCheckParam.Value quotaCheckParam, final DiCheckType type) {
        return new ActiveCheckKwargs(
                pathPrefix + "/api/v1/check-quotas?" + quotaCheckParam.toQueryParams(),
                Collections.singleton(QuotaCheckService.STATUS_OK),
                type == DiCheckType.WARN ? Collections.singleton(QuotaCheckService.STATUS_FAIL) : Collections.emptyList(),
                true
        );
    }

    @NotNull
    private static String getProjectTag(final Project project) {
        return "project_" + project.getPublicKey();
    }


    private static JugglerNotification createNotification(final Template template, final Collection<String> login,
                                                          final List<DiCheckNotificationType> notificationTypes) {
        final TemplateKwargs template_kwargs;
        switch (template) {
            case ON_STATUS_CHANGE:
            case ON_DESC_CHANGE:
                template_kwargs = new OnChangeKwargs(login, getMethods(notificationTypes), NOTIFY_STATUSES);
                break;
            case STARTREK:
                template_kwargs = new StartrekKwargs("DISPENSERALERT", Collections.emptySet(), login, NOTIFY_STATUSES);
                break;
            case GOLEM:
            default:
                template_kwargs = null;
        }
        return new JugglerNotification(template, template_kwargs, null);
    }

    @NotNull
    private DiCheck convertToJson(final Check check) {
        final ActiveCheckKwargs activeKwargs = check.getActiveKwargs();
        final String path = activeKwargs.getPath();
        final QuotaCheckParam.Value quotaCheckParam = QuotaCheckParam.fromPath(path);

        final Set<String> persons = Sets.newHashSet();
        final EnumSet<DiCheckNotificationType> types = EnumSet.noneOf(DiCheckNotificationType.class);

        final BiMap<String, DiCheckNotificationType> inverse = NOTIFICATION_TYPE_METHOD_NAME_MAP.inverse();

        final Collection<JugglerNotification> notifications = check.getNotifications();
        for (final JugglerNotification notification : notifications) {
            final TemplateKwargs templateKwargs = notification.getTemplate_kwargs();

            if (templateKwargs == null) {
                continue;
            }

            switch (notification.getTemplate_name()) {
                case ON_STATUS_CHANGE:
                case ON_DESC_CHANGE:

                    final OnChangeKwargs kwargs = (OnChangeKwargs) templateKwargs;
                    final List<String> login = kwargs.getLogin();

                    if (login != null) {
                        persons.addAll(login);
                    }

                    if (kwargs.getMethod() != null) {
                        for (final String method : kwargs.getMethod()) {
                            final DiCheckNotificationType type = inverse.get(method);
                            if (type != null) {
                                types.add(type);
                            }
                        }
                    }

                    break;
                case STARTREK:
                    final StartrekKwargs startrekKwargs = (StartrekKwargs) templateKwargs;
                    final List<String> followers = startrekKwargs.getFollowers();
                    if (followers != null && !followers.isEmpty()) {
                        types.add(DiCheckNotificationType.TRACKER);
                        persons.addAll(followers);
                    }
                    break;
                case GOLEM:
                default:
            }
        }

        final String projectKey = quotaCheckParam.getProject().getPublicKey();
        String service = check.getService();
        if (service.startsWith(projectKey + "_")) {
            service = service.substring(projectKey.length() + 1);
        }

        return new DiCheck.Builder()
                .setKey(service)
                .setProjectKey(projectKey)
                .setQuotaSpecKey(quotaCheckParam.getQuotaSpecKey().toString())
                .setNotificationTypes(types)
                .setPersons(persons)
                .setType(DiCheckType.CRIT)
                .setAmount(quotaCheckParam.getCheckAmount())
                .setCompareType(quotaCheckParam.getCompareType())
                .setSegments(quotaCheckParam.getSegmentKeys())
                .setType(activeKwargs.getWarnCodes().isEmpty() ? DiCheckType.CRIT : DiCheckType.WARN)
                .build();
    }

    public void remove(final Project project, final String checkKey) {
        robot.runAuthorized(() -> jugglerApi.removeCheck(host, getServiceKey(project, checkKey), null, 1));
    }

    @NotNull
    public DiCheck update(final Project project, final String checkKey, final DiCheck.Body checkBody) {
        final Check check = createCheck(project, checkKey, checkBody);

        robot.runAuthorized(() -> jugglerApi.addOrUpdate(check, 1));

        return convertToJson(check);
    }

    @NotNull
    public DiCheck create(final Project project, final DiCheck.Body checkBody) {
        final Check check = createCheck(project, checkBody);

        robot.runAuthorized(() -> jugglerApi.addOrUpdate(check, 1));

        return convertToJson(check);
    }

    @NotNull
    public DiCheck get(final Project project, final String checkKey) {
        final Map<String, Map<String, CheckData>> serviceChecks = jugglerApi.getServiceChecks(host, getServiceKey(project, checkKey), 1, "", 1);

        final List<DiCheck> checks = getChecks(serviceChecks);
        return checks.get(0);
    }

    @NotNull
    public List<DiCheck> getAll(final Project project) {
        final Map<String, Map<String, CheckData>> serviceChecks = jugglerApi.getServiceChecks(host, null, 1, getProjectTag(project), 1);
        return getChecks(serviceChecks);
    }

    @NotNull
    private List<DiCheck> getChecks(final Map<String, Map<String, CheckData>> serviceChecks) {
        final List<DiCheck> checks = new ArrayList<>();
        for (final String checkHost : serviceChecks.keySet()) {
            final Map<String, CheckData> hostChecks = serviceChecks.get(checkHost);
            for (final String service : hostChecks.keySet()) {
                final DiCheck check = convertToJson(new Check(checkHost, service, hostChecks.get(service)));
                checks.add(check);
            }
        }
        return checks;
    }
}
