package ru.yandex.qe.dispenser.ws;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Set;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import ru.yandex.qe.dispenser.api.v1.DiPersonGroup;
import ru.yandex.qe.dispenser.api.v1.field.DiField;
import ru.yandex.qe.dispenser.api.v1.field.DiProjectFields;
import ru.yandex.qe.dispenser.api.v1.project.DiExtendedProject;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.ProjectFieldsContext;
import ru.yandex.qe.dispenser.domain.YaGroup;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.domain.util.ApplicationContextProvider;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.QuotaRequestWorkflowManager;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.ResourceWorkflow;

public enum ProjectField {
    KEY(DiField.KEY, Project::getPublicKey),
    NAME(DiField.NAME, Project::getName),
    DESCRIPTION(DiField.DESCRIPTION, Project::getDescription),
    ABC_SERVICE_ID(DiProjectFields.ABC_SERVICE, Project::getAbcServiceId),
    PARENT_PROJECT(DiProjectFields.PARENT_PROJECT, (p, ctx) -> p.isRoot() ? null :
            toView(p.getParent(), ProjectService.REFERENCE_PROJECT_FIELDS, ctx.forkSingle())),

    ANCESTORS(DiProjectFields.ANCESTORS, (project, ctx) -> {
        return project.getPathFromRoot().stream().map(p -> {
            return toView(p, ProjectService.REFERENCE_PROJECT_FIELDS, ctx.forkSingle());
        }).collect(Collectors.toList());
    }),
    MEMBERS(DiProjectFields.MEMBERS, project -> {
        final Set<Person> linkedMemberPersons = Hierarchy.get().getProjectReader().getLinkedMembers(project);
        final DiPersonGroup.Builder membersBuilder = DiPersonGroup.builder()
                .addPersons(CollectionUtils.map(linkedMemberPersons, Person::getLogin));

        final Set<YaGroup> linkedMemberGroups = Hierarchy.get().getProjectReader().getLinkedMemberGroups(project);
        for (final YaGroup group : linkedMemberGroups) {
            membersBuilder.addYaGroups(group.getType(), group.getUrl());
        }
        return membersBuilder.build();
    }),
    RESPONSIBLES(DiProjectFields.RESPONSIBLES, project -> {
        final Set<Person> linkedResponsibles = Hierarchy.get().getProjectReader().getLinkedResponsibles(project);
        return DiPersonGroup.builder()
                .addPersons(CollectionUtils.map(linkedResponsibles, Person::getLogin))
                .build();
    }),

    ALL_MEMBERS(DiProjectFields.ALL_MEMBERS, (project, ctx) -> {
        return ctx.getAllMembersOfProject(project);
    }),

    PERMISSIONS(DiProjectFields.PERMISSIONS, (project, ctx) -> {
        final Person performer = Session.WHOAMI.get();
        final QuotaRequestWorkflowManager quotaRequestWorkflowManager = ApplicationContextProvider.getContext().getBean(QuotaRequestWorkflowManager.class);
        final ResourceWorkflow workflow = quotaRequestWorkflowManager.getResourceWorkflow();
        final Set<DiExtendedProject.Permission> permissions = new HashSet<>();
        if (workflow.canUserCreateRequestByProject(performer, project, ctx)) {
            permissions.add(DiExtendedProject.Permission.CAN_CREATE_QUOTA_REQUEST);
        }
        if (ResourceWorkflow.canUserViewBotMoney(performer)) {
            permissions.add(DiExtendedProject.Permission.CAN_VIEW_BOT_PREORDER_COSTS);
        }
        return permissions;
    }),
    SUB_PROJECT_KEYS(DiProjectFields.SUB_PROJECT_KEYS, project -> {
        return project.getRealSubprojects().stream()
                .map(Project::getPublicKey)
                .collect(Collectors.toList());
    }),
    PARENT_PROJECT_KEY(DiProjectFields.PARENT_PROJECT_KEY, project -> project.isRoot() ? null :
            project.getParent().getPublicKey()),

    MAIL_LIST(DiProjectFields.MAIL_LIST, Project::getMailList),
    ;


    @NotNull
    private final DiField<?> field;
    @NotNull
    private final BiFunction<Project, ProjectFieldsContext, ?> provider;

    <T> ProjectField(@NotNull final DiField<T> field, final @NotNull BiFunction<Project, ProjectFieldsContext, T> provider) {
        this.field = field;
        this.provider = provider;
    }

    <T> ProjectField(@NotNull final DiField<T> field, final @NotNull Function<Project, ?> provider) {
        this.field = field;
        this.provider = (p, ctx) -> provider.apply(p);
    }

    @NotNull
    public static DiExtendedProject toView(final Project project, @NotNull final Set<DiField<?>> fields, @NotNull final ProjectFieldsContext projectFieldsContext) {
        return new DiExtendedProject(
                fields.stream()
                        .map(ProjectField::byField)
                        .collect(HashMap::new, (m, f) -> m.put(f.getField(), f.getValue(project, projectFieldsContext)), HashMap::putAll) // can't use Collectors.toMap due to null-values
        );
    }

    @Nullable
    public Object getValue(@NotNull final Project project, @NotNull final ProjectFieldsContext context) {
        return provider.apply(project, context);
    }

    @NotNull
    public DiField<?> getField() {
        return field;
    }

    @Nonnull
    public static ProjectField byField(@NotNull final DiField<?> field) {
        for (final ProjectField value : values()) {
            if (value.field == field) {
                return value;
            }
        }
        throw new IllegalArgumentException("No project field binding for " + field);
    }
}
