package ru.yandex.qe.dispenser.ws;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.ws.rs.DELETE;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.InternalServerErrorException;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Response;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DataIntegrityViolationException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.RequestBody;

import ru.yandex.qe.dispenser.api.v1.DiCampaign;
import ru.yandex.qe.dispenser.api.v1.DiPersonGroup;
import ru.yandex.qe.dispenser.api.v1.DiProject;
import ru.yandex.qe.dispenser.api.v1.DiYandexGroupType;
import ru.yandex.qe.dispenser.api.v1.field.DiField;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.ProjectFieldsContext;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.YaGroup;
import ru.yandex.qe.dispenser.domain.abc.AbcPerson;
import ru.yandex.qe.dispenser.domain.dao.group.GroupDao;
import ru.yandex.qe.dispenser.domain.dao.person.PersonDao;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectDao;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectManager;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.Role;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.domain.util.ValidationUtils;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.abc.AbcApiHelper;
import ru.yandex.qe.dispenser.ws.aspect.AccessAspect;
import ru.yandex.qe.dispenser.ws.param.ProjectFieldsParam;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.QuotaRequestWorkflowManager;
import ru.yandex.qe.dispenser.ws.reqbody.ProjectBody;

import static ru.yandex.qe.dispenser.ws.param.FieldsParam.FIELDS_PARAM;
import static ru.yandex.qe.dispenser.ws.param.FieldsParam.FIELD_PARAM;

@Controller
@Path("/v1")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("project")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class ProjectService extends ServiceBase {
    public static final String PROJECT_KEY = "project_key";
    public static final String SHOW_PERSONS = "showPersons";
    public static final Set<DiField<?>> REFERENCE_PROJECT_FIELDS = ImmutableSet.of(DiField.KEY, DiField.NAME);

    @Autowired
    private AbcApiHelper abcApiHelper;
    @Autowired
    private ProjectManager projectManager;
    @Autowired
    private ProjectFieldsContextFactory projectFieldsContextFactory;

    @Autowired
    private ProjectDao projectDao;

    @Autowired
    private QuotaDao quotaDao;

    @Autowired
    private PersonDao personDao;

    @Autowired
    private GroupDao groupDao;

    @Autowired
    private QuotaRequestWorkflowManager quotaRequestWorkflowManager;

    /*
      curl "http://localhost:8082/api/v0/project/yandex"
      curl "https://dispenser-dev.yandex-team.ru/api/v0/project/yandex"
     */
    @GET
    @Access
    @NotNull
    @Path("/projects/{" + Access.PROJECT_KEY + "}")
    public Response getProject(@PathParam(Access.PROJECT_KEY) @NotNull final Project project,
                               @QueryParam(SHOW_PERSONS) @DefaultValue("true") final boolean showPersons,
                               @Nullable @QueryParam(FIELDS_PARAM) final String fieldsParam,
                               @Nullable @QueryParam(FIELD_PARAM) final Set<String> fieldParam) {
        final ProjectFieldsParam fields = new ProjectFieldsParam(fieldsParam, fieldParam);
        return Response
                .ok(!fields.isEmpty() ? ProjectField.toView(project, fields.get(), projectFieldsContextFactory.single(fields)) : project.toView(showPersons))
                .build();
    }

    @GET
    @Access
    @NotNull
    @Path("/projects/_filter-projects")
    public Response filterProjects(@QueryParam("leaf") @DefaultValue("false") final boolean leaf,
                                   @QueryParam("responsible") @NotNull final List<Person> responsibles,
                                   @QueryParam("member") @NotNull final List<Person> members,
                                   @QueryParam("project") @NotNull final List<Project> projects,
                                   @QueryParam(SHOW_PERSONS) @DefaultValue("false") final boolean showPersons,
                                   @QueryParam(NO_CACHE) @DefaultValue("false") final boolean cacheDisabled,
                                   @Nullable @QueryParam(FIELDS_PARAM) final String fieldsParam,
                                   @Nullable @QueryParam(FIELD_PARAM) final Set<String> fieldParam) {
        final ProjectFieldsParam fields = new ProjectFieldsParam(fieldsParam, fieldParam);
        if (projects.isEmpty()) {
            if (cacheDisabled) {
                projects.addAll(projectDao.getAllReal());
            } else {
                projects.addAll(Hierarchy.get().getProjectReader().getAllReal());
            }
        }
        if (leaf) {
            projects.removeIf(p -> !p.isRealLeaf());
            responsibles.forEach(responsible -> projects.retainAll(Hierarchy.get().getResponsilbeRealLeafProjects(responsible)));
            members.forEach(member -> projects.retainAll(Hierarchy.get().getMemberRealLeafProjects(member)));
        } else {
            responsibles.forEach(responsible -> projects.retainAll(Hierarchy.get().getResponsibleProjects(responsible)));
            members.forEach(member -> projects.retainAll(Hierarchy.get().getMemberProjects(member)));
        }

        final Function<Project, ?> projectToViewMapper;
        final ProjectFieldsContext projectFieldsContext = projectFieldsContextFactory.multi(fields);

        if (!fields.isEmpty()) {
            projectToViewMapper = p -> ProjectField.toView(p, fields.get(), projectFieldsContext);
        } else if (showPersons) {
            final Project.BulkProjectSerializationContext context = new Project.BulkProjectSerializationContext(projects);
            projectToViewMapper = p -> p.toView(true, context);
        } else {
            projectToViewMapper = p -> p.toView(false);
        }

        return Response.ok(new DiListResponse<>(projects
                .stream()
                .map(projectToViewMapper)
                .collect(Collectors.toList()))
        ).build();
    }

    /*
       curl -H "Authorization: whistler" -H "Content-Type: application/json" -X POST -d '{"key":"java_report","name":"Java Report","responsibles":["lyadzhin"]}' "http://localhost:8082/api/v0/project/yandex/create-subproject" -i | less
       curl -H "Authorization: whistler" -H "Content-Type: application/json" -X POST -d '{"key":"java_report","name":"Java Report","responsibles":["lyadzhin"]}' "https://dispenser-dev.yandex-team.ru/api/v0/project/yandex/create-subproject" -i | less
     */
    @POST
    @Mutator
    @NotNull
    @Path("/projects/{" + Access.PROJECT_KEY + "}/create-subproject")
    @Access(role = Role.RESPONSIBLE)
    public DiProject createSubproject(@PathParam(Access.PROJECT_KEY) @NotNull final Project parent,
                                      @RequestBody @NotNull final ProjectBody body,
                                      @QueryParam(SHOW_PERSONS) @DefaultValue("true") @NotNull final Boolean showPersons) {
        if (body.getSubprojectKeys().isEmpty()) {
            return createLeafSubproject(parent, body).toView(showPersons);
        }
        return insertIntermediateSubproject(parent, body).toView(showPersons);
    }

    @NotNull
    private Project createLeafSubproject(@NotNull final Project parent, @NotNull final ProjectBody body) {
        requireNoPersonalSubprojects(parent);

        abcApiHelper.validateAbcServiceHierarchy(parent, body.getAbcServiceId());

        return doCreateProject(body, parent);
    }

    @NotNull
    private Project insertIntermediateSubproject(@NotNull final Project parent, @NotNull final ProjectBody body) {
        final Collection<Project> readerProjects = Hierarchy.get()
                .getProjectReader()
                .readByPublicKeys(body.getSubprojectKeys())
                .values();

        abcApiHelper.validateAbcServiceHierarchy(parent, body.getAbcServiceId(), readerProjects);

        final Collection<Project> subprojects = projectDao.lockForUpdate(readerProjects)
                .stream()
                .filter(project -> !project.isRemoved())
                .collect(Collectors.toSet());

        subprojects.forEach(sp -> {
            if (!sp.getParent().equals(parent)) {
                throw new IllegalArgumentException(
                        "Project '" + sp.getPublicKey() + "' is not a direct child of '" + parent.getPublicKey() + "'");
            }
        });

        final Project newProject = doCreateProject(body, parent);
        subprojects.forEach(sp -> {
            doUpdateProject(Project.copyOf(sp).parent(newProject).build());
        });

        if (!subprojects.isEmpty()) {
            final Set<Quota> subProjectsQuota = quotaDao.getProjectsQuotasForUpdate(subprojects);

            final Map<Quota.Key, Long> newQuotaValues = quotaDao.getQuotas(newProject)
                    .stream()
                    .map(Quota::getKey)
                    .collect(Collectors.toMap(
                            Function.identity(),
                            key -> {
                                final Set<Quota.Key> subProjectQuotaKeys = subprojects.stream()
                                        .map(key::withProject)
                                        .collect(Collectors.toSet());

                                return subProjectsQuota.stream()
                                        .filter(q -> subProjectQuotaKeys.contains(q.getKey()))
                                        .mapToLong(Quota::getMax)
                                        .sum();
                            }
                    ));
            quotaDao.applyChanges(newQuotaValues, Collections.emptyMap(), Collections.emptyMap());
        }

        return projectDao.read(newProject.getKey());
    }

    @NotNull
    private Project doCreateProject(@NotNull final ProjectBody body, @NotNull final Project parent) {

        final Project lockedParent = lockForUpdateExistingProject(parent);

        ValidationUtils.requireNonNull(body.getAbcServiceId(), "ABC service ID is required");
        if (body.getMailList() != null) {
            ValidationUtils.validateYaTeamMail(body.getMailList());
        }

        final Project project;
        try {
            project = projectDao.create(Project.withKey(body.getKey())
                    .name(body.getName())
                    .description(body.getDescription())
                    .abcServiceId(body.getAbcServiceId())
                    .parent(lockedParent)
                    .mailList(body.getMailList())
                    .build());
        } catch (DataIntegrityViolationException e) {
            throw new IllegalArgumentException(String.format("Project with key = '%s' or name = '%s' already exists", body.getKey(), body.getName()));
        }

        updatePersons(project, body);

        return project;
    }

    /*
       curl -H "Authorization: whistler" -H "Content-Type: application/json" -X PUT -d '{"key":"key","name":"name_update","responsibles":["whistler"]}' "http://localhost:8082/api/v0/project/key" -i | less
     */
    @POST
    @Mutator
    @NotNull
    @Access(role = Role.RESPONSIBLE)
    @Path("projects/{" + Access.PROJECT_KEY + "}")
    public DiProject updateProject(@PathParam(Access.PROJECT_KEY) @NotNull final Project origin,
                                   @RequestBody @NotNull final ProjectBody body,
                                   @DefaultValue("true") @QueryParam(SHOW_PERSONS) @NotNull final Boolean showPersons) {
        checkAbcServiceIdIsNotRemoved(body, origin);

        final Project newParent = computeParent(origin, body.getParentProjectKey());

        if (body.getMailList() != null) {
            ValidationUtils.validateYaTeamMail(body.getMailList());
        }

        if (newParent != null && newParent != origin.getParent()) {

            final Project lca = origin.getLeastCommonAncestor(newParent);
            if (lca.equals(origin)) {
                throw new IllegalArgumentException("New parent can't be in origin subtree!");
            }

            final Person performer = Session.WHOAMI.get();

            if (!AccessAspect.isDispenserAdmin(performer)) {
                throw new IllegalArgumentException("Only dispenser admins can change project parents!");
            }

            projectManager.moveQuotas(origin, newParent);
        }

        final Project projectToUpdate = Project.copyOf(origin)
                .name(body.getName())
                .description(body.getDescription())
                .abcServiceId(body.getAbcServiceId())
                .parent(newParent)
                .mailList(body.getMailList())
                .build();

        if (!Objects.equals(origin.getAbcServiceId(), projectToUpdate.getAbcServiceId())) {
            abcApiHelper.validateAbcServiceHierarchy(projectToUpdate);
        }
        doUpdateProject(projectToUpdate);
        updatePersons(projectToUpdate, body);

        return projectDao.read(origin.getKey()).toView(showPersons);
    }

    private void doUpdateProject(@NotNull final Project projectToUpdate) {
        if (!projectDao.update(projectToUpdate)) {
            throw new InternalServerErrorException("Can't update project, internal error!");
        }
    }

    @DELETE
    @NotNull
    @Mutator
    @Access(role = Role.RESPONSIBLE)
    @Path("projects/{" + Access.PROJECT_KEY + "}")
    public DiProject removeProject(@PathParam(Access.PROJECT_KEY) @NotNull final Project project) {
        return removeProjectRecursive(project).toView();
    }

    private Project removeProjectRecursive(final Project originalProject) {

        final Project lockedProject = lockForUpdateExistingProject(originalProject);

        final Set<Quota> quotas = quotaDao.getQuotas(lockedProject);

        checkQuotaUsage(quotas);

        lockedProject.getExistingSubProjects().forEach(this::removeProjectRecursive);

        final Project project = Project.copyOf(lockedProject)
                .removed(true)
                .build();

        doUpdateProject(project);

        final Map<Quota.Key, Long> zeroQuotaValues = quotas.stream()
                .collect(Collectors.toMap(Quota::getKey, q -> 0L));
        quotaDao.applyChanges(zeroQuotaValues, Collections.emptyMap(), zeroQuotaValues);
        return lockedProject;
    }

    @Nullable
    private Project computeParent(@NotNull final Project origin, @Nullable final String parentProjectKey) {
        if (parentProjectKey == null) {
            return !origin.isRoot() ? origin.getParent() : null;
        }
        if (origin.getPublicKey().equals(parentProjectKey)) {
            throw new IllegalArgumentException("Project '" + origin.getPublicKey() + "' can't be self-parent!");
        }
        if (origin.isRoot()) {
            throw new IllegalArgumentException("Root '" + origin.getPublicKey() + "' can't change parent!");
        }
        if (origin.getParent().getPublicKey().equals(parentProjectKey)) {
            return origin.getParent();
        }
        final Project newParent = Hierarchy.get().getProjectReader().readExisting(parentProjectKey);

        final Set<Quota> newParentQuotas = quotaDao.getQuotas(newParent);

        checkQuotaUsage(newParentQuotas);
        requireNoPersonalSubprojects(newParent);

        final Person performer = ValidationUtils.requireNonNull(Session.WHOAMI.get(), "No personal authorizarion in request!");
        AccessAspect.checkRole(performer, newParent, Role.RESPONSIBLE);

        return newParent;
    }

    @NotNull
    private Project requireNoPersonalSubprojects(@NotNull final Project project) {
        if (project.hasPersonalSubprojects()) {
            throw new IllegalArgumentException(String.format("Project '%s' is leaf project which already has quota usages. Only leaf projects without quota usages or non-leaf projects can be used as parent.", project.getPublicKey()));
        }
        return project;
    }

    private void checkAbcServiceIdIsNotRemoved(@NotNull final ProjectBody requestBody, @NotNull final Project project) {
        if (requestBody.getAbcServiceId() == null && project.getAbcServiceId() != null) {
            throw new IllegalArgumentException("ABC service ID cannot be removed");
        }
    }

    private void checkQuotaUsage(@NotNull final Set<Quota> quotas) {
        for (final Quota quota : quotas) {
            if (quota.getResource().getService().getSettings().requireZeroQuotaUsageForProjectDeletion() && quota.getOwnActual() > 0) {
                throw new IllegalArgumentException("Project can't be deleted or can't become a parent due to non-zero quota usage: " + quota);
            }
        }
    }

    /**
     * Used by https://abc.yandex-team.ru/services/sbs3/
     * <p>
     * curl -H "Authorization: whistler" -H "Content-Type: application/json" -X POST -d '["amosov-f", "starligth"]' "http://localhost:8082/api/v1/projects/yandex/attach-members" -i | less
     */
    @POST
    @Mutator
    @NotNull
    @Path("projects/{" + Access.PROJECT_KEY + "}/attach-members")
    @Access(role = Role.RESPONSIBLE)
    public DiProject attachMembers(@PathParam(Access.PROJECT_KEY) @NotNull final Project project,
                                   @RequestBody @NotNull final List<Person> members,
                                   @DefaultValue("true") @QueryParam(SHOW_PERSONS) @NotNull final Boolean showPersons) {
        projectDao.attachAll(members, Collections.emptyList(), project, Role.MEMBER);
        return project.toView(showPersons);
    }

    /**
     * Used by https://abc.yandex-team.ru/services/sbs3/
     * <p>
     * curl -H "Authorization: whistler" -H "Content-Type: application/json" -X POST -d '["amosov-f", "starligth"]' "http://localhost:8082/api/v1/projects/yandex/detach-members" -i | less
     */
    @POST
    @Mutator
    @NotNull
    @Path("projects/{" + Access.PROJECT_KEY + "}/detach-members")
    @Access(role = Role.RESPONSIBLE)
    public DiProject detachMembers(@PathParam(Access.PROJECT_KEY) @NotNull final Project project,
                                   @RequestBody @NotNull final List<Person> members,
                                   @DefaultValue("true") @QueryParam(SHOW_PERSONS) @NotNull final Boolean showPersons) {
        projectDao.detachAll(members, Collections.emptyList(), project, Role.MEMBER);
        return project.toView(showPersons);
    }

    @POST
    @Mutator
    @NotNull
    @Path("projects/{" + Access.PROJECT_KEY + "}/attach-responsibles")
    @Access(role = Role.RESPONSIBLE)
    public DiProject attachResponsibles(@PathParam(Access.PROJECT_KEY) @NotNull final Project project,
                                        @RequestBody @NotNull final List<Person> members,
                                        @DefaultValue("true") @QueryParam(SHOW_PERSONS) @NotNull final Boolean showPersons) {
        projectDao.attachAll(members, Collections.emptyList(), project, Role.RESPONSIBLE);
        return project.toView(showPersons);
    }

    @POST
    @Mutator
    @NotNull
    @Path("projects/{" + Access.PROJECT_KEY + "}/detach-responsibles")
    @Access(role = Role.RESPONSIBLE)
    public DiProject detachResponsibles(@PathParam(Access.PROJECT_KEY) @NotNull final Project project,
                                        @RequestBody @NotNull final List<Person> members,
                                        @DefaultValue("true") @QueryParam(SHOW_PERSONS) @NotNull final Boolean showPersons) {
        projectDao.detachAll(members, Collections.emptyList(), project, Role.RESPONSIBLE);
        return project.toView(showPersons);
    }

    @GET
    @NotNull
    @Path("projects/{" + Access.PROJECT_KEY + "}/availableCampaigns")
    @Access
    public List<DiCampaign> getAvailableCampaigns(@PathParam(Access.PROJECT_KEY) @NotNull Project project) {
        Person performer = Session.WHOAMI.get();
        return quotaRequestWorkflowManager.getResourceWorkflow().getEligibleCampaigns(performer, project)
                .stream().map(Campaign::toView).toList();
    }

    @NotNull
    private Project lockForUpdateExistingProject(@NotNull final Project project) {
        final Project lockedProject = projectDao.lockForUpdate(project);
        if (lockedProject.isRemoved()) {
            throw new EmptyResultDataAccessException("No project with key = " + project.getPublicKey(), 1);
        }
        return project;
    }

    private void updatePersons(@NotNull final Project project, @NotNull final ProjectBody body) {
        final Set<Person> incomingResponsiblePersons = validatePersons(body.getResponsibles());
        final Set<YaGroup> responsibleGroups = validateGroups(body.getResponsibles());
        final Set<Person> incomingMemberPersons = validatePersons(body.getMembers());
        final Set<YaGroup> memberGroups = validateGroups(body.getMembers());
        validateResponsibles(project, incomingResponsiblePersons, responsibleGroups);
        validateMembers(project, incomingMemberPersons, memberGroups);
        final Collection<Person> actualResponsiblePersons = resolveActualPersonsForProject(project, incomingResponsiblePersons,
                abcApiHelper::getServiceResponsibles);
        final Collection<Person> actualMembersPersons = resolveActualPersonsForProject(project, incomingMemberPersons, abcApiHelper::getServiceMembers);
        projectDao.detachAll(project);
        projectDao.attachAll(actualResponsiblePersons, responsibleGroups, project, Role.RESPONSIBLE);
        projectDao.attachAll(actualMembersPersons, memberGroups, project, Role.MEMBER);
    }

    private void validateResponsibles(@NotNull final Project project, @NotNull final Set<Person> persons, @NotNull final Set<YaGroup> groups) {
        final Set<String> errors = new HashSet<>();
        if (!groups.isEmpty()) {
            errors.add("Group can't be responsible of a project.");
        }
        if (project.isSyncedWithAbc() && project.getAbcServiceId() != null) {
            if (!persons.isEmpty()) {
                errors.add("Direct person setting is not allowed for responsible.");
            }
        }
        if (!errors.isEmpty()) {
            throw new IllegalArgumentException(String.join(" ", errors));
        }
    }

    private void validateMembers(@NotNull final Project project, @NotNull final Set<Person> persons, @NotNull final Set<YaGroup> groups) {
        final Set<String> errors = new HashSet<>();
        if (project.isSyncedWithAbc() && project.getAbcServiceId() != null) {
            if (!persons.isEmpty()) {
                errors.add("Direct person setting is not allowed for members.");
            }
            if (!groups.isEmpty()) {
                errors.add("Group can't be responsible of a project with ABC synchronization.");
            }
        }
        if (!errors.isEmpty()) {
            throw new IllegalArgumentException(String.join(" ", errors));
        }
    }

    private Set<Person> validatePersons(@NotNull final DiPersonGroup personGroup) {
        final Set<Person> foundPersons = Hierarchy.get().getPersonReader().tryReadPersonsByLogins(personGroup.getPersons());
        final Set<String> foundLogins = foundPersons.stream().map(Person::getLogin).collect(Collectors.toSet());
        final Set<String> missingLogins = Sets.difference(personGroup.getPersons(), foundLogins);
        if (!missingLogins.isEmpty()) {
            throw new IllegalArgumentException("Users with logins [" + String.join(", ", missingLogins) + "] were not found.");
        }
        return foundPersons;
    }

    private Set<YaGroup> validateGroups(@NotNull final DiPersonGroup personGroup) {
        final Set<String> errors = new HashSet<>();
        final Set<String> groupUrls = new HashSet<>();
        personGroup.getYandexGroupsByType().forEach((type, urls) -> groupUrls.addAll(urls));
        final Set<YaGroup> foundGroups = groupDao.tryReadYaGroupsByUrls(groupUrls);
        final Set<String> foundUrls = foundGroups.stream().map(YaGroup::getUrl).collect(Collectors.toSet());
        final Set<String> missingUrls = Sets.difference(groupUrls, foundUrls);
        if (!missingUrls.isEmpty()) {
            errors.add("Groups with urls [" + String.join(", ", missingUrls) + "] were not found.");
        }
        final Set<String> deletedUrls = foundGroups.stream().filter(YaGroup::isDeleted).map(YaGroup::getUrl).collect(Collectors.toSet());
        if (!deletedUrls.isEmpty()) {
            errors.add("Groups with urls [" + String.join(", ", deletedUrls) + "] are deleted.");
        }
        final Map<DiYandexGroupType, Set<String>> foundUrlsByType = foundGroups.stream().collect(Collectors.groupingBy(YaGroup::getType,
                Collectors.mapping(YaGroup::getUrl, Collectors.toSet())));
        final Set<String> invalidTypeUrls = new HashSet<>();
        personGroup.getYandexGroupsByType().forEach((type, urls) -> {
            invalidTypeUrls.addAll(Sets.difference(urls, foundUrlsByType.getOrDefault(type, new HashSet<>())));
        });
        final Set<String> actuallyInvalidTypeUrls = Sets.difference(invalidTypeUrls, missingUrls);
        if (!actuallyInvalidTypeUrls.isEmpty()) {
            errors.add("Groups with urls [" + String.join(", ", actuallyInvalidTypeUrls) + "] are of invalid type.");
        }
        if (!errors.isEmpty()) {
            throw new IllegalArgumentException(String.join(" ", errors));
        }
        return foundGroups;
    }

    @NotNull
    private Set<Person> resolveActualPersonsForProject(@NotNull final Project project, @NotNull final Set<Person> persons,
                                                       @NotNull final Function<Integer, Stream<AbcPerson>> abcProvider) {
        final Integer abcServiceId = project.getAbcServiceId();
        if (project.isSyncedWithAbc() && abcServiceId != null) {
            return getPersonsFromAbc(abcServiceId, abcProvider);
        }
        return persons;
    }

    @NotNull
    private Set<Person> getPersonsFromAbc(@NotNull final Integer abcServiceId, @NotNull final Function<Integer, Stream<AbcPerson>> abcProvider) {
        final Stream<AbcPerson> persons = abcProvider.apply(abcServiceId);
        final Set<String> logins = persons.map(AbcPerson::getLogin).collect(Collectors.toSet());
        final Set<Person> foundPersons = Hierarchy.get().getPersonReader().tryReadPersonsByLogins(logins);
        final Set<String> foundLogins = foundPersons.stream().map(Person::getLogin).collect(Collectors.toSet());
        final Set<String> missingLogins = Sets.difference(logins, foundLogins);
        if (!missingLogins.isEmpty()) {
            throw new IllegalArgumentException("Logins [" + String.join(", ", missingLogins) + "] received " +
                    " from ABC were not synced with Staff. ABC, Staff and Dispenser are probably out of sync.");
        }
        return foundPersons;
    }

}
