package ru.yandex.qe.dispenser.ws;


import java.util.Collection;
import java.util.Collections;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import ru.yandex.qe.dispenser.api.v1.DiProperty;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.domain.dao.property.PropertyReader;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.property.Property;
import ru.yandex.qe.dispenser.domain.property.PropertyManager;
import ru.yandex.qe.dispenser.domain.util.ValidationUtils;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;

@Path("/v1/properties")
@Controller
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("properties")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class PropertyService {

    private final static String ENTITY_KEY = "entity";
    private final static String PROPERTY_KEY = "key";

    @Autowired
    private PropertyManager propertyManager;

    @GET
    public DiListResponse<DiProperty> getProperties(@Nullable @QueryParam(ENTITY_KEY) final String entityKey,
                                                    @Nullable @QueryParam(PROPERTY_KEY) final String propertyKey) {
        final PropertyReader propertyReader = Hierarchy.get().getPropertyReader();

        final Collection<Property> properties;

        if (entityKey != null) {
            if (propertyKey != null) {
                properties = propertyReader.read(entityKey, propertyKey)
                        .map(Collections::singleton)
                        .orElse(Collections.emptySet());
            } else {
                properties = propertyReader.read(entityKey);
            }
        } else {
            properties = propertyReader.getAll();
        }

        return new DiListResponse<>(properties.stream()
                .map(Property::toView)
                .collect(Collectors.toList()));
    }

    @PUT
    @Access(dispenserAdmin = true)
    public DiProperty setProperty(final DiProperty.Body body) {
        final Property.Value<?> value = Property.Type.values().stream()
                .map(t -> t.createValue(body.getValue()))
                .filter(Objects::nonNull)
                .findFirst()
                .orElseThrow(() -> new IllegalArgumentException("Invalid property type"));

        final String entityKeyNonEmpty = ValidationUtils.requireNonNull(body.getEntityKey(), "entityKey field should be not empty");
        final String entityKey = ValidationUtils.validateKey(entityKeyNonEmpty, "Invalid entity key");

        final String propertyKeyNonEmpty = ValidationUtils.requireNonNull(body.getPropertyKey(), "propertyKey field should be not empty");
        final String propertyKey = ValidationUtils.validateKey(propertyKeyNonEmpty, "Invalid property key");

        final Property property = propertyManager.setProperty(entityKey, propertyKey, value);

        return property.toView();
    }

    @DELETE
    @Access(dispenserAdmin = true)
    @Path("/{entityKey}/{propertyKey}")
    public Boolean deleteProperty(@PathParam("entityKey") @NotNull final String entityKey,
                                  @PathParam("propertyKey") @NotNull final String propertyKey) {
        return propertyManager.deleteProperty(entityKey, propertyKey);
    }
}
