package ru.yandex.qe.dispenser.ws;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import com.google.common.base.Enums;
import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.web.server.ServerWebInputException;

import ru.yandex.qe.dispenser.api.v1.DiOrder;
import ru.yandex.qe.dispenser.api.v1.DiQuotaLightView;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.domain.EntitySpec;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.property.PropertyReader;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.property.Property;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.d.DLegacyApi;
import ru.yandex.qe.dispenser.ws.param.EntitySpecFilterParam;
import ru.yandex.qe.dispenser.ws.param.ResourceFilterParam;
import ru.yandex.qe.dispenser.ws.v2.QuotaService;

import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.ENTITY_SPEC_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.LEAF_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.MEMBER_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.ORDER_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.PROJECT_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.RESOURCE_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.SEGMENT_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.SERVICE_PARAM;

/**
 * ProxyDServer.
 *
 * @author Vladimir Zaytsev <vzay@yandex-team.ru>
 * @since 21-04-2022
 */
@Path("/v2")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("quota-proxy-2")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value =
        DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class ProxyDServerV2 extends ServiceBase {
    private static final String PROPERTY_ENTITY_KEY = "ProxyDServer";
    private static final String MODE_PROPERTY_KEY = "ProxyDServerMode";

    @Autowired
    @Qualifier("d-tvm")
    private TvmDestination dTvmDestination;

    @SuppressWarnings("SpringJavaInjectionPointsAutowiringInspection")
    @Autowired
    private DLegacyApi dLegacyApi;

    @Autowired
    private QuotaService quotaService;

    public ProxyDServerV2() {
    }


    @GET
    @Path("/quotas")
    public DiListResponse<DiQuotaLightView> read(
            @QueryParam(RESOURCE_PARAM) final List<ResourceFilterParam> resourceParams,
            @QueryParam(ENTITY_SPEC_PARAM) final List<EntitySpecFilterParam> entitySpecParams,
            @QueryParam(SEGMENT_PARAM) final Set<String> segmentKeys,
            @QueryParam(SERVICE_PARAM) final Set<Service> services,
            @QueryParam(PROJECT_PARAM) final Set<Project> projects,
            @QueryParam(MEMBER_PARAM) final Set<Person> members,
            @QueryParam(LEAF_PARAM) @DefaultValue("false") final boolean leafProjects,
            @QueryParam(ORDER_PARAM) final DiOrder order
    ) {
        return switch (getMode()) {
            case OLD_DISPENSER -> quotaService.read(
                    resourceParams, entitySpecParams, segmentKeys, services, projects, members, leafProjects, order
            );
            case OUT_OF_SERVICE -> throw new ServerWebInputException("Service temporary unavailable.");
            case PROXY_TO_D -> dTvmDestination.runAuthorizedProxy(() -> {
                List<String> resourceParamsKeys = resourceParams.stream().map(resourceFilterParam -> {
                    String serviceKey = resourceFilterParam.get().getService().getKey();
                    String resourceKey = resourceFilterParam.get().getPublicKey();
                    return "/" + serviceKey + "/" + resourceKey;
                }).collect(Collectors.toList());
                List<String> entitySpecParamsKeys = entitySpecParams.stream().map(entitySpecFilterParam -> {
                    String serviceKey = entitySpecFilterParam.get().getService().getKey();
                    EntitySpec.Key entitySpecKey = entitySpecFilterParam.get().getKey();
                    return "/" + serviceKey + "/" + entitySpecKey;
                }).collect(Collectors.toList());
                return dLegacyApi.read(
                        resourceParamsKeys,
                        entitySpecParamsKeys,
                        segmentKeys,
                        services.stream().map(Service::getKey).collect(Collectors.toSet()),
                        projects.stream().map(Project::getPublicKey).collect(Collectors.toSet()),
                        members.stream().map(Person::getLogin).collect(Collectors.toSet()),
                        leafProjects,
                        order != null ? order.name() : null
                );
            });
        };
    }

    private Mode getMode() {
        final PropertyReader propertyReader = Hierarchy.get().getPropertyReader();
        return propertyReader.read(PROPERTY_ENTITY_KEY, MODE_PROPERTY_KEY)
                .map((Property p) -> Enums.getIfPresent(Mode.class, p.getValue().getAs(Property.Type.STRING))
                        .or(Mode.OLD_DISPENSER))
                .orElse(Mode.OLD_DISPENSER);
    }

    private enum Mode {
        OLD_DISPENSER, OUT_OF_SERVICE, PROXY_TO_D
    }
}
