package ru.yandex.qe.dispenser.ws;

import java.io.BufferedWriter;
import java.io.OutputStreamWriter;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.function.ToLongFunction;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.BeanParam;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.StreamingOutput;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import com.opencsv.CSVWriter;
import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import ru.yandex.qe.dispenser.api.v1.DiResourcePreorderReasonType;
import ru.yandex.qe.dispenser.api.v1.DiResourceType;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.CampaignOwningCost;
import ru.yandex.qe.dispenser.domain.GoalQuestionHelper;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.ResourceGroup;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.domain.bot.BigOrderConfig;
import ru.yandex.qe.dispenser.domain.dao.bot.SimplePreOrder;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.MappedPreOrderDao;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.change.BotPreOrderChangeDao;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.request.BotPreOrderRequestDao;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.request.ExtendedPreOrderRequest;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignOwningCostCache;
import ru.yandex.qe.dispenser.domain.dao.goal.OkrAncestors;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestReader;
import ru.yandex.qe.dispenser.domain.dao.quota.request.ReportQuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.dao.resource.ResourceReader;
import ru.yandex.qe.dispenser.domain.dictionaries.impl.FrontDictionariesManager;
import ru.yandex.qe.dispenser.domain.dictionaries.model.CampaignProvidersSettingsDictionary;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.domain.index.LongIndexable;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.bot.BigOrderManager;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.param.QuotaChangeRequestFilterParam;
import ru.yandex.qe.dispenser.ws.param.QuotaChangeRequestFilterParamConverter;
import ru.yandex.qe.dispenser.ws.quota.request.ticket.QuotaChangeRequestTicketManager;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.ResourceWorkflow;

import static ru.yandex.qe.dispenser.ws.quota.request.owning_cost.campaignOwningCost.CampaignOwningCostRefreshTransactionWrapper.VALID_STATUSES;
import static ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula.ProviderOwningCostFormula.owningCostToOutputFormat;
import static ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula.ProviderOwningCostFormula.percentageOfCampaignOwningCostOutputString;

@ParametersAreNonnullByDefault
@Path("/v1/quota-requests/export")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("quota-requests-export")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class QuotaChangeRequestExportService {
    public static final String CONTENT_DISPOSITION_TEMPLATE = "attachment; filename=%s";
    private static final String[] STRING_INITIAL_ARRAY = new String[0];
    private static final String ABC_SERVICE_COLUMN = "ABC_SERVICE";
    private static final String BIG_ORDER_COLUMN = "BIG_ORDER";
    private static final String DC_COLUMN = "DC";
    private static final String SERVICE_COLUMN = "PROVIDER";
    private static final String RESPONSIBLES_COLUMN = "RESPONSIBLES";
    private static final String[] HEADS_COLUMNS = {"HEAD_1", "HEAD_2", "HEAD_3"};
    private static final String[] HEAD_DEPARTMENTS_COLUMNS = {"HEAD_DEPARTMENT_1", "HEAD_DEPARTMENT_2", "HEAD_DEPARTMENT_3"};
    private static final String TICKET_COLUMN = "SUBTICKET";
    private static final String STATUS_COLUMN = "STATUS";
    private static final String PREORDER_COLUMN = "BOT_PREORDER";
    private static final String CAMPAIGN_COLUMN = "CAMPAIGN";
    private static final String JUSTIFICATION_COLUMN = "JUSTIFICATION";
    private static final String GOAL_URL_COLUMN = "GOAL_URL";
    private static final String GOAL_NAME_COLUMN = "GOAL_NAME";
    private static final String SUMMARY_COLUMN = "SUMMARY";
    private static final String COST_COLUMN = "COST";
    private static final String PERCENTAGE_OF_CAMPAIGN_OWNING_COST = "PERCENTAGE_OF_CAMPAIGN_OWNING_COST";
    private static final String ABC_SERVICE_SLUG_COLUMN = "ABC_SERVICE_SLUG";
    private static final String IMPORTANCE_COLUMN = "IMPORTANCE";
    private static final String UNBALANCED_COLUMN = "UNBALANCED";

    private static final NumberFormat AMOUNT_FORMAT = new DecimalFormat("#0.##");
    private static final LoadingCache<String, DecimalFormat> CACHED_FORMATS = CacheBuilder.newBuilder()
            .maximumSize(100)
            .expireAfterWrite(1, TimeUnit.HOURS)
            .build(new CacheLoader<>() {
                @Override
                public DecimalFormat load(final String key) {
                    return new DecimalFormat(key);
                }
            });

    private static final Map<DiResourceType, DiUnit> PREFERRED_UNITS = new EnumMap<>(Map.of(
            DiResourceType.STORAGE, DiUnit.TEBIBYTE,
            DiResourceType.MEMORY, DiUnit.GIBIBYTE,
            DiResourceType.PROCESSOR, DiUnit.CORES,
            DiResourceType.ENUMERABLE, DiUnit.COUNT,
            DiResourceType.STORAGE_BASE, DiUnit.TEBIBYTE_BASE,
            DiResourceType.MEMORY_BASE, DiUnit.GIBIBYTE_BASE
    ));
    public static final EnumSet<AmountType> AMOUNT_TYPES = EnumSet.of(AmountType.READY, AmountType.ALLOCATED);

    @Autowired
    private QuotaChangeRequestDao quotaChangeRequestDao;

    @Autowired
    private BotPreOrderChangeDao botPreOrderChangeDao;

    @Value("${goal.service.url}")
    private String goalsUrl;

    @Autowired
    private GoalQuestionHelper goalQuestionHelper;

    @Autowired
    private BotPreOrderRequestDao preOrderRequestDao;

    @Autowired
    private MappedPreOrderDao mappedPreOrderDao;

    @Autowired
    private FrontDictionariesManager frontDictionariesManager;

    @Autowired
    private QuotaChangeRequestFilterParamConverter filterParamConverter;

    @Autowired
    private BigOrderManager bigOrderManager;

    @Autowired
    private CampaignOwningCostCache campaignOwningCostCache;

    @NotNull
    @GET
    @Access
    @Produces(ServiceBase.TEXT_CSV_UTF_8)
    @Path("/csv")
    public Response exportRequestsToCsv(@BeanParam final QuotaChangeRequestFilterParam filterParams,
                                        @QueryParam("allResources") @DefaultValue("false") final Boolean showAllResources,
                                        @Nullable @QueryParam("allResourcesCampaignId") final Long allResourcesCampaignId,
                                        @QueryParam("goalQuestions") @DefaultValue("false") final boolean showGoalQuestions,
                                        @QueryParam("servers") @DefaultValue("false") final boolean showServers,
                                        @QueryParam("cost") @DefaultValue("false") final boolean costRequested,
                                        @QueryParam("showReadyAndAllocated") @DefaultValue("false") final boolean showReadyAndAllocated,
                                        @QueryParam("showGoalHierarchy") @DefaultValue("false") final boolean showGoalHierarchy,
                                        @QueryParam("filterEmptyResources") @DefaultValue("false") final boolean filterEmptyResources,
                                        @QueryParam("withoutReserves") @DefaultValue("false") final boolean costAndServersWithoutReserves,
                                        @QueryParam("withOwningCost") @DefaultValue("true") final boolean withOwningCost) {

        final Optional<QuotaChangeRequestReader.QuotaChangeRequestFilter> requestFilter = filterParamConverter.fromParam(filterParams);
        final List<ReportQuotaChangeRequest> reportRequests = requestFilter
                .map(filter -> quotaChangeRequestDao.readReport(filter, showGoalQuestions, filterEmptyResources, showGoalHierarchy).collect(Collectors.toList()))
                .orElse(Collections.emptyList());

        final Map<Long, QuotaChangeRequest> requestById = reportRequests.stream()
                .collect(Collectors.toMap(r -> r.getRequest().getId(), ReportQuotaChangeRequest::getRequest));

        final Table<Long, Long, Double> preOrderIdsByChangeId = botPreOrderChangeDao.getPreOrderIdsByChangeId(requestById.keySet());

        final Table<RequestPartKey, String, Double> requestConfigurationCount = HashBasedTable.create();
        final List<String> configurations = new ArrayList<>();

        final Person performer = Session.WHOAMI.get();

        final Table<RequestPartKey, SimplePreOrder, Double> requestConfigurationCost = HashBasedTable.create();

        if (showServers || costRequested) {
            final Set<Long> usedOrderIds = reportRequests.stream()
                    .flatMap(r -> r.getRequest().getChanges().stream())
                    .map(QuotaChangeRequest.ChangeAmount::getBigOrder)
                    .filter(Objects::nonNull)
                    .map(QuotaChangeRequest.BigOrder::getId)
                    .collect(Collectors.toSet());

            final Set<BigOrder> bigOrders = bigOrderManager.getByIds(usedOrderIds);
            final Map<Long, BigOrderConfig> configById = bigOrders.stream()
                    .flatMap(b -> b.getConfigs().stream())
                    .collect(Collectors.toMap(BigOrderConfig::getId, c -> c));

            final Table<Long, Long, ExtendedPreOrderRequest> preOrderRequestCost = HashBasedTable.create();
            final Collection<ExtendedPreOrderRequest> preOrderRequests = preOrderRequestDao.getRequestPreorders(requestById.keySet());
            for (final ExtendedPreOrderRequest preOrderRequest : preOrderRequests) {
                preOrderRequestCost.put(
                        preOrderRequest.getPreOrderId(),
                        preOrderRequest.getRequestId(),
                        preOrderRequest
                );
            }
            final Map<Long, SimplePreOrder> preOrderById = new HashMap<>();
            if (costRequested) {
                final Collection<SimplePreOrder> simplePreOrders = mappedPreOrderDao.readSimplePreOrders(preOrderRequestCost.rowKeySet());
                for (final SimplePreOrder simplePreOrder : simplePreOrders) {
                    preOrderById.put(simplePreOrder.getId(), simplePreOrder);
                }
            }
            for (final QuotaChangeRequest request : requestById.values()) {

                final boolean showCost = costRequested && ResourceWorkflow.canUserViewBotMoney(performer);

                final Map<RequestPartKey, List<QuotaChangeRequest.Change>> parts = request.getChanges().stream()
                        .collect(Collectors.groupingBy(c -> RequestPartKey.of(request.getId(), c)));

                final Table<RequestPartKey, Long, Double> partOrderProportion = HashBasedTable.create();
                final Map<Long, Double> preOrderProportionSum = new HashMap<>();

                for (final RequestPartKey partKey : parts.keySet()) {
                    final Map<Long, Double> partInPreOrder = new HashMap<>();
                    for (final QuotaChangeRequest.Change change : parts.get(partKey)) {
                        final Map<Long, Double> orderProportion = preOrderIdsByChangeId.row(change.getId());
                        for (final Long orderId : orderProportion.keySet()) {
                            partInPreOrder.put(orderId, Math.max(partInPreOrder.getOrDefault(orderId, 0.0), orderProportion.get(orderId)));
                        }
                    }
                    for (final Long orderId : partInPreOrder.keySet()) {
                        preOrderProportionSum.put(orderId, preOrderProportionSum.getOrDefault(orderId, 0.0) + partInPreOrder.get(orderId));
                    }
                    partOrderProportion.row(partKey).putAll(partInPreOrder);
                }

                for (final RequestPartKey partKey : parts.keySet()) {
                    final Map<Long, Double> partInPreOrder = partOrderProportion.row(partKey);

                    for (final Long preOrderId : partInPreOrder.keySet()) {
                        final Double preOrderSum = preOrderProportionSum.getOrDefault(preOrderId, 0.0);
                        final double preorderProportionMultiplier = preOrderSum == 0 ? 0 : 1 / preOrderSum;
                        final ExtendedPreOrderRequest preOrderRequest = preOrderRequestCost.get(preOrderId, request.getId());
                        if (preOrderRequest != null) {
                            Double sum = requestConfigurationCount.get(partKey, preOrderRequest.getConfigurationName());
                            if (sum == null) {
                                sum = 0.0;
                            }
                            double preOrderMultiplier = partInPreOrder.get(preOrderId) * preorderProportionMultiplier;
                            if (costAndServersWithoutReserves) {
                                final double nonReserveRate = 1.0 - preOrderRequest.getReserveRate();
                                final double nonReserveMultiplier = nonReserveRate == 0.0 ? 0.0 : 1.0 / nonReserveRate;
                                preOrderMultiplier *= nonReserveMultiplier;
                            }
                            if (showServers) {
                                final long bigOrderConfigId = preOrderRequest.getBigOrderConfigId();
                                final BigOrderConfig config = configById.get(bigOrderConfigId);
                                String configurationName = preOrderRequest.getConfigurationName();
                                if (config != null && config.isUpgrade()) {
                                    configurationName = "UPGRADE " + configurationName;
                                }
                                requestConfigurationCount.put(partKey, configurationName,
                                        sum + preOrderRequest.getServersQuantity() * preOrderMultiplier);
                            }
                            if (showCost) {
                                final double cost = preOrderRequest.getCost() * preOrderMultiplier;
                                requestConfigurationCost.put(partKey, preOrderById.get(preOrderId), cost);
                            }
                        }
                    }
                }
            }
            if (showServers) {
                configurations.addAll(requestConfigurationCount.columnKeySet());
            }
        }

        Map<Long, BigInteger> owningCostByCampaignId = campaignOwningCostCache.getAll().stream()
                .collect(Collectors.toMap(CampaignOwningCost::getCampaignId, CampaignOwningCost::getOwningCost));

        final List<String> headers = new ArrayList<>(Arrays.asList(ABC_SERVICE_COLUMN, BIG_ORDER_COLUMN, RESPONSIBLES_COLUMN,
                HEADS_COLUMNS[0], HEAD_DEPARTMENTS_COLUMNS[0], HEADS_COLUMNS[1], HEAD_DEPARTMENTS_COLUMNS[1],
                HEADS_COLUMNS[2], HEAD_DEPARTMENTS_COLUMNS[2], SERVICE_COLUMN, STATUS_COLUMN, IMPORTANCE_COLUMN, UNBALANCED_COLUMN,
                PERCENTAGE_OF_CAMPAIGN_OWNING_COST, TICKET_COLUMN, DC_COLUMN, PREORDER_COLUMN, CAMPAIGN_COLUMN, JUSTIFICATION_COLUMN,
                GOAL_URL_COLUMN, GOAL_NAME_COLUMN, SUMMARY_COLUMN, ABC_SERVICE_SLUG_COLUMN));
        final List<Object> keys = new ArrayList<>(headers);
        final List<Map<Object, String>> listOfRows = new ArrayList<>();

        final Map<Pair<String, Long>, Pair<String, UUID>> configServiceToHeaderNameId = new HashMap<>();
        final Map<Long, Pair<String, UUID>> configPreorderCostToHeaderNameId = new HashMap<>();

        final StreamingOutput output = outputStream -> {
            final BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(outputStream));
            final CSVWriter csvWriter = new CSVWriter(writer);

            final Set<Resource> resources = new HashSet<>();

            for (final ReportQuotaChangeRequest reportRequest : reportRequests) {
                final QuotaChangeRequest request = reportRequest.getRequest();
                final Map<Object, String> staticPartOfRow = createStaticPartOfRow(request, owningCostByCampaignId);

                if (showGoalQuestions) {
                    final Map<Long, String> goalData = reportRequest.getGoalData();
                    for (final Long answerId : goalData.keySet()) {
                        final DiResourcePreorderReasonType resourcePreorderReasonType = request.getResourcePreorderReasonType();
                        if (resourcePreorderReasonType == DiResourcePreorderReasonType.GROWTH || resourcePreorderReasonType == DiResourcePreorderReasonType.GOAL) {
                            final GoalQuestionHelper.RequestGoalQuestionId questionId
                                    = goalQuestionHelper.getRequestGoalQuestionId(resourcePreorderReasonType, answerId);
                            staticPartOfRow.put(questionId, goalData.get(answerId));
                        }
                    }
                }
                if (showGoalHierarchy) {
                    final Map<OkrAncestors.OkrType, ReportQuotaChangeRequest.Goal> goalHierarchy = reportRequest.getGoalHierarchy();
                    for (final OkrAncestors.OkrType okrType : goalHierarchy.keySet()) {
                        final ReportQuotaChangeRequest.Goal goal = goalHierarchy.get(okrType);
                        staticPartOfRow.put(okrType.name() + "_url", getGoalUrl(goal.getId()));
                        staticPartOfRow.put(okrType.name() + "_name", goal.getName());
                    }
                }

                final Map<RequestPartKey, List<QuotaChangeRequest.Change>> parts = request.getChanges().stream()
                        .collect(Collectors.groupingBy(c -> RequestPartKey.of(request.getId(), c)));

                for (final RequestPartKey partKey : parts.keySet()) {
                    final Map<Object, String> row = new HashMap<>(staticPartOfRow);

                    final Set<Segment> segments = partKey.getSegments();

                    final String segmentsView = segments.stream()
                            .map(Segment::getName)
                            .sorted()
                            .collect(Collectors.joining(", "));

                    final List<QuotaChangeRequest.Change> changes = parts.get(partKey);
                    final Map<Pair<Resource, AmountType>, String> part = createResourcesPartOfRow(changes, AmountType.ORDERED);

                    String bigOrder = "";
                    if (partKey.getOrder() != null) {
                        bigOrder = partKey.getOrder().getDate().format(DateTimeFormatter.ISO_LOCAL_DATE);
                    }
                    row.put(BIG_ORDER_COLUMN, bigOrder);
                    row.put(SERVICE_COLUMN, partKey.getService().getName());

                    if (showServers) {
                        for (final String configuration : configurations) {

                            final Double amount = requestConfigurationCount.get(partKey, configuration);
                            if (amount != null) {
                                final Service service = partKey.getService();
                                final Pair<String, Long> key = Pair.of(configuration, service.getId());

                                if (!configServiceToHeaderNameId.containsKey(key)) {
                                    configServiceToHeaderNameId.put(key, Pair.of(service.getName() + " : " + configuration, UUID.randomUUID()));
                                }
                                final Pair<String, UUID> stringUUIDPair = configServiceToHeaderNameId.get(key);

                                final String view = formatAmount(amount);
                                row.put(stringUUIDPair.getRight(), view);
                            }
                        }
                    }

                    final boolean showCost = ResourceWorkflow.canUserViewMoney(performer);
                    if (showCost) {
                        final Map<SimplePreOrder, Double> costsForRow = requestConfigurationCost.row(partKey);
                        double sum = 0.0;
                        for (final SimplePreOrder preOrder : costsForRow.keySet()) {
                            final Long preOrderId = preOrder.getId();
                            if (!configPreorderCostToHeaderNameId.containsKey(preOrderId)) {
                                final String title = String.format("%s : %s : Cost in %s %s preorder %d", preOrder.getService().getName(),
                                        preOrder.getConfigurationName(), preOrder.getBigOrderDate(), preOrder.getBigOrderLocation(), preOrderId);

                                configPreorderCostToHeaderNameId.put(preOrderId, Pair.of(title, UUID.randomUUID()));
                            }
                            final Pair<String, UUID> stringUUIDPair = configPreorderCostToHeaderNameId.get(preOrderId);

                            final Double amount = costsForRow.get(preOrder);
                            if (amount == null) {
                                continue;
                            }
                            sum += amount;
                            final String view = formatAmount(amount);
                            row.put(stringUUIDPair.getRight(), view);
                        }
                        row.put(COST_COLUMN, String.valueOf(sum));
                    }

                    if (withOwningCost && showCost) {
                        row.putAll(createResourcesOwningCostPartOfRow(changes));
                    }

                    if (showReadyAndAllocated) {
                        for (final AmountType value : AMOUNT_TYPES) {
                            row.putAll(createResourcesPartOfRow(changes, value));
                        }
                    }

                    final String preOrders = changes.stream()
                            .map(LongIndexable::getId)
                            .flatMap(id -> preOrderIdsByChangeId.row(id).keySet().stream())
                            .filter(Objects::nonNull)
                            .distinct()
                            .map(Object::toString)
                            .collect(Collectors.joining(", "));

                    row.put(PREORDER_COLUMN, preOrders);
                    row.put(DC_COLUMN, segmentsView);
                    resources.addAll(part.keySet().stream().map(Pair::getKey).collect(Collectors.toSet()));

                    row.putAll(part);

                    listOfRows.add(row);
                }

            }

            if (showAllResources) {
                final ResourceReader resourceReader = Hierarchy.get().getResourceReader();
                final CampaignProvidersSettingsDictionary settings = frontDictionariesManager.getCampaignProviderSettings(allResourcesCampaignId);
                final Stream<Resource> allResources = Optional.ofNullable(settings)
                        .map(s -> s.getProviders()
                                .stream().flatMap(p -> p.getResources().stream())
                                .map(r -> resourceReader.read(r.getId()))
                        ).orElseGet(() -> Stream.of(Provider.values())
                                .map(Provider::getService)
                                .filter(Objects::nonNull)
                                .flatMap(s -> resourceReader.getByService(s).stream())
                        );

                allResources.forEach(resources::add);
            }

            final ArrayList<Resource> sortedResources = new ArrayList<>(resources);
            Collections.sort(sortedResources);

            final Map<Resource, String> nameByResource = new HashMap<>();
            final Map<Resource, String> nameByResourceForOwningCost = new HashMap<>();
            for (final Resource resource : sortedResources) {
                final StringBuilder resourceName = new StringBuilder()
                        .append(resource.getService().getName())
                        .append(" ")
                        .append(resource.getName());

                final ResourceGroup group = resource.getGroup();
                if (group != null) {
                    resourceName.append("-").append(group.getName());
                }

                nameByResourceForOwningCost.put(resource, resourceName.toString());

                final DiUnit preferredUnits = PREFERRED_UNITS.get(resource.getType());
                if (preferredUnits != null) {
                    resourceName.append(" (")
                            .append(preferredUnits.getAbbreviation())
                            .append(")");
                }

                final String name = resourceName.toString();
                nameByResource.put(resource, name);

                headers.add(name);
                keys.add(Pair.of(resource, AmountType.ORDERED));
            }

            if (withOwningCost) {
                for (final Resource resource : sortedResources) {
                    final String name = nameByResourceForOwningCost.get(resource) + ' ' + AmountType.OWNING_COST.name();

                    headers.add(name);
                    keys.add(Pair.of(resource, AmountType.OWNING_COST));
                }
            }

            if (showGoalQuestions) {
                for (final GoalQuestionHelper.RequestGoalQuestionId questionId : goalQuestionHelper.getRequestGoalQuestionIds()) {
                    keys.add(questionId);
                    headers.add(goalQuestionHelper.getRequestGoalQuestion(questionId));
                }
            }

            if (showServers) {
                for (final Map.Entry<?, Pair<String, UUID>> entry : configServiceToHeaderNameId.entrySet()) {

                    final Pair<String, UUID> value = entry.getValue();
                    final UUID keyId = value.getRight();
                    final String name = value.getLeft();

                    keys.add(keyId);
                    headers.add(name);
                }
            }

            if (costRequested) {
                headers.add(COST_COLUMN);
                keys.add(COST_COLUMN);

                for (final Map.Entry<?, Pair<String, UUID>> entry : configPreorderCostToHeaderNameId.entrySet()) {

                    final Pair<String, UUID> value = entry.getValue();
                    final UUID keyId = value.getRight();
                    final String name = value.getLeft();

                    keys.add(keyId);
                    headers.add(name);
                }
            }

            if (showReadyAndAllocated) {
                for (final AmountType value : AMOUNT_TYPES) {
                    for (final Resource resource : sortedResources) {
                        final String name = nameByResource.get(resource) + ' ' + value.name();

                        headers.add(name);
                        keys.add(Pair.of(resource, value));
                    }
                }
            }

            if (showGoalHierarchy) {
                for (final OkrAncestors.OkrType okrType : OkrAncestors.OkrType.values()) {
                    headers.add(okrType.name() + ": Url");
                    keys.add(okrType.name() + "_url");
                    headers.add(okrType.name() + ": Name");
                    keys.add(okrType.name() + "_name");
                }
            }

            csvWriter.writeNext(headers.toArray(STRING_INITIAL_ARRAY));

            for (final Map<Object, String> row : listOfRows) {
                final List<String> params = new ArrayList<>();
                for (final Object key : keys) {
                    params.add(row.get(key));
                }

                csvWriter.writeNext(params.toArray(STRING_INITIAL_ARRAY));
            }

            writer.close();
        };
        return Response.ok(output)
                .header(HttpHeaders.CONTENT_DISPOSITION, String.format(CONTENT_DISPOSITION_TEMPLATE, "requests.csv"))
                .build();
    }

    public Map<Object, String> createStaticPartOfRow(final QuotaChangeRequest quotaChangeRequest,
                                                     Map<Long, BigInteger> owningCostByCampaignId) {
        final Project requestProject = quotaChangeRequest.getProject();
        final List<Project> pathFromRoot = requestProject.getPathFromRoot();

        final List<Set<Person>> heads = new ArrayList<>();
        final List<String> headDepartments = new ArrayList<>();
        for (final Project project : pathFromRoot) {
            final Set<Person> projectResponsibles = Hierarchy.get().getProjectReader().getLinkedResponsibles(project);
            heads.add(projectResponsibles);
            headDepartments.add(project.getName());
        }
        final Map<Object, String> row = new HashMap<>();
        row.put(ABC_SERVICE_COLUMN, requestProject.getName());
        row.put(ABC_SERVICE_SLUG_COLUMN, requestProject.getKey().getPublicKey());
        row.put(RESPONSIBLES_COLUMN, QuotaChangeRequestTicketManager.getAssignee(quotaChangeRequest, Hierarchy.get()));
        for (int i = 1; i <= HEADS_COLUMNS.length; i++) {
            if (i < pathFromRoot.size()) {
                row.put(HEADS_COLUMNS[i - 1], heads.get(i).stream().map(Person::getLogin).collect(Collectors.joining(",")));
                row.put(HEAD_DEPARTMENTS_COLUMNS[i - 1], (headDepartments.get(i)));
            } else {
                row.put(HEADS_COLUMNS[i - 1], "");
                row.put(HEAD_DEPARTMENTS_COLUMNS[i - 1], "");
            }
        }
        row.put(STATUS_COLUMN, quotaChangeRequest.getStatus().name());
        row.put(IMPORTANCE_COLUMN, String.valueOf(quotaChangeRequest.isImportantRequest()));
        row.put(UNBALANCED_COLUMN, String.valueOf(quotaChangeRequest.isUnbalanced()));
        row.put(PERCENTAGE_OF_CAMPAIGN_OWNING_COST, owningCostByCampaignId.containsKey(quotaChangeRequest.getCampaignId()) && VALID_STATUSES.contains(quotaChangeRequest.getStatus())
                ? percentageOfCampaignOwningCostOutputString(quotaChangeRequest.getRequestOwningCost(), owningCostByCampaignId.get(quotaChangeRequest.getCampaignId()))
                : "");
        row.put(TICKET_COLUMN, quotaChangeRequest.getTrackerIssueKey());
        final String campaignKey = quotaChangeRequest.getCampaign() != null ? quotaChangeRequest.getCampaign().getKey() : "";
        row.put(CAMPAIGN_COLUMN, campaignKey);
        final String justification = quotaChangeRequest.getResourcePreorderReasonType() != null
                ? quotaChangeRequest.getResourcePreorderReasonType().name() : "";
        row.put(JUSTIFICATION_COLUMN, justification);
        final String goalUrl = quotaChangeRequest.getGoal() != null ? getGoalUrl(quotaChangeRequest.getGoal().getId()) : "";
        row.put(GOAL_URL_COLUMN, goalUrl);
        final String goalName = quotaChangeRequest.getGoal() != null ? quotaChangeRequest.getGoal().getName() : "";
        row.put(GOAL_NAME_COLUMN, goalName);
        row.put(SUMMARY_COLUMN, quotaChangeRequest.getSummary() != null ? quotaChangeRequest.getSummary() : "");

        return row;
    }

    @NotNull
    private String getGoalUrl(final long goalId) {
        return goalsUrl + "/filter?goal=" + goalId;
    }

    public static Map<Pair<Resource, AmountType>, String> createResourcesPartOfRow(final List<QuotaChangeRequest.Change> changes,
                                                                                   final AmountType amountType) {
        final Map<Pair<Resource, AmountType>, String> rowRightPart = new HashMap<>();
        for (final QuotaChangeRequest.Change change : changes) {
            final Resource resource = change.getResource();

            final DiResourceType type = resource.getType();
            final DiUnit preferredUnits = PREFERRED_UNITS.get(type);

            final String value;

            final long amount = amountType.getExtractor().applyAsLong(change);
            if (preferredUnits != null) {
                final DiUnit baseUnit = type.getBaseUnit();
                final double amountInPreferredUnits = preferredUnits.convert((double) amount, baseUnit);
                value = formatAmount(amountInPreferredUnits);
            } else {
                value = String.valueOf(amount);
            }
            rowRightPart.put(Pair.of(resource, amountType), value);
        }
        return rowRightPart;
    }

    public static Map<Pair<Resource, AmountType>, String> createResourcesOwningCostPartOfRow(final List<QuotaChangeRequest.Change> changes) {
        final Map<Pair<Resource, AmountType>, String> rowRightPart = new HashMap<>();
        for (final QuotaChangeRequest.Change change : changes) {
            final Resource resource = change.getResource();

            final String value = String.valueOf(AmountType.OWNING_COST.extractor.applyAsLong(change));
            rowRightPart.put(Pair.of(resource, AmountType.OWNING_COST), value);
        }
        return rowRightPart;
    }

    private static String formatAmount(final double amount) {
        if (Math.abs(amount) >= 0.01d) {
            return AMOUNT_FORMAT.format(amount);
        }
        final int zeroPaddingLength = countLeadingZeroesInFractionalPart(amount);
        final String format = "#0." + StringUtils.repeat('#', zeroPaddingLength + 2);
        return CACHED_FORMATS.getUnchecked(format).format(amount);
    }

    private static int countLeadingZeroesInFractionalPart(final double value) {
        final BigDecimal decimalValue = BigDecimal.valueOf(Math.abs(value));
        final BigDecimal fractionalPart = decimalValue.subtract(new BigDecimal(decimalValue.toBigInteger()));
        if (fractionalPart.compareTo(BigDecimal.ZERO) == 0) {
            return 0;
        }
        int leadingZeroes = 0;
        BigDecimal currentValue = fractionalPart;
        while (currentValue.compareTo(BigDecimal.ONE) < 0) {
            currentValue = currentValue.multiply(BigDecimal.TEN);
            leadingZeroes++;
        }
        leadingZeroes--;
        return leadingZeroes;
    }

    public static class RequestPartKey {
        private final Long requestId;
        private final Service service;
        @Nullable
        private final QuotaChangeRequest.BigOrder order;
        private final Set<Segment> segments;

        public static RequestPartKey of(final Long requestId, final QuotaChangeRequest.Change change) {
            return new RequestPartKey(requestId, change.getResource().getService(), change.getKey().getBigOrder(), change.getSegments());
        }

        private RequestPartKey(final Long requestId, final Service service, @Nullable final QuotaChangeRequest.BigOrder order, final Set<Segment> segments) {
            this.requestId = requestId;
            this.service = service;
            this.order = order;
            this.segments = segments;
        }

        public Long getRequestId() {
            return requestId;
        }

        public Service getService() {
            return service;
        }

        @Nullable
        public QuotaChangeRequest.BigOrder getOrder() {
            return order;
        }

        public Set<Segment> getSegments() {
            return segments;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final RequestPartKey that = (RequestPartKey) o;
            return Objects.equals(requestId, that.requestId) && Objects.equals(service, that.service) && Objects.equals(order, that.order) && Objects.equals(segments, that.segments);
        }

        @Override
        public int hashCode() {
            return Objects.hash(requestId, service, order, segments);
        }
    }

    private enum AmountType {
        ORDERED(QuotaChangeRequest.Change::getAmount),
        READY(QuotaChangeRequest.Change::getAmountReady),
        ALLOCATED(QuotaChangeRequest.Change::getAmountAllocated),
        OWNING_COST((change) -> owningCostToOutputFormat(change.getOwningCost()).longValue());

        private final ToLongFunction<QuotaChangeRequest.Change> extractor;

        AmountType(final ToLongFunction<QuotaChangeRequest.Change> extractor) {
            this.extractor = extractor;
        }

        public ToLongFunction<QuotaChangeRequest.Change> getExtractor() {
            return extractor;
        }
    }
}
