package ru.yandex.qe.dispenser.ws;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.BeanParam;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.PATCH;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.UriInfo;

import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeRequest;
import ru.yandex.qe.dispenser.api.v1.ExpandQuotaChangeRequest;
import ru.yandex.qe.dispenser.api.v1.request.quota.Body;
import ru.yandex.qe.dispenser.api.v1.request.quota.BodyUpdate;
import ru.yandex.qe.dispenser.api.v1.request.unbalance.DiQuotaChangeRequestUnbalancedContext;
import ru.yandex.qe.dispenser.api.v1.request.unbalance.DiQuotaChangeRequestUnbalancedResult;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceChange;
import ru.yandex.qe.dispenser.domain.base_resources.WithBaseResourceChanges;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestReader;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.util.Page;
import ru.yandex.qe.dispenser.domain.util.PageInfo;
import ru.yandex.qe.dispenser.domain.util.ValidationUtils;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.param.PaginationParam;
import ru.yandex.qe.dispenser.ws.param.QuotaChangeRequestFilterParam;
import ru.yandex.qe.dispenser.ws.param.QuotaChangeRequestFilterParamConverter;
import ru.yandex.qe.dispenser.ws.quota.request.QuotaChangeRequestManager;
import ru.yandex.qe.dispenser.ws.quota.request.ticket.QuotaChangeRequestTicketManager;
import ru.yandex.qe.dispenser.ws.quota.request.unbalanced.QuotaChangeRequestUnbalancedResult;
import ru.yandex.qe.dispenser.ws.quota.request.unbalanced.QuotaChangeUnbalancedManager;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.QuotaRequestWorkflowManager;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.ResourceWorkflow;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.CreateRequestContext;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.PerformerContext;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.RequestContext;

import static ru.yandex.qe.dispenser.ws.param.PageParam.LIMIT_PARAM;
import static ru.yandex.qe.dispenser.ws.param.PageParam.PAGE_PARAM;
import static ru.yandex.qe.dispenser.ws.param.PageParam.PAGE_SIZE_PARAM;


@ParametersAreNonnullByDefault
@Path("/v1/quota-requests")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("quota-requests")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class QuotaChangeRequestService extends ServiceBase {
    public static final String REQUEST_ID = "request_id";
    public static final String STATUS = "status";
    public static final String EXPAND = "expand";
    public static final String CAMPAIGN = "campaign";
    public static final String SUPPRESS_SUMMON = "suppressSummon";

    @Autowired
    private QuotaRequestWorkflowManager quotaRequestWorkflowManager;
    @Autowired
    private QuotaChangeRequestServiceHelper quotaChangeRequestServiceHelper;

    @Autowired
    private QuotaChangeRequestTicketManager quotaChangeRequestTicketManager;

    @Autowired
    private QuotaRequestSerializer quotaRequestSerializer;

    @Autowired
    private QuotaChangeRequestDao quotaChangeRequestDao;

    @Autowired
    private QuotaChangeRequestManager quotaChangeRequestManager;

    @Autowired
    private QuotaChangeRequestFilterParamConverter filterParamConverter;

    @Autowired
    private CampaignDao campaignDao;

    @Autowired
    private QuotaChangeUnbalancedManager quotaChangeUnbalancedManager;

    @NotNull
    @GET
    @Access
    public DiListResponse<DiQuotaChangeRequest> getRequests(@BeanParam final QuotaChangeRequestFilterParam filterParams,
                                                            @QueryParam("pagination") final boolean pagination,
                                                            @QueryParam(PAGE_PARAM) final Long pageNumberParam,
                                                            @QueryParam(PAGE_SIZE_PARAM) final Long pageSizeParam,
                                                            @QueryParam(LIMIT_PARAM) final Long limitParam,
                                                            @QueryParam(EXPAND) Set<ExpandQuotaChangeRequest> expand,
                                                            @Context final UriInfo uriInfo) {
        final PaginationParam paginationParam = PaginationParam.from(pageNumberParam, pageSizeParam, limitParam, uriInfo);
        final Optional<QuotaChangeRequestReader.QuotaChangeRequestFilter> filter = filterParamConverter.fromParam(filterParams);
        ServiceEndpointUtils.memoizeServiceForChangeRequestFilter(filterParams);
        final Person performer = Session.WHOAMI.get();

        if (pagination) {
            final PageInfo pageInfo = getPageInfo(paginationParam.getPageParam());

            final Page<QuotaChangeRequest> page = filter.map(f -> quotaChangeRequestDao.readPage(f, pageInfo))
                    .orElse(Page.empty());

            return toResponsePageMulti(paginationParam, page, items -> {
                Set<Long> quotaRequestIds = items.stream().map(LongIndexBase::getId).collect(Collectors.toSet());
                Map<Long, Set<BaseResourceChange>> baseResourceChanges = quotaChangeRequestManager
                        .getBaseResourceChanges(quotaRequestIds, expand);
                return quotaRequestSerializer.toViewCollection(items, performer, expand, baseResourceChanges);
            });
        }

        final List<QuotaChangeRequest> requests = filter.map(quotaChangeRequestDao::read)
                .orElse(Collections.emptyList());

        Set<Long> quotaRequestIds = requests.stream().map(LongIndexBase::getId).collect(Collectors.toSet());
        Map<Long, Set<BaseResourceChange>> baseResourceChanges = quotaChangeRequestManager
                .getBaseResourceChanges(quotaRequestIds, expand);
        return quotaRequestSerializer.toView(requests, performer, expand, baseResourceChanges);
    }

    @NotNull
    @GET
    @Access
    @Path("{" + REQUEST_ID + "}")
    public DiQuotaChangeRequest getRequest(@PathParam(REQUEST_ID) @NotNull final QuotaChangeRequest request,
                                           @QueryParam(EXPAND) Set<ExpandQuotaChangeRequest> expand) {
        ServiceEndpointUtils.memoizeServiceForChangeRequest(request);
        final Person performer = Session.WHOAMI.get();
        Set<BaseResourceChange> baseResourceChanges = quotaChangeRequestManager
                .getBaseResourceChanges(request.getId(), expand);
        return quotaRequestSerializer.toView(request, performer, expand, baseResourceChanges);
    }

    @NotNull
    @POST
    @Access
    public DiListResponse<DiQuotaChangeRequest> createRequests(@QueryParam(EXPAND) Set<ExpandQuotaChangeRequest> expand,
                                                               @QueryParam(CAMPAIGN) Long campaignId,
                                                               @QueryParam(SUPPRESS_SUMMON) @DefaultValue("false") boolean suppressSummon,
                                                               @NotNull final Body body) {
        final Person person = Session.WHOAMI.get();
        if (body.getType() != DiQuotaChangeRequest.Type.RESOURCE_PREORDER) {
            throw new IllegalArgumentException("Unsupported request type");
        }
        final Campaign campaign = quotaChangeRequestServiceHelper.tryLoadCampaignWithDefault(campaignId);
        final WithBaseResourceChanges<List<QuotaChangeRequest>> requests = quotaChangeRequestServiceHelper
                .createRequests(body, person, campaign);
        return quotaRequestSerializer.toView(requests.getValue(), person, expand, requests.getChangesByQuotaRequest());
    }

    @NotNull
    @POST
    @Access
    @Path("batch")
    public DiListResponse<DiQuotaChangeRequest> batchCreateRequests(@QueryParam(EXPAND) Set<ExpandQuotaChangeRequest> expand,
                                                                    @QueryParam(CAMPAIGN) Long campaignId,
                                                                    @QueryParam(SUPPRESS_SUMMON) @DefaultValue("false") boolean suppressSummon,
                                                                    @NotNull final Collection<Body> bodies) {
        bodies.forEach(ValidationUtils::validateQuotaChangeRequestCreateBody);
        ServiceEndpointUtils.memoizeServiceForChangeRequests(bodies);
        if (bodies.isEmpty()) {
            return new DiListResponse<>(Collections.emptyList());
        }

        final Person author = Session.WHOAMI.get();

        QuotaChangeRequest.Type requestType = null;

        for (final Body body : bodies) {

            final QuotaChangeRequest.Type type = QuotaChangeRequest.Type.of(body.getType());

            if (requestType == null) {
                requestType = type;
            }
            if (type != requestType) {
                throw new IllegalArgumentException("Batch creating requests mast have same type");
            }
        }
        if (requestType != QuotaChangeRequest.Type.RESOURCE_PREORDER) {
            throw new IllegalArgumentException("Unsupported request type");
        }

        final Campaign campaign = quotaChangeRequestServiceHelper.tryLoadCampaignWithDefault(campaignId);
        final ResourceWorkflow workflow = quotaRequestWorkflowManager.getResourceWorkflow();

        final Collection<CreateRequestContext> contexts = workflow.buildCreateRequestContext(bodies, author, campaign);

        final WithBaseResourceChanges<List<QuotaChangeRequest>> requests = workflow.createRequests(contexts);

        return quotaRequestSerializer.toView(requests.getValue(), author, expand, requests.getChangesByQuotaRequest());
    }

    @NotNull
    @Access
    @PUT
    @Path("{" + REQUEST_ID + "}/status/{" + STATUS + "}")
    public DiQuotaChangeRequest setStatus(@PathParam(REQUEST_ID) @NotNull final QuotaChangeRequest request,
                                          @PathParam(STATUS) @NotNull final DiQuotaChangeRequest.Status diStatus,
                                          @QueryParam(SUPPRESS_SUMMON) @DefaultValue("false") boolean suppressSummon,
                                          @QueryParam(EXPAND) Set<ExpandQuotaChangeRequest> expand) {
        final Person person = Session.WHOAMI.get();
        final QuotaChangeRequest modifiedRequest = quotaChangeRequestServiceHelper.setStatus(request, diStatus, person,
                suppressSummon);
        Set<BaseResourceChange> baseResourceChanges = quotaChangeRequestManager
                .getBaseResourceChanges(request.getId(), expand);
        return quotaRequestSerializer.toView(modifiedRequest, person, expand, baseResourceChanges);
    }

    @Access
    @POST
    @Path("{" + REQUEST_ID + "}/create-ticket")
    public DiQuotaChangeRequest createTicketIfNotExists(@PathParam(REQUEST_ID) @NotNull final QuotaChangeRequest request,
                                                        @QueryParam(EXPAND) Set<ExpandQuotaChangeRequest> expand) {
        ServiceEndpointUtils.memoizeServiceForChangeRequest(request);

        if (request.getTrackerIssueKey() != null) {
            throw new IllegalArgumentException("Ticket already created");
        }

        final Person person = Session.WHOAMI.get();

        if (request.getType() != QuotaChangeRequest.Type.RESOURCE_PREORDER) {
            throw new IllegalArgumentException("Unsupported request type");
        }

        final ResourceWorkflow workflow = quotaRequestWorkflowManager.getResourceWorkflow();
        workflow.canUserUpdateRequest(request, new RequestContext(person, request));

        final String trackerIssueKey = quotaChangeRequestTicketManager.createTicketForQuotaChangeRequest(request);
        if (trackerIssueKey == null) {
            throw new IllegalArgumentException("Can't create ticket");
        }

        final WithBaseResourceChanges<QuotaChangeRequest> modifiedRequest = quotaChangeRequestManager
                .updateIssueKey(request, trackerIssueKey, new PerformerContext(person), true);
        return quotaRequestSerializer.toView(modifiedRequest.getValue(), person, expand,
                modifiedRequest.getChangesByQuotaRequest().getOrDefault(modifiedRequest.getValue().getId(), Set.of()));
    }

    @NotNull
    @Access
    @PATCH
    @Path("{" + REQUEST_ID + "}")
    public DiQuotaChangeRequest updateRequest(@PathParam(REQUEST_ID) @NotNull final QuotaChangeRequest request,
                                              @QueryParam(EXPAND) Set<ExpandQuotaChangeRequest> expand,
                                              @QueryParam(SUPPRESS_SUMMON) @DefaultValue("false") boolean suppressSummon,
                                              @NotNull final BodyUpdate body) {
        final Person person = Session.WHOAMI.get();
        final WithBaseResourceChanges<QuotaChangeRequest> updatedRequest = quotaChangeRequestServiceHelper
                .updateRequest(request, body, person, suppressSummon);
        return quotaRequestSerializer.toView(updatedRequest.getValue(), person, expand,
                updatedRequest.getChangesByQuotaRequest().getOrDefault(updatedRequest.getValue().getId(), Set.of()));
    }

    @NotNull
    @Access
    @PUT
    @Path("unbalance/dryRun")
    public DiQuotaChangeRequestUnbalancedResult unbalanceDryRun(@NotNull final DiQuotaChangeRequestUnbalancedContext body) {
        QuotaChangeRequestUnbalancedResult calculate = quotaChangeUnbalancedManager.calculate(body);

        return calculate.toView();
    }
}
