package ru.yandex.qe.dispenser.ws;

import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Inject;

import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Component;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeRequest;
import ru.yandex.qe.dispenser.api.v1.request.quota.Body;
import ru.yandex.qe.dispenser.api.v1.request.quota.BodyUpdate;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.base_resources.WithBaseResourceChanges;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.exception.SingleMessageException;
import ru.yandex.qe.dispenser.domain.util.ValidationUtils;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.QuotaRequestWorkflowManager;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.ResourceWorkflow;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.CreateRequestContext;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.RequestContext;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.UpdateRequestContext;

@Component
@ParametersAreNonnullByDefault
public class QuotaChangeRequestServiceHelper {

    private final QuotaRequestWorkflowManager quotaRequestWorkflowManager;
    private final CampaignDao campaignDao;

    @Inject
    public QuotaChangeRequestServiceHelper(final QuotaRequestWorkflowManager quotaRequestWorkflowManager,
                                           final CampaignDao campaignDao) {
        this.quotaRequestWorkflowManager = quotaRequestWorkflowManager;
        this.campaignDao = campaignDao;
    }

    public WithBaseResourceChanges<List<QuotaChangeRequest>> createRequests(final Body body, final Person person,
                                                                            @NotNull final Campaign campaign) {
        ValidationUtils.validateQuotaChangeRequestCreateBody(body);
        ServiceEndpointUtils.memoizeServiceForChangeRequest(body);

        final ResourceWorkflow workflow = quotaRequestWorkflowManager.getResourceWorkflow();

        final CreateRequestContext ctx = workflow.buildCreateRequestContext(body, person, campaign);

        return workflow.createRequest(ctx);
    }

    public QuotaChangeRequest setStatus(final QuotaChangeRequest request,
                                        final DiQuotaChangeRequest.Status diStatus, final Person person,
                                        final boolean suppressSummon) {
        ServiceEndpointUtils.memoizeServiceForChangeRequest(request);

        final QuotaChangeRequest.Status targetStatus = QuotaChangeRequest.Status.of(diStatus);
        if (request.getType() != QuotaChangeRequest.Type.RESOURCE_PREORDER) {
            throw new IllegalArgumentException("Unsupported request type");
        }
        final ResourceWorkflow workflow = quotaRequestWorkflowManager.getResourceWorkflow();

        return workflow.setStatus(request, targetStatus, new RequestContext(person, request), suppressSummon);
    }

    public WithBaseResourceChanges<QuotaChangeRequest> updateRequest(final QuotaChangeRequest request,
                                                                     final BodyUpdate body, final Person person,
                                                                     final boolean suppressSummon) {
        ValidationUtils.validateQuotaChangeRequestUpdateBody(body);
        ServiceEndpointUtils.memoizeServiceForChangeRequest(request);
        if (request.getType() != QuotaChangeRequest.Type.RESOURCE_PREORDER) {
            throw new IllegalArgumentException("Unsupported request type");
        }
        final ResourceWorkflow workflow = quotaRequestWorkflowManager.getResourceWorkflow();

        final UpdateRequestContext context = workflow.buildUpdateRequestContext(request, body, person);

        return workflow.updateRequest(request, context, suppressSummon);
    }

    @NotNull
    public Campaign tryLoadCampaignWithDefault(@Nullable Long campaignId) {
        if (campaignId == null) {
            // TODO Remove default later, make campaignId required field
            return campaignDao.getLastActiveDraft().orElseThrow(() -> SingleMessageException.illegalArgument("no.active.campaign"));
        } else {
            return campaignDao.readOptional(campaignId).orElseThrow(() -> SingleMessageException.illegalArgument("campaign.with.id.not.found", campaignId));
        }
    }
}
