package ru.yandex.qe.dispenser.ws;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collector;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.BeanParam;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Maps;
import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Value;
import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeStatistic;
import ru.yandex.qe.dispenser.api.v1.DiResourceType;
import ru.yandex.qe.dispenser.domain.CampaignOwningCost;
import ru.yandex.qe.dispenser.domain.MessageHelper;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignOwningCostCache;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestReader;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.domain.util.MathUtils;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.param.QuotaChangeRequestFilterParam;
import ru.yandex.qe.dispenser.ws.param.QuotaChangeRequestFilterParamConverter;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula.ProviderOwningCostFormula;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.ResourceWorkflow;

import static ru.yandex.qe.dispenser.domain.util.MoreFunctions.combiner;
import static ru.yandex.qe.dispenser.ws.quota.request.owning_cost.campaignOwningCost.CampaignOwningCostRefreshTransactionWrapper.VALID_STATUSES;
import static ru.yandex.qe.dispenser.ws.quota.request.owning_cost.formula.ProviderOwningCostFormula.percentageOfCampaignOwningCostOutputForStatisticString;


@ParametersAreNonnullByDefault
@Path("/v1/quota-requests/statistics")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("quota-requests-statistics")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class QuotaChangeRequestStatisticService {

    private static final EnumSet<QuotaChangeRequest.Status> STATUSES_IN_STAT = EnumSet.of(
            QuotaChangeRequest.Status.NEW,
            QuotaChangeRequest.Status.CANCELLED,
            QuotaChangeRequest.Status.REJECTED,
            QuotaChangeRequest.Status.APPROVED,
            QuotaChangeRequest.Status.NEED_INFO,
            QuotaChangeRequest.Status.CONFIRMED,
            QuotaChangeRequest.Status.READY_FOR_REVIEW,
            QuotaChangeRequest.Status.COMPLETED
    );

    private static final Set<String> SUPPORTED_SEGMENTATIONS = ImmutableSet.of(
            "dbaas_db",
            "yt_cluster",
            "logbroker",
            "yp_segment",
            "qloud_segment",
            "sandbox_type",
            "distbuild_segment"
    );

    public enum AmountType {
        ORDERED(QuotaChangeRequest.Change::getAmount),
        READY(QuotaChangeRequest.Change::getAmountReady),
        ALLOCATED(QuotaChangeRequest.Change::getAmountAllocated),
        NOT_READY(c -> c.getAmount() < c.getAmountReady() ? 0 : c.getAmount() - c.getAmountReady()),
        ORDERED_NOT_ALLOCATED(c -> c.getAmount() < c.getAmountAllocated() ? 0 : c.getAmount() - c.getAmountAllocated()),
        READY_NOT_ALLOCATED(c -> c.getAmountReady() < c.getAmountAllocated() ? 0 : c.getAmountReady() - c.getAmountAllocated()),
        ;

        private final Function<QuotaChangeRequest.Change, Long> extractor;

        AmountType(final Function<QuotaChangeRequest.Change, Long> extractor) {
            this.extractor = extractor;
        }

        public Long get(final QuotaChangeRequest.Change change) {
            return extractor.apply(change);
        }
    }

    private static final String EMPTY_TITLE = "";
    private static final String TITLE = "quota.request.statistic.%s.title";
    private static final String SEGMENT_TITLE = "quota.request.statistic.%s.segment.title";
    private static final String NONE_TITLE = "quota.request.statistic.%s.none.title";

    private final QuotaChangeRequestDao quotaChangeRequestDao;

    private final MessageHelper messageHelper;

    private final String locationSegmentationKey;

    private final String invisibleSeparator;

    private final QuotaChangeRequestFilterParamConverter filterParamConverter;

    private final HierarchySupplier hierarchySupplier;

    private final CampaignOwningCostCache campaignOwningCostCache;

    public QuotaChangeRequestStatisticService(final QuotaChangeRequestDao quotaChangeRequestDao,
                                              final MessageHelper messageHelper,
                                              @Value("${dispenser.location.segmentation.key}") final String locationSegmentationKey,
                                              @Value("${dispenser.invisible_separator}") final String invisibleSeparator,
                                              final QuotaChangeRequestFilterParamConverter filterParamConverter,
                                              final HierarchySupplier hierarchySupplier,
                                              final CampaignOwningCostCache campaignOwningCostCache) {
        this.quotaChangeRequestDao = quotaChangeRequestDao;
        this.messageHelper = messageHelper;
        this.locationSegmentationKey = locationSegmentationKey;
        this.invisibleSeparator = invisibleSeparator;
        this.filterParamConverter = filterParamConverter;
        this.hierarchySupplier = hierarchySupplier;
        this.campaignOwningCostCache = campaignOwningCostCache;
    }

    @NotNull
    @GET
    @Access
    public DiQuotaChangeStatistic getRequestsStat(@BeanParam final QuotaChangeRequestFilterParam filterParams,
                                                  @QueryParam("amountType") @DefaultValue("ORDERED") final AmountType amountType) {
        final Person performer = Session.WHOAMI.get();
        final Optional<QuotaChangeRequestReader.QuotaChangeRequestFilter> filter = filterParamConverter.fromParam(filterParams);
        final List<QuotaChangeRequest> requests = filter.map(quotaChangeRequestDao::readRequestsWithFilteredChanges).orElse(Collections.emptyList());

        final RequestCollector requestCollector = new RequestCollector(locationSegmentationKey, messageHelper,
                invisibleSeparator, amountType, hierarchySupplier, campaignOwningCostCache);

        return new DiQuotaChangeStatistic(requestCollector.toGroups(requests, filter, performer));
    }

    private static class RequestCollector {

        private final String locationSegmentationKey;
        private final MessageHelper messageHelper;
        private final String invisibleSeparator;
        private final AmountType amountType;
        private final HierarchySupplier hierarchySupplier;
        private final CampaignOwningCostCache campaignOwningCostCache;

        private RequestCollector(final String locationSegmentationKey,
                                 final MessageHelper messageHelper,
                                 final String invisibleSeparator,
                                 final AmountType amountType,
                                 final HierarchySupplier hierarchySupplier,
                                 final CampaignOwningCostCache campaignOwningCostCache
        ) {
            this.locationSegmentationKey = locationSegmentationKey;
            this.messageHelper = messageHelper;
            this.invisibleSeparator = invisibleSeparator;
            this.amountType = amountType;
            this.hierarchySupplier = hierarchySupplier;
            this.campaignOwningCostCache = campaignOwningCostCache;
        }

        @NotNull
        private HashMap<StatisticKey, StatisticValue> getStatisticValueByKey(final List<QuotaChangeRequest> requests,
                                                                             final Map<QuotaChangeRequest.Status, Integer> requestCountByStatus,
                                                                             final Set<Segmentation> segmentations) {
            return requests.stream()
                    .peek(request -> {
                        MathUtils.increment(requestCountByStatus, request.getStatus(), 1);
                        segmentations.addAll(request.getChanges().stream()
                                .flatMap(e -> e.getSegments().stream())
                                .map(Segment::getSegmentation)
                                .collect(Collectors.toSet()));
                    })
                    .collect(statisticCollector());
        }

        @NotNull
        private Collector<QuotaChangeRequest, HashMap<StatisticKey, StatisticValue>, HashMap<StatisticKey, StatisticValue>> statisticCollector() {
            return Collector.of(HashMap::new, (map, request) -> request.getChanges().forEach(statisticMapChangeConsumer(map)), combiner());
        }

        @NotNull
        private Consumer<QuotaChangeRequest.Change> statisticMapChangeConsumer(final Map<StatisticKey, StatisticValue> map) {
            return change -> {
                final StatisticKey statisticKey = new StatisticKey(change.getResource().getService(), change.getResource(), change.getSegments());

                if (!map.containsKey(statisticKey)) {
                    map.put(statisticKey, new StatisticValue());
                }

                map.get(statisticKey).add(change, amountType);
            };
        }

        private boolean isLocationSegmentation(final Segmentation segment) {
            return segment.getKey().getPublicKey().equals(locationSegmentationKey);
        }

        @NotNull
        private List<DiQuotaChangeStatistic.Group> toGroups(
                final List<QuotaChangeRequest> requests,
                final Optional<QuotaChangeRequestReader.QuotaChangeRequestFilter> filterParams,
                final Person performer) {
            final Map<QuotaChangeRequest.Status, Integer> requestCountByStatus = Maps.newHashMap();
            final Set<Segmentation> segmentations = new HashSet<>();
            final Map<StatisticKey, StatisticValue> statisticsValueByKey = getStatisticValueByKey(requests, requestCountByStatus, segmentations);
            final Set<Map.Entry<StatisticKey, StatisticValue>> statistics = statisticsValueByKey.entrySet();

            final Map<String, String> summary = toSummary(filterParams, requestCountByStatus);
            final ImmutableMap<String, DiQuotaChangeStatistic.CountWithUnit> summaryWithUnits = toSummaryWithUnits();

            final List<DiQuotaChangeStatistic.Group> groups = new ArrayList<>();

            boolean hasCostPermission = ResourceWorkflow.canUserViewMoney(performer);
            String totalOwningCost = hasCostPermission ? getTotalOwningCost(requests) : null;
            Map<String, String> summaryOwningCosts = hasCostPermission
                    ? getOwningCostByStatus(requests, filterParams.orElse(null)) : null;
            Map<Long, String> summaryPercentageByCampaignOwningCosts = getSummaryPercentageByCampaignOwningCosts(requests);

            groups.add(new DiQuotaChangeStatistic.Group(messageHelper.format("quota.request.statistic.total.title"),
                    Collections.singletonList(toResourcesByServiceSections(statistics, EMPTY_TITLE, hasCostPermission)),
                    summary, summaryWithUnits, totalOwningCost, summaryOwningCosts, summaryPercentageByCampaignOwningCosts));

            groups.add(new DiQuotaChangeStatistic.Group(messageHelper.format("quota.request.statistic.campaign.title"),
                    toCampaignSections(requests, hasCostPermission), summary, summaryWithUnits, totalOwningCost,
                    summaryOwningCosts, summaryPercentageByCampaignOwningCosts));

            groups.add(new DiQuotaChangeStatistic.Group(messageHelper.format("quota.request.statistic.delivery_date.title"),
                    toDeliveryDateSections(requests, hasCostPermission), summary, summaryWithUnits, totalOwningCost,
                    summaryOwningCosts, summaryPercentageByCampaignOwningCosts));

            if (segmentations.stream().anyMatch(this::isLocationSegmentation)) {
                groups.add(new DiQuotaChangeStatistic.Group(messageHelper.format("quota.request.statistic.dc.title"),
                        toDcLocations(statistics, hasCostPermission), summary, summaryWithUnits, totalOwningCost,
                        summaryOwningCosts, summaryPercentageByCampaignOwningCosts));
            }

            segmentations.stream()
                    .filter(segmentation -> !isLocationSegmentation(segmentation))
                    .sorted(Comparator.comparing(Segmentation::getKey))
                    .forEach(segmentation -> groups.add(new DiQuotaChangeStatistic.Group(getTittleBySegmentation(segmentation),
                            toLocations(statistics, segmentation, hasCostPermission), summary, summaryWithUnits,
                            totalOwningCost, summaryOwningCosts, summaryPercentageByCampaignOwningCosts)));

            return groups;
        }

        private String getTotalOwningCost(List<QuotaChangeRequest> requests) {
            BigDecimal result = BigDecimal.ZERO;
            for (QuotaChangeRequest request : requests) {
                for (QuotaChangeRequest.Change change : request.getChanges()) {
                    if (change.getOwningCost() != null) {
                        BigDecimal rounded = ProviderOwningCostFormula.owningCostToOutputFormat(change.getOwningCost());
                        if (rounded.compareTo(BigDecimal.ZERO) > 0) {
                            result = result.add(rounded);
                        }
                    }
                }
            }
            return ProviderOwningCostFormula.owningCostToOutputString(result);
        }

        private Map<String, String> getOwningCostByStatus(
                List<QuotaChangeRequest> requests,
                @Nullable QuotaChangeRequestReader.QuotaChangeRequestFilter filterParams) {
            Set<QuotaChangeRequest.Status> statusesInFilter = Optional.ofNullable(filterParams)
                    .map(QuotaChangeRequestReader.QuotaChangeRequestFilter::getStatus).orElse(Set.of());
            Set<QuotaChangeRequest.Status> statuses = statusesInFilter.isEmpty() ? STATUSES_IN_STAT : statusesInFilter;
            Map<QuotaChangeRequest.Status, BigDecimal> totalByStatus = new HashMap<>();
            for (QuotaChangeRequest request : requests) {
                for (QuotaChangeRequest.Change change : request.getChanges()) {
                    if (change.getOwningCost() != null) {
                        BigDecimal rounded = ProviderOwningCostFormula.owningCostToOutputFormat(change.getOwningCost());
                        if (rounded.compareTo(BigDecimal.ZERO) > 0) {
                            totalByStatus.put(request.getStatus(), totalByStatus
                                    .getOrDefault(request.getStatus(), BigDecimal.ZERO).add(rounded));
                        }
                    }
                }
            }
            Map<String, String> result = new HashMap<>();
            for (QuotaChangeRequest.Status status : statuses) {
                result.put(messageHelper.format("quota.request.statistic.summary." + status.name() + ".title"),
                        ProviderOwningCostFormula.owningCostToOutputString(totalByStatus.getOrDefault(status,
                                BigDecimal.ZERO)));
            }
            return result;
        }

        @NotNull
        private Map<Long, String> getSummaryPercentageByCampaignOwningCosts(List<QuotaChangeRequest> requests) {
            if (requests.isEmpty()) {
                return Map.of();
            }

            Map<Long, BigInteger> totalByCampaign = new HashMap<>();
            requests.stream()
                    .filter(r -> r.getCampaignId() != null && VALID_STATUSES.contains(r.getStatus()))
                    .forEach(r -> totalByCampaign.compute(r.getCampaignId(),
                            (k,v) -> v == null ? BigInteger.valueOf(r.getRequestOwningCost()) : v.add(BigInteger.valueOf(r.getRequestOwningCost()))));

            Map<Long, BigInteger> owningCostByCampaignId = campaignOwningCostCache.getAll().stream()
                    .collect(Collectors.toMap(CampaignOwningCost::getCampaignId, CampaignOwningCost::getOwningCost));

            return totalByCampaign.entrySet().stream()
                    .filter(e -> owningCostByCampaignId.containsKey(e.getKey()))
                    .collect(Collectors.toMap(Map.Entry::getKey,
                            v -> percentageOfCampaignOwningCostOutputForStatisticString(v.getValue(), owningCostByCampaignId.get(v.getKey()))));
        }

        private String getTittleBySegmentation(final Segmentation segmentation) {
            final String publicKey = segmentation.getKey().getPublicKey();
            if (SUPPORTED_SEGMENTATIONS.contains(publicKey)) {
                return messageHelper.format(String.format(TITLE, publicKey));
            }

            return segmentation.getName();
        }

        private String getNoneTittleBySegmentation(final Segmentation segmentation) {
            final String publicKey = segmentation.getKey().getPublicKey();
            if (SUPPORTED_SEGMENTATIONS.contains(publicKey)) {
                return messageHelper.format(String.format(NONE_TITLE, publicKey));
            }

            return "Без " + segmentation.getName();
        }

        private String getSegmentTittleBySegment(final Segment segment) {
            final String publicKey = segment.getSegmentation().getKey().getPublicKey();
            final String name = segment.getName();

            if (SUPPORTED_SEGMENTATIONS.contains(publicKey)) {
                return messageHelper.format(String.format(SEGMENT_TITLE, publicKey), name, invisibleSeparator);
            }

            return name;
        }

        private ImmutableMap<String, String> toSummary(final Optional<QuotaChangeRequestReader.QuotaChangeRequestFilter> filterParams,
                                                       final Map<QuotaChangeRequest.Status, Integer> requestCountByStatus) {
            final ImmutableMap.Builder<String, String> summaryBuilder = ImmutableMap.builder();

            final Set<QuotaChangeRequest.Status> statusesInFilter = filterParams.map(QuotaChangeRequestReader.QuotaChangeRequestFilter::getStatus).orElse(Collections.emptySet());
            final Set<QuotaChangeRequest.Status> statuses = statusesInFilter.isEmpty() ? STATUSES_IN_STAT : statusesInFilter;
            for (final QuotaChangeRequest.Status status : statuses) {
                summaryBuilder.put(messageHelper.format(
                        "quota.request.statistic.summary." + status.name()
                                + ".title"), String.valueOf(requestCountByStatus.getOrDefault(status, 0)));
            }

            return summaryBuilder.build();
        }

        private ImmutableMap<String, DiQuotaChangeStatistic.CountWithUnit> toSummaryWithUnits() {
            final ImmutableMap.Builder<String, DiQuotaChangeStatistic.CountWithUnit> summaryWithCountBuilder = ImmutableMap.builder();

            final Long totalPrice = null;

            //noinspection ConstantConditions temporary money hiding: DISPENSER-2214
            if (totalPrice != null) {
                summaryWithCountBuilder.put(messageHelper.format("quota.request.statistic.summary.sum.title"),
                        new DiQuotaChangeStatistic.CountWithUnit(totalPrice, "usd"));
            }

            return summaryWithCountBuilder.build();
        }

        private List<DiQuotaChangeStatistic.Section> toDeliveryDateSections(final List<QuotaChangeRequest> requests,
                                                                            final boolean hasCostPermission) {
            final Map<QuotaChangeRequest.BigOrder, List<QuotaChangeRequest.Change>> changesByBigOrder = requests.stream()
                    .map(QuotaChangeRequest::getChanges)
                    .flatMap(Collection::stream)
                    .filter(e -> e.getKey().getBigOrder() != null)
                    .collect(Collectors.groupingBy(e -> e.getKey().getBigOrder()));

            final Map<QuotaChangeRequest.BigOrder, HashMap<StatisticKey, StatisticValue>> statisticMapByBigOrder = changesByBigOrder.entrySet().stream()
                    .collect(Collectors.toMap(Map.Entry::getKey,
                            value -> value.getValue().stream()
                                    .collect(Collector.of(HashMap::new, (map, change) -> {
                                        statisticMapChangeConsumer(map).accept(change);
                                    }, combiner()))));

            final Map<QuotaChangeRequest.BigOrder, DiQuotaChangeStatistic.Section> sectionsByBigOrder = statisticMapByBigOrder.entrySet().stream()
                    .collect(Collectors.toMap(Map.Entry::getKey,
                            value -> toResourcesByServiceSections(value.getValue().entrySet(),
                                    value.getKey().getDate().format(DateTimeFormatter.ISO_LOCAL_DATE), hasCostPermission)));

            return new ArrayList<>(sectionsByBigOrder.values());
        }

        private List<DiQuotaChangeStatistic.Section> toCampaignSections(final List<QuotaChangeRequest> requests,
                                                                        final boolean hasCostPermission) {
            final Map<QuotaChangeRequest.Campaign, List<QuotaChangeRequest>> requestByCampaign = requests
                    .stream()
                    .filter(request -> request.getCampaign() != null)
                    .collect(Collectors.groupingBy(QuotaChangeRequest::getCampaign));

            final Map<QuotaChangeRequest.Campaign, HashMap<StatisticKey, StatisticValue>> statisticMapByCampaign = requestByCampaign.entrySet().stream()
                    .collect(Collectors.toMap(Map.Entry::getKey, value -> value.getValue().stream().collect(statisticCollector())));

            final Map<QuotaChangeRequest.Campaign, DiQuotaChangeStatistic.Section> sectionsByCampaign = statisticMapByCampaign.entrySet().stream()
                    .collect(Collectors.toMap(Map.Entry::getKey,
                            value -> toResourcesByServiceSections(value.getValue().entrySet(),
                                    value.getKey().getName(), hasCostPermission)));

            return new ArrayList<>(sectionsByCampaign.values());
        }

        private DiQuotaChangeStatistic.Section toResourcesByServiceSections(
                final Set<Map.Entry<StatisticKey, StatisticValue>> statistics,
                final String title,
                final boolean hasCostPermission) {
            final Map<Service, List<Map.Entry<StatisticKey, StatisticValue>>> statByService = statistics
                    .stream()
                    .collect(Collectors.groupingBy(stat -> stat.getKey().getService()));

            final List<DiQuotaChangeStatistic.Item> items = statByService.entrySet().stream()
                    .map(entries -> {
                        final List<Map.Entry<StatisticKey, StatisticValue>> stats = new ArrayList<>(entries.getValue());
                        final List<DiQuotaChangeStatistic.Resource> resources = getResources(stats, hasCostPermission);
                        return new DiQuotaChangeStatistic.Item(entries.getKey().getName(), resources);
                    })
                    .collect(Collectors.toList());

            return new DiQuotaChangeStatistic.Section(title, items);
        }

        private List<DiQuotaChangeStatistic.Section> toDcLocations(
                final Collection<Map.Entry<StatisticKey, StatisticValue>> statistics,
                final boolean hasCostPermission) {
            final Map<Boolean, List<Map.Entry<StatisticKey, StatisticValue>>> statisticsByLocationPresents = statistics.stream()
                    .collect(Collectors.partitioningBy(stat -> stat.getKey().getLocation().stream()
                            .map(Segment::getSegmentation)
                            .anyMatch(this::isLocationSegmentation)));

            final List<DiQuotaChangeStatistic.Section> byLocations = statisticsByLocationPresents.get(true).stream()
                    .collect(Collectors.groupingBy(stat -> stat.getKey().getLocation().stream()
                            .filter(segment -> isLocationSegmentation(segment.getSegmentation()))
                            .findFirst()
                            .get()))
                    .entrySet().stream()
                    .map(entry -> {
                        return toResourcesByServiceSections(new HashSet<>(entry.getValue()), entry.getKey().getName(),
                                hasCostPermission);
                    })
                    .collect(Collectors.toList());

            final List<Map.Entry<StatisticKey, StatisticValue>> entries = statisticsByLocationPresents.get(false);
            if (!entries.isEmpty()) {
                byLocations.add(toResourcesByServiceSections(new HashSet<>(entries),
                        messageHelper.format("quota.request.statistic.location.none.title"), hasCostPermission));
            }

            return byLocations;
        }

        private List<DiQuotaChangeStatistic.Section> toLocations(final Set<Map.Entry<StatisticKey, StatisticValue>> statistics,
                                                                 final Segmentation segmentation,
                                                                 final boolean hasCostPermission) {
            final Map<Boolean, List<Map.Entry<StatisticKey, StatisticValue>>> statisticsByLocationPresents = statistics.stream()
                    .collect(Collectors.partitioningBy(stat -> stat.getKey().getLocation().stream()
                            .map(Segment::getSegmentation)
                            .anyMatch(segmentation::equals)));

            final List<DiQuotaChangeStatistic.Section> sections = statisticsByLocationPresents.get(true).stream()
                    .collect(Collectors.groupingBy(stat -> stat.getKey().getLocation().stream().filter(segment -> segmentation.equals(segment.getSegmentation())).findFirst().get()))
                    .entrySet().stream()
                    .map(entry -> {
                        final List<Map.Entry<StatisticKey, StatisticValue>> stats = entry.getValue();
                        final List<DiQuotaChangeStatistic.Resource> resources = getResources(stats, hasCostPermission);
                        final String title = getSegmentTittleBySegment(entry.getKey());
                        final DiQuotaChangeStatistic.Item item = new DiQuotaChangeStatistic.Item(EMPTY_TITLE, resources);
                        return new DiQuotaChangeStatistic.Section(title, Collections.singletonList(item));
                    })
                    .collect(Collectors.toList());

            final List<Map.Entry<StatisticKey, StatisticValue>> entries = statisticsByLocationPresents.get(false);
            if (!entries.isEmpty()) {
                sections.add(toResourcesByServiceSections(new HashSet<>(entries),
                        getNoneTittleBySegmentation(segmentation), hasCostPermission));
            }

            return sections;
        }

        public StatisticResourceKey getStatisticResourceKey(final Resource resource) {
            return new StatisticResourceKey(resource.getKey().getPublicKey(), resource.getName(), resource.getService().getKey(), resource.getType());
        }

        @NotNull
        private List<DiQuotaChangeStatistic.Resource> getResources(
                final List<Map.Entry<StatisticKey, StatisticValue>> entries,
                final boolean hasCostPermission) {
            final Map<StatisticResourceKey, List<Map.Entry<StatisticKey, StatisticValue>>> byKeys = entries.stream()
                    .collect(Collectors.groupingBy(entry -> getStatisticResourceKey(entry.getKey().getResource())));

            final Map<StatisticResourceKey, StatisticValue> resourceValues = byKeys.entrySet().stream()
                    .collect(Collectors.toMap(Map.Entry::getKey, entry -> {
                        return entry.getValue().stream()
                                .reduce(new StatisticValue(), (total, statistic) -> total.add(statistic.getValue()), StatisticValue::add);
                    }));

            return resourceValues.entrySet().stream()
                    .map(s -> toResource(s, hasCostPermission))
                    .collect(Collectors.toList());
        }


        @NotNull
        private DiQuotaChangeStatistic.Resource toResource(
                final Map.Entry<StatisticResourceKey, StatisticValue> statistic,
                final boolean hasCostPermission) {
            final BigInteger amount = statistic.getValue().getAmount();
            final StatisticResourceKey key = statistic.getKey();

            final Optional<DiAmount> normalizedValue = key.getType().getBaseUnit().normalize(amount, false);
            String owningCost = hasCostPermission
                    ? ProviderOwningCostFormula.owningCostToOutputString(statistic.getValue().getOwningCost()) : null;

            if (normalizedValue.isPresent()) {
                return new DiQuotaChangeStatistic.Resource(new DiQuotaChangeStatistic.Resource.ResourceKey(key.getKey(),
                        key.getServiceKey()), key.getName(), normalizedValue.get(), statistic.getValue().getCost(),
                        owningCost);
            } else {
                throw new IllegalArgumentException("Too big amount value: " + amount);
            }
        }

    }

    private static class StatisticResourceKey {
        private final String key;
        private final String name;
        private final String serviceKey;
        private final DiResourceType type;

        private StatisticResourceKey(final String key, final String name, final String serviceKey, final DiResourceType type) {
            this.key = key;
            this.name = name;
            this.serviceKey = serviceKey;
            this.type = type;
        }

        public String getKey() {
            return key;
        }

        public String getName() {
            return name;
        }

        public String getServiceKey() {
            return serviceKey;
        }

        public DiResourceType getType() {
            return type;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final StatisticResourceKey that = (StatisticResourceKey) o;
            return key.equals(that.key) &&
                    name.equals(that.name) &&
                    serviceKey.equals(that.serviceKey) &&
                    type == that.type;
        }

        @Override
        public int hashCode() {
            return Objects.hash(key, name, serviceKey, type);
        }
    }


    private static class StatisticKey {
        @Nonnull
        private final Service service;
        @Nonnull
        private final Resource resource;
        @Nonnull
        private final Set<Segment> location;

        private StatisticKey(final Service service, final Resource resource, final Set<Segment> location) {
            this.service = service;
            this.resource = resource;
            this.location = location;
        }

        @Nonnull
        public Service getService() {
            return service;
        }

        @Nonnull
        public Resource getResource() {
            return resource;
        }

        @Nonnull
        public Set<Segment> getLocation() {
            return location;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final StatisticKey that = (StatisticKey) o;
            return service.equals(that.service) &&
                    resource.equals(that.resource) &&
                    Objects.equals(location, that.location);
        }

        @Override
        public int hashCode() {
            return Objects.hash(service, resource, location);
        }
    }

    private static class StatisticValue {
        private BigInteger amount = BigInteger.ZERO;
        private long cost;
        private BigDecimal owningCost = BigDecimal.ZERO;

        public void add(final QuotaChangeRequest.Change change, final AmountType amountType) {
            amount = amount.add(BigInteger.valueOf(amountType.get(change)));
            //noinspection ConstantConditions
            cost += 0;
            if (change.getOwningCost() != null) {
                BigDecimal rounded = ProviderOwningCostFormula.owningCostToOutputFormat(change.getOwningCost());
                if (rounded.compareTo(BigDecimal.ZERO) > 0) {
                    owningCost = owningCost.add(rounded);
                }
            }
        }

        public StatisticValue add(final StatisticValue value) {
            amount = amount.add(value.getAmount());
            cost += value.getCost();
            owningCost = owningCost.add(value.getOwningCost());
            return this;
        }

        public BigInteger getAmount() {
            return amount;
        }

        public long getCost() {
            return cost;
        }

        public BigDecimal getOwningCost() {
            return owningCost;
        }

    }
}
