package ru.yandex.qe.dispenser.ws;

import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.UriInfo;

import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import ru.yandex.qe.dispenser.api.v1.DiOrder;
import ru.yandex.qe.dispenser.api.v1.DiQuotaHistoryEvent;
import ru.yandex.qe.dispenser.api.v1.response.DiListRelativePageResponse;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaHistoryEvent;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.history.HistoryFilter;
import ru.yandex.qe.dispenser.domain.dao.history.quota.QuotaHistoryDao;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaReader;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.util.RelativePage;
import ru.yandex.qe.dispenser.domain.util.RelativePageInfo;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.param.EntitySpecFilterParam;
import ru.yandex.qe.dispenser.ws.param.HistoryParams;
import ru.yandex.qe.dispenser.ws.param.QuotaGetParams;
import ru.yandex.qe.dispenser.ws.param.QuotaKeyParam;
import ru.yandex.qe.dispenser.ws.param.RelativePageParam;
import ru.yandex.qe.dispenser.ws.param.ResourceFilterParam;

import static ru.yandex.qe.dispenser.ws.Access.PROJECT_KEY;
import static ru.yandex.qe.dispenser.ws.Access.SERVICE_KEY;
import static ru.yandex.qe.dispenser.ws.QuotaSpecService.QUOTA_SPEC_KEY;
import static ru.yandex.qe.dispenser.ws.ServiceResourceService.RESOURCE_KEY;
import static ru.yandex.qe.dispenser.ws.param.QuotaCheckParam.SEGMENTS;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.ENTITY_SPEC_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.LEAF_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.MEMBER_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.PROJECT_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.RESOURCE_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.SEGMENT_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.SERVICE_PARAM;
import static ru.yandex.qe.dispenser.ws.param.RelativePageParam.DEFAULT_ORDER;
import static ru.yandex.qe.dispenser.ws.param.RelativePageParam.PAGE_ORDER;

@Controller
@Path("/v1/history/quotas")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class QuotaHistoryService extends ServiceBase {
    private static final String EVENT_ID = "event_id";

    @Autowired
    private QuotaHistoryDao quotaHistoryDao;

    @GET
    @Access
    public DiListRelativePageResponse<DiQuotaHistoryEvent> history(@QueryParam(RESOURCE_PARAM) final List<ResourceFilterParam> resourceParams,
                                                                   @QueryParam(ENTITY_SPEC_PARAM) final List<EntitySpecFilterParam> entitySpecParams,
                                                                   @QueryParam(SEGMENT_PARAM) final Set<String> segmentKeys,
                                                                   @QueryParam(SERVICE_PARAM) final Set<Service> services,
                                                                   @QueryParam(PROJECT_PARAM) final Set<Project> projects,
                                                                   @QueryParam(MEMBER_PARAM) final Set<Person> members,
                                                                   @QueryParam(LEAF_PARAM) @DefaultValue("false") final boolean leafProjects,
                                                                   @QueryParam(HistoryParams.FROM) final String from,
                                                                   @QueryParam(HistoryParams.TO) final String to,
                                                                   @QueryParam(HistoryParams.PERFORMER) final Set<Person> performers,
                                                                   @QueryParam(HistoryParams.TVM_ID) final Set<Long> tvmIds,
                                                                   @QueryParam(RelativePageParam.PAGE_SIZE)
                                                                       @DefaultValue(RelativePageParam.DEFAULT_PAGE_SIZE) final Long pageSize,
                                                                   @QueryParam(RelativePageParam.FROM_ID) final Long fromId,
                                                                   @Context final UriInfo uriInfo,
                                                                   @QueryParam(PAGE_ORDER) @DefaultValue(DEFAULT_ORDER) final DiOrder order) {
        final QuotaGetParams quotaGetParams = new QuotaGetParams(resourceParams, entitySpecParams, segmentKeys, services, projects, members, leafProjects, order);
        final HistoryParams historyParams = new HistoryParams(from, to, performers, tvmIds);
        final RelativePageParam pageParam = new RelativePageParam(pageSize, fromId, uriInfo, order);

        final Optional<QuotaReader.QuotaFilterParams> quotaFilterParams = quotaGetParams.toQuotaFilterParams();
        if (!quotaFilterParams.isPresent()) {
            throw new IllegalArgumentException("Incorrect params!");
        }
        final QuotaReader.QuotaFilterParams filter = quotaFilterParams.get();
        final HistoryFilter historyFilter = historyParams.toHistoryFilter();
        final RelativePageInfo pageInfo = getPageInfo(pageParam);

        final RelativePage<QuotaHistoryEvent> page = quotaHistoryDao.read(filter, historyFilter, pageInfo, order);
        return toResponsePage(pageParam, page, QuotaHistoryEvent::toView);
    }

    @GET
    @Path("/{" + PROJECT_KEY + "}/{" + SERVICE_KEY + "}/{" + RESOURCE_KEY + "}/{" + QUOTA_SPEC_KEY + "}")
    @Access
    public DiListRelativePageResponse<DiQuotaHistoryEvent> historyByQuota(@PathParam(PROJECT_KEY) @NotNull final Project project,
                                                                          @PathParam(SERVICE_KEY) @NotNull final Service service,
                                                                          @PathParam(RESOURCE_KEY) @NotNull final String resourceKey,
                                                                          @PathParam(QUOTA_SPEC_KEY) @NotNull final String quotaSpecKey,
                                                                          @QueryParam(SEGMENTS) final Set<String> segments,
                                                                          @QueryParam(HistoryParams.FROM) final String from,
                                                                          @QueryParam(HistoryParams.TO) final String to,
                                                                          @QueryParam(HistoryParams.PERFORMER) final Set<Person> performers,
                                                                          @QueryParam(HistoryParams.TVM_ID) final Set<Long> tvmIds,
                                                                          @QueryParam(RelativePageParam.PAGE_SIZE)
                                                                              @DefaultValue(RelativePageParam.DEFAULT_PAGE_SIZE) final Long pageSize,
                                                                          @QueryParam(RelativePageParam.FROM_ID) final Long fromId,
                                                                          @Context final UriInfo uriInfo,
                                                                          @QueryParam(PAGE_ORDER) @DefaultValue(DEFAULT_ORDER) final DiOrder order) {
        final RelativePageParam pageParam = new RelativePageParam(pageSize, fromId, uriInfo, order);
        final QuotaKeyParam quotaKeyParam = new QuotaKeyParam(project, service, resourceKey, quotaSpecKey);
        final HistoryParams historyParams = new HistoryParams(from, to, performers, tvmIds);

        final Set<@NotNull Segment> loadedSegments = segments.stream()
                .map(Hierarchy.get().getSegmentReader()::read)
                .collect(Collectors.toSet());
        final Quota.Key quotaKey = quotaKeyParam.get().withSegments(loadedSegments);

        final RelativePage<QuotaHistoryEvent> page = quotaHistoryDao.read(quotaKey, historyParams.toHistoryFilter(), getPageInfo(pageParam), order);
        return toResponsePage(pageParam, page, QuotaHistoryEvent::toView);
    }

    @GET
    @Path("/events/{" + EVENT_ID + "}")
    @Access
    public DiQuotaHistoryEvent event(@PathParam(EVENT_ID) final long eventId) {
        final QuotaHistoryEvent event = quotaHistoryDao.read(eventId);
        return event.toView();
    }
}
