package ru.yandex.qe.dispenser.ws;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import javax.ws.rs.Consumes;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;

import ru.yandex.qe.dispenser.api.util.NumberUtils;
import ru.yandex.qe.dispenser.api.v1.DiOrder;
import ru.yandex.qe.dispenser.api.v1.DiQuota;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.api.v1.response.DiQuotaGetResponse;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaView;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaUtils;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.domain.hierarchy.Role;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.param.EntitySpecFilterParam;
import ru.yandex.qe.dispenser.ws.param.QuotaGetParams;
import ru.yandex.qe.dispenser.ws.param.QuotaKeyParam;
import ru.yandex.qe.dispenser.ws.param.ResourceFilterParam;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.PerformerContext;
import ru.yandex.qe.dispenser.ws.quota.update.QuotaUpdateManager;
import ru.yandex.qe.dispenser.ws.reqbody.MaxValueBody;

import static ru.yandex.qe.dispenser.ws.QuotaSpecService.QUOTA_SPEC_KEY;
import static ru.yandex.qe.dispenser.ws.ServiceResourceService.RESOURCE_KEY;
import static ru.yandex.qe.dispenser.ws.ServiceService.SERVICE_KEY;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.ENTITY_SPEC_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.LEAF_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.MEMBER_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.ORDER_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.PROJECT_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.RESOURCE_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.SEGMENT_PARAM;
import static ru.yandex.qe.dispenser.ws.param.QuotaGetParams.SERVICE_PARAM;

@Path("/v1")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("quota-crud")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class QuotaReadUpdateService extends ServiceBase {
    public static final String PATH = "/quotas";

    private static final Logger LOG = LoggerFactory.getLogger(QuotaReadUpdateService.class);

    @Autowired
    private QuotaUpdateManager quotaUpdateManager;

    @GET
    @NotNull
    @Path(PATH)
    public DiQuotaGetResponse read(@QueryParam(RESOURCE_PARAM) final List<ResourceFilterParam> resourceParams,
                                   @QueryParam(ENTITY_SPEC_PARAM) final List<EntitySpecFilterParam> entitySpecParams,
                                   @QueryParam(SEGMENT_PARAM) final Set<String> segmentKeys,
                                   @QueryParam(SERVICE_PARAM) final Set<Service> services,
                                   @QueryParam(PROJECT_PARAM) final Set<Project> projects,
                                   @QueryParam(MEMBER_PARAM) final Set<Person> members,
                                   @QueryParam(LEAF_PARAM) @DefaultValue("false") final boolean leafProjects,
                                   @QueryParam(ORDER_PARAM) final DiOrder order) {
        final QuotaGetParams quotaGetParams = new QuotaGetParams(resourceParams, entitySpecParams, segmentKeys, services, projects, members, leafProjects, order);
        ServiceEndpointUtils.memoizeServiceForQuotas(quotaGetParams);
        LOG.debug("Started processing");
        final List<DiQuota> quotaViews = quotaGetParams.getQuotaStream()
                .map(QuotaUtils::toView)
                .collect(Collectors.toList());
        LOG.debug("Returning response");
        return new DiQuotaGetResponse(quotaViews);
    }

    @POST
    @NotNull
    @Consumes(MediaType.APPLICATION_JSON)
    @Access(role = Role.RESPONSIBLE, parent = true)
    @Path(PATH + "/old/{" + Access.PROJECT_KEY + "}/{" + SERVICE_KEY + "}/{" + RESOURCE_KEY + "}/{" + QUOTA_SPEC_KEY + "}")
    public DiQuota update(@PathParam(Access.PROJECT_KEY) @NotNull final Project project,
                          @PathParam(Access.SERVICE_KEY) @NotNull final Service service,
                          @PathParam(RESOURCE_KEY) @NotNull final String resourceKey,
                          @PathParam(QUOTA_SPEC_KEY) @NotNull final String quotaSpecKey,
                          @RequestBody @NotNull final MaxValueBody maxValue) {
        final QuotaKeyParam quotaKeyParam = new QuotaKeyParam(project, service, resourceKey, quotaSpecKey);
        ServiceEndpointUtils.memoizeService(service);
        final Quota.Key totalQuotaKey = quotaKeyParam.get();
        final Resource resource = totalQuotaKey.getSpec().getResource();
        final Set<Segment> segments = SegmentUtils.getCompleteSegmentSet(resource, maxValue.getSegments());
        final Quota.Key quotaKey = totalQuotaKey.withSegments(segments);

        final Person person = Session.WHOAMI.get();
        ServiceService.checkCanUpdateMax(person, quotaKey);
        ServiceService.checkCanChangeQuotaMaxWithOldStyleMethod(quotaKey);

        final DiUnit unit = maxValue.getUnit();
        final long newMaxInUnit = NumberUtils.requireNotNegative(maxValue.getMaxValue());
        final long newMax = resource.getType().getBaseUnit().convert(newMaxInUnit, unit);
        final long newOwnMax = resource.getType().getBaseUnit().convert(maxValue.getOwnMaxValue(), unit);

        final Quota.ChangeHolder changeHolder = new Quota.ChangeHolder();
        changeHolder.setMax(quotaKey, newMax);
        if (newOwnMax > 0) {
            changeHolder.setOwnMax(quotaKey, newOwnMax);
        }

        final QuotaView updatedQuota = CollectionUtils.first(quotaUpdateManager.updateQuota(changeHolder, new PerformerContext(person), null));

        return QuotaUtils.toView(updatedQuota);
    }
}
