package ru.yandex.qe.dispenser.ws;

import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.concurrent.TimeUnit;

import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaView;
import ru.yandex.qe.dispenser.solomon.Solomon;
import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.encode.spack.StringPoolBuilder;
import ru.yandex.monlib.metrics.encode.spack.compression.EncodeStream;
import ru.yandex.monlib.metrics.encode.spack.format.CompressionAlg;
import ru.yandex.monlib.metrics.encode.spack.format.MetricTypes;
import ru.yandex.monlib.metrics.encode.spack.format.MetricValuesType;
import ru.yandex.monlib.metrics.encode.spack.format.SpackHeader;
import ru.yandex.monlib.metrics.encode.spack.format.TimePrecision;

public final class QuotaSpackEncoder {

    private QuotaSpackEncoder() {
    }

    public static void encode(@NotNull final TimePrecision timePrecision, @NotNull final CompressionAlg compressionAlg,
                              @NotNull final OutputStream out, @NotNull final Collection<QuotaView> quotas) {
        final StringPoolBuilder keysPool = new StringPoolBuilder();
        final StringPoolBuilder valuesPool = new StringPoolBuilder();
        final int quotasToEncodeCount = (int) quotas.stream().filter(q -> q.getProject().isReal()).count();
        final long[] quotaActualValues = new long[quotasToEncodeCount];
        final long[] quotaMaxValues = new long[quotasToEncodeCount];
        final Labels labels = new Labels(quotasToEncodeCount, keysPool, valuesPool);
        int quotaIndex = 0;
        for (final QuotaView q : quotas) {
            if (!q.getProject().isReal()) {
                continue;
            }
            if (q.getResource().getService().getSettings().usesProjectHierarchy()) {
                quotaActualValues[quotaIndex] = q.getTotalActual();
                quotaMaxValues[quotaIndex] = q.getMax();
            } else {
                quotaActualValues[quotaIndex] = q.getOwnActual();
                quotaMaxValues[quotaIndex] = q.getOwnMax();
            }

            labels.fillLabels(q, quotaIndex);
            quotaIndex++;
        }
        keysPool.sortByFrequencies();
        valuesPool.sortByFrequencies();
        try (final EncodeStream encodedOut = EncodeStream.create(compressionAlg, out)) {
            final SpackHeader header = new SpackHeader(
                    timePrecision,
                    compressionAlg,
                    calcPoolSize(keysPool),
                    calcPoolSize(valuesPool),
                    quotasToEncodeCount * 2,
                    quotasToEncodeCount * 2);
            // Header
            encodedOut.writeHeader(header);
            // Keys pool
            keysPool.forEachString(s -> writeString(encodedOut, s));
            // Values pool
            valuesPool.forEachString(s -> writeString(encodedOut, s));
            // Common time
            writeTime(0, encodedOut, timePrecision);
            // Zero common labels
            encodedOut.writeVarint32(0);
            for (int i = 0; i < quotasToEncodeCount; i++) {
                writeQuota(encodedOut, quotaActualValues[i], false, labels, i);
                writeQuota(encodedOut, quotaMaxValues[i], true, labels, i);
            }
        }
    }

    private static void writeQuota(@NotNull final EncodeStream out, final long value, final boolean max, @NotNull final Labels labels,
                                   final int index) {
        // Type
        out.writeByte(MetricTypes.pack(MetricType.DGAUGE, MetricValuesType.ONE_WITHOUT_TS));
        // Flags
        out.writeByte((byte) 0x00);
        // Labels
        labels.writeLabels(out, index, max);
        // Value
        out.writeDouble((double) value);
    }

    private static void writeTime(final long tsMillis, @NotNull final EncodeStream out, @NotNull final TimePrecision timePrecision) {
        if (timePrecision == TimePrecision.SECONDS) {
            out.writeIntLe((int) TimeUnit.MILLISECONDS.toSeconds(tsMillis));
        } else {
            out.writeLongLe(tsMillis);
        }
    }

    private static int calcPoolSize(@NotNull final StringPoolBuilder pool) {
        final int[] size = new int[1];
        pool.forEachString(s -> {
            size[0] += s.getBytes(StandardCharsets.UTF_8).length + 1;
        });
        return size[0];
    }

    private static void writeString(@NotNull final EncodeStream out, @NotNull final String value) {
        final byte[] bytes = value.getBytes(StandardCharsets.UTF_8);
        out.write(bytes, 0, bytes.length);
        out.writeByte((byte) 0);
    }

    private static final class Labels {

        private final StringPoolBuilder valuesPool;
        private final StringPoolBuilder.PooledString[] sensorLabelValues;
        private final StringPoolBuilder.PooledString[] projectKeyLabelValues;
        private final StringPoolBuilder.PooledString[] serviceNameLabelValues;
        private final StringPoolBuilder.PooledString[] typeLabelValues;
        private final StringPoolBuilder.PooledString[] specLabelValues;
        private final StringPoolBuilder.PooledString[] segmentLabelValues;
        private final StringPoolBuilder.PooledString sensorLabelKey;
        private final StringPoolBuilder.PooledString projectKeyLabelKey;
        private final StringPoolBuilder.PooledString serviceNameLabelKey;
        private final StringPoolBuilder.PooledString typeLabelKey;
        private final StringPoolBuilder.PooledString specLabelKey;
        private final StringPoolBuilder.PooledString attributeLabelKey;
        private final StringPoolBuilder.PooledString segmentLabelKey;
        private final StringPoolBuilder.PooledString actualAttributeLabelValue;
        private final StringPoolBuilder.PooledString maxAttributeLabelValue;

        private Labels(final int size, @NotNull final StringPoolBuilder keysPool, @NotNull final StringPoolBuilder valuesPool) {
            this.valuesPool = valuesPool;
            this.sensorLabelValues = new StringPoolBuilder.PooledString[size];
            this.projectKeyLabelValues = new StringPoolBuilder.PooledString[size];
            this.serviceNameLabelValues = new StringPoolBuilder.PooledString[size];
            this.typeLabelValues = new StringPoolBuilder.PooledString[size];
            this.specLabelValues = new StringPoolBuilder.PooledString[size];
            this.segmentLabelValues = new StringPoolBuilder.PooledString[size];
            // Following keys are used twice for each quota (actual and max)
            this.sensorLabelKey = keysPool.putIfAbsent("sensor");
            this.sensorLabelKey.frequency = size * 2;
            this.projectKeyLabelKey = keysPool.putIfAbsent("project-key");
            this.projectKeyLabelKey.frequency = size * 2;
            this.serviceNameLabelKey = keysPool.putIfAbsent("service-name");
            this.serviceNameLabelKey.frequency = size * 2;
            this.typeLabelKey = keysPool.putIfAbsent("type");
            this.typeLabelKey.frequency = size * 2;
            this.specLabelKey = keysPool.putIfAbsent("spec");
            this.specLabelKey.frequency = size * 2;
            this.attributeLabelKey = keysPool.putIfAbsent("attribute");
            this.attributeLabelKey.frequency = size * 2;
            this.segmentLabelKey = keysPool.putIfAbsent("segment");
            this.segmentLabelKey.frequency = size * 2;
            // Following values are used once for each quota
            this.actualAttributeLabelValue = valuesPool.putIfAbsent("actual");
            this.actualAttributeLabelValue.frequency = size;
            this.maxAttributeLabelValue = valuesPool.putIfAbsent("max");
            this.maxAttributeLabelValue.frequency = size;
        }

        private void fillLabels(@NotNull final QuotaView quotaView, final int index) {
            // Each label value is used twice for each quota (actual and max)
            final Quota.Key quotaKey = quotaView.getKey();
            final String sensorName = Solomon.getSensorName(quotaKey).replaceAll("\0", "");
            sensorLabelValues[index] = valuesPool.putIfAbsent(sensorName);
            sensorLabelValues[index] = valuesPool.putIfAbsent(sensorName);
            final String projectKey = quotaKey.getProject().getPublicKey().replaceAll("\0", "");
            projectKeyLabelValues[index] = valuesPool.putIfAbsent(projectKey);
            projectKeyLabelValues[index] = valuesPool.putIfAbsent(projectKey);
            final String serviceName = quotaKey.getSpec().getResource().getService().getName().replaceAll("\0", "");
            serviceNameLabelValues[index] = valuesPool.putIfAbsent(serviceName);
            serviceNameLabelValues[index] = valuesPool.putIfAbsent(serviceName);
            final String type = quotaKey.getSpec().getResource().getType().name();
            typeLabelValues[index] = valuesPool.putIfAbsent(type);
            typeLabelValues[index] = valuesPool.putIfAbsent(type);
            final String spec = quotaKey.getSpec().getDescription().replaceAll("\0", "");
            specLabelValues[index] = valuesPool.putIfAbsent(spec);
            specLabelValues[index] = valuesPool.putIfAbsent(spec);
            final String segment = Solomon.getSegmentsLabelValue(quotaKey).replaceAll("\0", "");
            segmentLabelValues[index] = valuesPool.putIfAbsent(segment);
            segmentLabelValues[index] = valuesPool.putIfAbsent(segment);
        }

        private void writeLabels(@NotNull final EncodeStream out, final int index, final boolean max) {
            // Seven sensor labels total
            out.writeVarint32(7);
            out.writeVarint32(sensorLabelKey.index);
            out.writeVarint32(sensorLabelValues[index].index);
            out.writeVarint32(projectKeyLabelKey.index);
            out.writeVarint32(projectKeyLabelValues[index].index);
            out.writeVarint32(serviceNameLabelKey.index);
            out.writeVarint32(serviceNameLabelValues[index].index);
            out.writeVarint32(typeLabelKey.index);
            out.writeVarint32(typeLabelValues[index].index);
            out.writeVarint32(specLabelKey.index);
            out.writeVarint32(specLabelValues[index].index);
            out.writeVarint32(attributeLabelKey.index);
            if (max) {
                out.writeVarint32(maxAttributeLabelValue.index);
            } else {
                out.writeVarint32(actualAttributeLabelValue.index);
            }
            out.writeVarint32(segmentLabelKey.index);
            out.writeVarint32(segmentLabelValues[index].index);
        }

    }

}
