package ru.yandex.qe.dispenser.ws;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;

import ru.yandex.qe.dispenser.api.v1.DiQuotaSpec;
import ru.yandex.qe.dispenser.domain.QuotaSpec;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.domain.dao.quota.spec.QuotaSpecDao;
import ru.yandex.qe.dispenser.ws.param.QuotaSpecParam;
import ru.yandex.qe.dispenser.ws.param.ResourceParam;
import ru.yandex.qe.dispenser.ws.reqbody.QuotaSpecBody;

@Path("/v1/quota-specifications")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("quota-spec")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class QuotaSpecService extends ServiceBase {
    public static final String QUOTA_SPEC_KEY = "quota_spec_key";

    @Autowired
    private QuotaSpecDao quotaSpecDao;

    @PUT
    @Access(serviceAdmin = true)
    @Consumes(MediaType.APPLICATION_JSON)
    @Path("/{" + ServiceService.SERVICE_KEY + "}/{" + ServiceResourceService.RESOURCE_KEY + "}/{" + QUOTA_SPEC_KEY + "}")
    public DiQuotaSpec create(@PathParam(ServiceService.SERVICE_KEY) @NotNull final Service service,
                              @PathParam(ServiceResourceService.RESOURCE_KEY) @NotNull final String resourceKey,
                              @PathParam(QUOTA_SPEC_KEY) @NotNull final String quotaSpecKey,
                              @RequestBody @NotNull final QuotaSpecBody body) {
        final ResourceParam resourceParam = new ResourceParam(service, resourceKey);
        final Resource resource = resourceParam.get();
        ServiceEndpointUtils.memoizeServiceForResource(resource);
        final QuotaSpec quotaSpec = new QuotaSpec.Builder(quotaSpecKey, resource)
                .description(body.getDescription())
                .build();
        return quotaSpecDao.create(quotaSpec).toView();
    }

    @GET
    @Path("/{" + ServiceService.SERVICE_KEY + "}/{" + ServiceResourceService.RESOURCE_KEY + "}/{" + QUOTA_SPEC_KEY + "}")
    public DiQuotaSpec read(@PathParam(ServiceService.SERVICE_KEY) final Service service,
                            @PathParam(ServiceResourceService.RESOURCE_KEY) final String resourceKey,
                            @PathParam(QUOTA_SPEC_KEY) final String quotaSpecKey) {
        final QuotaSpecParam quotaSpecParam = new QuotaSpecParam(service, resourceKey, quotaSpecKey);
        final QuotaSpec quotaSpec = quotaSpecParam.get();
        ServiceEndpointUtils.memoizeServiceForQuotaSpec(quotaSpec);
        return quotaSpec.toView();
    }

    @POST
    @Access(serviceAdmin = true)
    @Consumes(MediaType.APPLICATION_JSON)
    @Path("/{" + Access.SERVICE_KEY + "}/{" + ServiceResourceService.RESOURCE_KEY + "}/{" + QUOTA_SPEC_KEY + "}")
    public DiQuotaSpec update(@PathParam(ServiceService.SERVICE_KEY) @NotNull final Service service,
                              @PathParam(ServiceResourceService.RESOURCE_KEY) @NotNull final String resourceKey,
                              @PathParam(QUOTA_SPEC_KEY) @NotNull final String quotaSpecKey,
                              @RequestBody final @NotNull QuotaSpecBody body) {
        final ResourceParam resourceParam = new ResourceParam(service, resourceKey);
        final Resource resource = resourceParam.get();
        ServiceEndpointUtils.memoizeServiceForResource(resource);
        final QuotaSpec quotaSpec = new QuotaSpec.Builder(quotaSpecKey, resource)
                .description(body.getDescription())
                .build();
        quotaSpec.setId(quotaSpecDao.read(quotaSpec.getKey()).getId());
        quotaSpecDao.update(quotaSpec);
        return quotaSpec.toView();
    }

    @DELETE
    @Access(serviceAdmin = true)
    @Path("/{" + Access.SERVICE_KEY + "}/{" + ServiceResourceService.RESOURCE_KEY + "}/{" + QUOTA_SPEC_KEY + "}")
    public DiQuotaSpec delete(@PathParam(ServiceService.SERVICE_KEY) final Service service,
                              @PathParam(ServiceResourceService.RESOURCE_KEY) final String resourceKey,
                              @PathParam(QUOTA_SPEC_KEY) final String quotaSpecKey) {
        final QuotaSpecParam quotaSpecParam = new QuotaSpecParam(service, resourceKey, quotaSpecKey);
        final QuotaSpec quotaSpec = quotaSpecParam.get();
        ServiceEndpointUtils.memoizeServiceForQuotaSpec(quotaSpec);
        quotaSpecDao.delete(quotaSpec);
        return quotaSpec.toView();
    }
}
