package ru.yandex.qe.dispenser.ws;

import java.util.Comparator;
import java.util.Set;
import java.util.stream.Collectors;

import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;

import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.qe.dispenser.api.v1.DiResourceGroup;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.domain.ResourceGroup;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.resource.group.ResourceGroupDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.util.ValidationUtils;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.param.ResourceGroupParam;
import ru.yandex.qe.dispenser.ws.reqbody.ResourceGroupBody;

import static ru.yandex.qe.dispenser.ws.ServiceService.SERVICE_KEY;

@Path("/v1/services/{" + SERVICE_KEY + "}/resource-groups")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("resource-groups")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class ResourceGroupService {
    public static final String RESOURCE_GROUP_KEY = "resource_group_key";

    public final static Comparator<ResourceGroup> GROUP_COMPARATOR = Comparator
            .comparing(ResourceGroup::getPriority, Comparator.nullsLast(Integer::compareTo))
            .thenComparing(ResourceGroup::getKey);

    @Autowired
    private ResourceGroupDao resourceGroupDao;

    @NotNull
    @GET
    public DiListResponse<DiResourceGroup> getResourceGroups(@PathParam(SERVICE_KEY) @NotNull final Service service) {
        final Set<ResourceGroup> groups = Hierarchy.get().getResourceGroupReader().read(service);

        return new DiListResponse<>(groups.stream()
                .sorted(GROUP_COMPARATOR)
                .map(ResourceGroup::toView)
                .collect(Collectors.toList())
        );
    }

    @GET
    @Path("/{" + RESOURCE_GROUP_KEY + "}")
    public DiResourceGroup getResourceGroup(@PathParam(SERVICE_KEY) final Service service,
                                            @PathParam(RESOURCE_GROUP_KEY) final String resourceGroupKey) {
        final ResourceGroupParam resourceGroupParam = new ResourceGroupParam(service, resourceGroupKey);
        return resourceGroupParam.get().toView();
    }

    @NotNull
    @POST
    @Access(serviceAdmin = true)
    public DiResourceGroup createResourceGroup(@PathParam(SERVICE_KEY) @NotNull final Service service, final ResourceGroupBody body) {
        final String key = ValidationUtils.requireNonNull(body.getKey(), "Key field is required");
        final String name = ValidationUtils.requireNonNull(body.getName(), "Name field is required");

        final ResourceGroup group = new ResourceGroup.Builder(key, service)
                .name(name)
                .priority(body.getPriority())
                .build();

        return resourceGroupDao.create(group).toView();
    }

    @NotNull
    @PUT
    @Path("/{" + RESOURCE_GROUP_KEY + "}")
    @Access(serviceAdmin = true)
    public DiResourceGroup updateResourceGroup(@PathParam(SERVICE_KEY) final Service service,
                                               @PathParam(RESOURCE_GROUP_KEY) final String resourceGroupKey,
                                               final ResourceGroupBody body) {
        final ResourceGroupParam resourceGroupParam = new ResourceGroupParam(service, resourceGroupKey);
        final ResourceGroup originalGroup = resourceGroupParam.get();

        final ResourceGroup modifiedGroup = new ResourceGroup.Builder(originalGroup)
                .name(ValidationUtils.requireNonNull(body.getName(), "Name field is required"))
                .priority(body.getPriority())
                .build();

        resourceGroupDao.update(modifiedGroup);

        return modifiedGroup.toView();
    }

    @DELETE
    @Path("/{" + RESOURCE_GROUP_KEY + "}")
    @Access(serviceAdmin = true)
    public DiResourceGroup deleteResourceGroup(@PathParam(SERVICE_KEY) final Service service,
                                               @PathParam(RESOURCE_GROUP_KEY) final String resourceGroupKey) {
        final ResourceGroupParam resourceGroupParam = new ResourceGroupParam(service, resourceGroupKey);
        final ResourceGroup resourceGroup = resourceGroupParam.get();
        resourceGroupDao.delete(resourceGroup);
        return resourceGroup.toView();
    }
}
