package ru.yandex.qe.dispenser.ws;

import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.DefaultValue;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Response;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestBody;

import ru.yandex.qe.dispenser.api.v1.DiTicketChangeNotification;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.quota.request.QuotaChangeRequestManager;
import ru.yandex.qe.dispenser.ws.quota.request.ticket.QuotaChangeRequestTicketManager;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.PerformerContext;

@ParametersAreNonnullByDefault
@Path("/v1/trigger")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@Service("resource-preorder-comment-trigger")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class ResourcePreOrderCommentTriggerService {

    @Autowired
    private QuotaChangeRequestDao requestDao;

    @Autowired
    private QuotaChangeRequestManager quotaChangeRequestManager;

    @Autowired
    private QuotaChangeRequestTicketManager quotaChangeRequestTicketManager;

    @Value("${tracker.trigger.action.allowed.perfomer}")
    private String allowedPerfomer;

    @POST
    @Access
    @Path("resource-preorder-comment")
    public Response onTicketComment(@QueryParam("suppressSummon") @DefaultValue("false") boolean suppressSummon,
                                    @RequestBody final TriggerData data) {
        final Person person = Session.WHOAMI.get();
        final String login = person.getLogin();
        if (!login.equals(allowedPerfomer)) {
            return Response
                    .status(Response.Status.FORBIDDEN)
                    .build();
        }

        requestDao.findByTicketKey(data.getTicketKey())
                .ifPresent(request -> tryReopenRequest(request, person, suppressSummon));

        return Response.ok().build();
    }

    @Access
    @POST
    @Path("_ticketChanged")
    public Response notifyTicketChanged(@QueryParam("suppressSummon") @DefaultValue("false") boolean suppressSummon,
                                        @RequestBody final DiTicketChangeNotification notification) {
        final Optional<QuotaChangeRequest> requestO = requestDao.findByTicketKey(notification.getTicketKey());
        if (!requestO.isPresent()) {
            return Response.status(Response.Status.NOT_FOUND).build();
        }

        final boolean res = quotaChangeRequestTicketManager.sendDescriptionChangedComment(requestO.get(), notification.getChangedBy(),
                notification.getAuthor(), suppressSummon);

        if (res) {
            return Response.ok().build();
        } else {
            return Response.accepted().build();
        }
    }

    private void tryReopenRequest(final QuotaChangeRequest quotaChangeRequest, final Person performer,
                                  boolean suppressSummon) {
        if (quotaChangeRequest.getStatus() != QuotaChangeRequest.Status.NEW
                && quotaChangeRequest.getType() == QuotaChangeRequest.Type.RESOURCE_PREORDER) {
            quotaChangeRequestManager.setStatus(quotaChangeRequest, QuotaChangeRequest.Status.NEW,
                    new PerformerContext(performer), suppressSummon);
        }
    }

    public static class TriggerData {
        private final String ticketKey;

        @JsonCreator
        public TriggerData(@JsonProperty("ticketKey") final String ticketKey) {
            this.ticketKey = ticketKey;
        }

        public String getTicketKey() {
            return ticketKey;
        }
    }
}
