package ru.yandex.qe.dispenser.ws;

import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Inject;
import javax.validation.constraints.NotNull;
import javax.ws.rs.BeanParam;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.HttpHeaders;
import javax.ws.rs.core.Response;
import javax.ws.rs.core.StreamingOutput;

import com.opencsv.CSVWriter;
import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;

import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestReader;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.param.QuotaChangeRequestFilterParam;
import ru.yandex.qe.dispenser.ws.param.QuotaChangeRequestFilterParamConverter;
import ru.yandex.qe.dispenser.ws.quota.request.RequestPreOrderAggregationReport;

import static ru.yandex.qe.dispenser.ws.QuotaChangeRequestExportService.CONTENT_DISPOSITION_TEMPLATE;

@ParametersAreNonnullByDefault
@Path("/v1/resource-preorder/export")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("quota-pre-order-export")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class ResourcePreOrderExportService {

    private final RequestPreOrderAggregationReport report;
    private final QuotaChangeRequestFilterParamConverter filterParamConverter;

    @Inject
    public ResourcePreOrderExportService(final RequestPreOrderAggregationReport report,
                                         final QuotaChangeRequestFilterParamConverter filterParamConverter) {
        this.report = report;
        this.filterParamConverter = filterParamConverter;
    }

    @NotNull
    @GET
    @Access
    @Produces(ServiceBase.TEXT_CSV_UTF_8)
    @Path("/csv")
    public Response exportRequestsToCsv(@BeanParam final QuotaChangeRequestFilterParam filterParam,
                                        @QueryParam("pivot") final RequestPreOrderAggregationReport.Pivot pivot) {

        final Optional<QuotaChangeRequestReader.QuotaChangeRequestFilter> filter = filterParamConverter.fromParam(filterParam);
        if (!filter.isPresent()) {
            return Response.ok("").header(HttpHeaders.CONTENT_DISPOSITION, String.format(CONTENT_DISPOSITION_TEMPLATE, "requests.csv")).build();
        }

        final StreamingOutput output = new StreamingOutput() {
            @Override
            public void write(final OutputStream outputStream) throws IOException, WebApplicationException {
                final BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(outputStream));
                final CSVWriter csvWriter = new CSVWriter(writer);
                report.writeReport(csvWriter, filter.get(), pivot);
                writer.close();
            }
        };

        return Response.ok(output).header(HttpHeaders.CONTENT_DISPOSITION, String.format(CONTENT_DISPOSITION_TEMPLATE, "requests.csv")).build();

    }
}
