package ru.yandex.qe.dispenser.ws;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.ws.rs.WebApplicationException;
import javax.ws.rs.core.Response;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.api.util.ValidationUtils;
import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiSetAmountResult;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.dao.service.ServiceReader;
import ru.yandex.qe.dispenser.domain.exception.SingleMessageException;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.domain.i18n.LocalizableString;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.util.LocalizationUtils;
import ru.yandex.qe.dispenser.ws.aspect.AccessAspect;
import ru.yandex.qe.dispenser.ws.bot.Provider;
import ru.yandex.qe.dispenser.ws.param.DiExceptionMapper;
import ru.yandex.qe.dispenser.ws.quota.request.SetResourceAmountBody;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.PerformerContext;
import ru.yandex.qe.dispenser.ws.reqbody.SetResourceAmountBodyOptional;

@Component
public class ResourcePreorderRequestUtils {

    private static final Set<QuotaChangeRequest.Status> STATUSES_FOR_QUOTA_CHANGE = Sets.immutableEnumSet(
            QuotaChangeRequest.Status.NEW,
            QuotaChangeRequest.Status.READY_FOR_REVIEW,
            QuotaChangeRequest.Status.APPROVED,
            QuotaChangeRequest.Status.NEED_INFO,
            QuotaChangeRequest.Status.CONFIRMED
    );

    private static final Set<QuotaChangeRequest.Status> STATUSES_FOR_ALLOCATION = Sets.immutableEnumSet(
            QuotaChangeRequest.Status.CONFIRMED
    );

    private static final Set<Provider> PROVIDERS_TO_UNCONFIRMED_ALLOCATION = Sets.immutableEnumSet(
            Provider.LOGBROKER,
            Provider.RTMR,
            Provider.RTMR_PROCESSING,
            Provider.RTMR_MIRROR
    );

    private static final Set<QuotaChangeRequest.Status> STATUTES_FOR_UNCONFIRMED_ALLOCATION = Sets.immutableEnumSet(
            QuotaChangeRequest.Status.READY_FOR_REVIEW,
            QuotaChangeRequest.Status.APPROVED,
            QuotaChangeRequest.Status.CONFIRMED
    );

    private final QuotaChangeRequestDao requestDao;
    private final CampaignDao campaignDao;
    private final MessageSource errorsMessageSource;

    public ResourcePreorderRequestUtils(final QuotaChangeRequestDao requestDao,
                                        final CampaignDao campaignDao,
                                        @Qualifier("errorMessageSource") final MessageSource errorsMessageSource) {
        this.requestDao = requestDao;
        this.campaignDao = campaignDao;
        this.errorsMessageSource = errorsMessageSource;
    }

    public QuotaRequestChangeValues getChanges(final SetResourceAmountBody body, final PerformerContext context) {

        final List<Error> errors = new ArrayList<>();
        final QuotaRequestChangeValues changeValues = new QuotaRequestChangeValues();

        final Set<Long> requestIds = getRequestIds(body);
        final Set<String> ticketKeys = getTicketKeys(body);

        checkIdentification(requestIds, ticketKeys);

        final Function<SetResourceAmountBody.Item, QuotaChangeRequest> requestExtractor;
        final Function<QuotaChangeRequest, SetResourceAmountBody.Item> itemExtractor;
        final Collection<QuotaChangeRequest> values;

        if (!requestIds.isEmpty()) {
            final Map<Long, QuotaChangeRequest> requestById = requestDao.read(requestIds);
            validateRequestIds(errors, Sets.difference(requestIds, requestById.keySet()));
            requestExtractor = item -> requestById.get(item.getRequestId());

            final Map<Long, SetResourceAmountBody.Item> itemByRequestId = getRequestToItemMapping(body, SetResourceAmountBody.Item::getRequestId);
            itemExtractor = req -> itemByRequestId.get(req.getId());
            values = requestById.values();
        } else {
            final Map<String, QuotaChangeRequest> requestByTicket = requestDao.findByTicketKeys(ticketKeys);
            validateTicketKeys(errors, Sets.difference(ticketKeys, requestByTicket.keySet()));
            requestExtractor = item -> requestByTicket.get(item.getTicketKey());

            final Map<String, SetResourceAmountBody.Item> itemByRequestKey = getRequestToItemMapping(body, SetResourceAmountBody.Item::getTicketKey);
            itemExtractor = req -> itemByRequestKey.get(req.getTrackerIssueKey());
            values = requestByTicket.values();
        }

        validateRequestsOnDuplicates(errors, body);

        final Map<QuotaChangeRequest, String> serviceKeyByRequest = new HashMap<>();
        final Map<QuotaChangeRequest, Long> bigOrderIdByRequest = new HashMap<>();
        for (final QuotaChangeRequest request : values) {
            if (request.getCampaign().isSingleProviderRequestModeEnabled() && !request.getChanges().isEmpty()) {
                final QuotaChangeRequest.Change change = request.getChanges().iterator().next();
                serviceKeyByRequest.put(request, change.getResource().getService().getKey());
                bigOrderIdByRequest.put(request, change.getKey().getBigOrder().getId());
            }
        }

        final Map<Boolean, List<QuotaChangeRequest>> requestsWithInvalidStatuses = values.stream()
                .filter(
                        r -> !STATUSES_FOR_QUOTA_CHANGE.contains(r.getStatus())
                                || cannotAllocateRequestChanges(r, itemExtractor.apply(r))
                )
                .collect(Collectors.partitioningBy(p -> STATUSES_FOR_QUOTA_CHANGE.contains(p.getStatus()),
                        Collectors.mapping(Function.identity(), Collectors.toList())));

        if (!requestsWithInvalidStatuses.isEmpty()) {
            if (!requestsWithInvalidStatuses.getOrDefault(false, List.of()).isEmpty()) {
                errors.add(Error.of(requestsWithInvalidStatuses.get(false), "wrong.status.of.requests"));
            }
            if (!requestsWithInvalidStatuses.getOrDefault(true, List.of()).isEmpty()) {
                errors.add(Error.of(requestsWithInvalidStatuses.get(true), "confirmed.status.required.for.requests"));
            }
        }

        final Multimap<String, QuotaChangeRequest> requestByServiceKey = HashMultimap.create();

        for (final SetResourceAmountBody.Item update : body.getUpdates()) {
            for (final SetResourceAmountBody.ChangeBody change : update.getChanges()) {
                final QuotaChangeRequest request = requestExtractor.apply(update);
                final String serviceKey = ValidationUtils.requireNonNull(
                        serviceKeyByRequest.getOrDefault(request, change.getServiceKey()),
                        "Service key must be non-null"
                );
                requestByServiceKey.put(serviceKey, request);
            }
        }

        final Set<QuotaChangeRequest> invalidServiceRequests = new HashSet<>();
        final ServiceReader serviceReader = Hierarchy.get().getServiceReader();
        for (final String serviceKey : requestByServiceKey.keySet()) {
            final Service service = serviceReader.read(serviceKey);
            if (!canPersonChangeServiceRequests(context.getPerson(), service)) {
                invalidServiceRequests.addAll(requestByServiceKey.get(serviceKey));
            }
        }

        if (!invalidServiceRequests.isEmpty()) {
            errors.add(Error.of(invalidServiceRequests, "only.service.admin.can.modify.requests"));
        }

        final Set<Long> campaignIds = values.stream()
                .map(QuotaChangeRequest::getCampaignId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        final Map<Long, QuotaChangeRequest.BigOrder> bigOrderById = campaignDao.read(campaignIds).values().stream()
                .flatMap(c -> c.getBigOrders().stream())
                .distinct()
                .collect(Collectors.toMap(Campaign.BigOrder::getBigOrderId, c -> new QuotaChangeRequest.BigOrder(c.getBigOrderId(), c.getDate(), true)));

        final Set<QuotaChangeRequest> requestsWithInvalidChangeKey = new HashSet<>();
        final Set<QuotaChangeRequest> requestsWithInvalidReadyAmount = new HashSet<>();
        final Set<QuotaChangeRequest> requestsWithInvalidAllocatedAmount = new HashSet<>();

        for (final SetResourceAmountBody.Item update : body.getUpdates()) {
            final QuotaChangeRequest request = requestExtractor.apply(update);
            if (request == null) {
                continue;
            }

            final Map<QuotaChangeRequest.ChangeKey, QuotaChangeRequest.Change> changeByKey = request.getChanges().stream()
                    .collect(Collectors.toMap(QuotaChangeRequest.Change::getKey, Function.identity(), (l, r) -> l));

            for (final SetResourceAmountBody.ChangeBody changeBody : update.getChanges()) {
                if (changeBody.isEmpty()) {
                    continue;
                }
                final Long bigOrderId =  ValidationUtils.requireNonNull(
                        bigOrderIdByRequest.getOrDefault(requestExtractor.apply(update), changeBody.getBigOrderId()),
                        "BigOrder id key must be non-null"
                );
                final QuotaChangeRequest.BigOrder bigOrder = bigOrderById.get(bigOrderId);
                final String serviceKey = ValidationUtils.requireNonNull(
                        serviceKeyByRequest.getOrDefault(requestExtractor.apply(update), changeBody.getServiceKey()),
                        "Service key must be non-null"
                );
                final QuotaChangeRequest.ChangeKey key = QuotaChangeRequest.ChangeKey.fromBodyValues(serviceKey, changeBody.getResourceKey(), changeBody.getSegmentKeys(), bigOrder);
                if (!changeByKey.containsKey(key)) {
                    requestsWithInvalidChangeKey.add(request);
                    break;
                }
                final QuotaChangeRequest.Change change = changeByKey.get(key);
                final DiUnit baseUnit = key.getResource().getType().getBaseUnit();
                final long changeId = change.getId();
                if (changeBody.getAmountReady() != null) {
                    final long amount = baseUnit.convert(changeBody.getAmountReady());
                    if (amount > change.getAmount()) {
                        requestsWithInvalidReadyAmount.add(request);
                    } else {
                        changeValues.setReady(request, changeId, baseUnit.convert(changeBody.getAmountReady()));
                    }
                }
                if (changeBody.getAmountAllocated() != null) {
                    final long amount = baseUnit.convert(changeBody.getAmountAllocated());
                    if (amount > change.getAmount() || amount < change.getAmountAllocated()) {
                        requestsWithInvalidAllocatedAmount.add(request);
                    } else {
                        if (change.getAmountAllocated() == change.getAmountAllocating() && amount > change.getAmountAllocated()) {
                            changeValues.setAllocated(request, change, amount);
                            changeValues.setAllocating(request, changeId, amount);
                        } else if (change.getAmountAllocating() > change.getAmountAllocated()) {
                            if (amount > change.getAmountAllocating()) {
                                requestsWithInvalidAllocatedAmount.add(request);
                            } else {
                                changeValues.setAllocated(request, change, amount);
                            }
                        }
                    }
                }
            }
            changeValues.setRequestComment(request, update.getComment());
        }

        validateResources(errors, requestsWithInvalidChangeKey);

        if (!requestsWithInvalidReadyAmount.isEmpty()) {
            errors.add(Error.of(requestsWithInvalidReadyAmount, "amount.ready.must.be.lesser.than.ordered"));
        }

        if (!requestsWithInvalidAllocatedAmount.isEmpty()) {
            errors.add(Error.of(requestsWithInvalidAllocatedAmount, "amount.allocated.must.be.correct"));
        }

        throwOnError(errors);

        return changeValues;
    }

    private static boolean cannotAllocateRequestChanges(QuotaChangeRequest request, SetResourceAmountBody.Item item) {
        return (cannotAllocateRequstCommon(request, item) && !canAllocateRequestChangesInSpecialProviders(request, item))
                || Objects.requireNonNull(request.getCampaign()).getType() != Campaign.Type.AGGREGATED;
    }

    private static boolean cannotAllocateRequstCommon(QuotaChangeRequest request, SetResourceAmountBody.Item item) {
        return !STATUSES_FOR_ALLOCATION.contains(request.getStatus()) &&
                item.getChanges()
                        .stream()
                        .anyMatch(c -> c.getAmountAllocated() != null);
    }

    private static boolean canAllocateRequestChangesInSpecialProviders(QuotaChangeRequest request,
                                                                SetResourceAmountBody.Item item) {
        return STATUTES_FOR_UNCONFIRMED_ALLOCATION.contains(request.getStatus()) &&
                item.getChanges()
                        .stream()
                        .allMatch(c -> c.getAmountAllocated() != null &&
                                Optional.ofNullable(c.getServiceKey())
                                        .flatMap(Provider::fromKey)
                                        .filter(PROVIDERS_TO_UNCONFIRMED_ALLOCATION::contains)
                                        .isPresent()
                        );
    }

    private void validateRequestsOnDuplicates(final List<Error> errors, final SetResourceAmountBody body) {
        final Set<Long> uniqueIds = new HashSet<>();

        final Set<Long> duplicateIds = body.getUpdates().stream()
                .map(SetResourceAmountBody.Item::getRequestId)
                .filter(Objects::nonNull)
                .filter(e -> !uniqueIds.add(e))
                .collect(Collectors.toSet());

        final Set<String> uniqueTickets = new HashSet<>();

        final Set<String> duplicateTickets = body.getUpdates().stream()
                .map(SetResourceAmountBody.Item::getTicketKey)
                .filter(Objects::nonNull)
                .filter(e -> !uniqueTickets.add(e))
                .collect(Collectors.toSet());


        if (!duplicateIds.isEmpty()) {
            errors.add(new Error(LocalizableString.of("user.request.has.duplicates"), duplicateTickets, duplicateIds));
        }
    }

    private static <K> Map<K, SetResourceAmountBody.Item> getRequestToItemMapping(final SetResourceAmountBody body,
                                                                                  final Function<SetResourceAmountBody.Item, K> keyExtractor) {
        return body.getUpdates().stream()
                .filter(i -> keyExtractor.apply(i) != null)
                .collect(Collectors.toMap(keyExtractor, Function.identity(), (l, r) -> l));
    }

    @NotNull
    private static Set<Long> getRequestIds(final SetResourceAmountBody body) {
        return body.getUpdates().stream()
                .map(SetResourceAmountBody.Item::getRequestId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
    }

    @NotNull
    private static Set<String> getTicketKeys(final SetResourceAmountBody body) {
        return body.getUpdates().stream()
                .map(SetResourceAmountBody.Item::getTicketKey)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
    }

    @NotNull
    private static Boolean canPersonChangeServiceRequests(final Person person, final Service service) {
        return AccessAspect.isServiceAdmin(person, service)
                || Hierarchy.get().getServiceReader().getTrustees(service).contains(person);
    }

    public SetResourceAmountBody getFullBody(@NotNull final SetResourceAmountBodyOptional body) {
        final List<SetResourceAmountBodyOptional.Item> updatesFromBody = body.getUpdates();
        final Set<Long> requestIds = getRequestIds(updatesFromBody);
        final Set<String> ticketKeys = getTicketKeys(updatesFromBody);
        checkIdentification(requestIds, ticketKeys);

        final List<Error> errors = new ArrayList<>();
        final Function<SetResourceAmountBodyOptional.Item, QuotaChangeRequest> requestExtractor;
        if (!requestIds.isEmpty()) {
            final Map<Long, QuotaChangeRequest> requestById = requestDao.read(requestIds);
            validateRequestIds(errors, Sets.difference(requestIds, requestById.keySet()));
            requestExtractor = item -> requestById.get(item.getRequestId());
        } else {
            final Map<String, QuotaChangeRequest> requestByTicket = requestDao.findByTicketKeys(ticketKeys);
            validateTicketKeys(errors, Sets.difference(ticketKeys, requestByTicket.keySet()));
            requestExtractor = item -> requestByTicket.get(item.getTicketKey());
        }

        final List<SetResourceAmountBody.Item> updates = new ArrayList<>();
        final Set<QuotaChangeRequest> requestsWithInvalidChangeKey = new HashSet<>();

        for (final SetResourceAmountBodyOptional.Item item :
                updatesFromBody) {

            final QuotaChangeRequest quotaChangeRequest = requestExtractor.apply(item);

            if (quotaChangeRequest == null) {
                continue;
            }

            final SetResourceAmountBodyOptional.UpdateFor updateFor = body.getUpdateFor();
            final List<QuotaChangeRequest.Change> changesFromRequest = quotaChangeRequest.getChanges();
            final List<SetResourceAmountBodyOptional.ChangeBody> changes = item.getChanges();

            final List<SetResourceAmountBody.ChangeBody> changesForBody = new ArrayList<>();

            if (changes == null) {
                changesForBody.addAll(getMappedChanges(updateFor, null, changesFromRequest));
            } else {
                for (final SetResourceAmountBodyOptional.ChangeBody changeBody :
                        changes) {

                    final Collection<QuotaChangeRequest.Change> optionalChanges =
                            getChangesForBody(changeBody, changesFromRequest);

                    if (optionalChanges == null) {
                        requestsWithInvalidChangeKey.add(quotaChangeRequest);
                    } else {
                        changesForBody.addAll(getMappedChanges(updateFor, changeBody, optionalChanges));
                    }
                }
            }
            updates.add(new SetResourceAmountBody.Item(item.getRequestId(), item.getTicketKey(), changesForBody, item.getComment()));
        }

        validateResources(errors, requestsWithInvalidChangeKey);
        throwOnError(errors);

        return new SetResourceAmountBody(updates);
    }

    @Nullable
    private static Collection<QuotaChangeRequest.Change> getChangesForBody(@NotNull final SetResourceAmountBodyOptional.ChangeBody changeBody,
                                                                           final List<QuotaChangeRequest.Change> changesFromRequest) {
        if (changeBody.isEmpty()) {
            return changesFromRequest;
        }

        final String serviceKey = changeBody.getServiceKey();
        final String resourceKey = changeBody.getResourceKey();
        final Set<String> segmentKeys = changeBody.getSegmentKeys();
        final Long orderId = changeBody.getOrderId();

        Stream<QuotaChangeRequest.Change> stream = changesFromRequest.stream();
        if (serviceKey != null) {
            stream = stream.filter(change -> change.getResource().getService().getKey().equals(serviceKey));
        }
        if (resourceKey != null) {
            stream = stream.filter(change -> change.getResource().getPublicKey().equals(resourceKey));
        }
        if (segmentKeys != null) {
            stream = stream.filter(change -> change.getSegments()
                    .stream()
                    .map(Segment::getPublicKey)
                    .collect(Collectors.toSet())
                    .equals(segmentKeys)
            );
        }
        if (orderId != null) {
            stream = stream.filter(c -> c.getBigOrder().getId() == orderId);
        }

        final List<QuotaChangeRequest.Change> result = stream.collect(Collectors.toList());
        if (result.isEmpty()) {
            return null;
        }
        return result;
    }

    private void throwOnError(@NotNull final List<Error> errors) {
        if (!errors.isEmpty()) {
            final List<DiSetAmountResult.Errors.Item> items = errors.stream()
                    .map(e -> new DiSetAmountResult.Errors.Item(LocalizationUtils.resolveWithDefaultAsKey(errorsMessageSource, e.getMessage(), Session.USER_LOCALE.get()),
                            e.getProblemRequestIds(), e.getProblemTicketKeys()))
                    .collect(Collectors.toList());

            throw new WebApplicationException(Response.status(DiExceptionMapper.ExtraStatus.UNPROCESSABLE_ENTITY)
                    .entity(new DiSetAmountResult.Errors(items))
                    .build());
        }
    }

    private static void validateResources(final List<Error> errors, @NotNull final Set<QuotaChangeRequest> requestsWithInvalidChangeKey) {
        if (!requestsWithInvalidChangeKey.isEmpty()) {
            errors.add(Error.of(requestsWithInvalidChangeKey, "modified.resources.not.presented"));
        }
    }

    private static void validateRequestIds(final List<Error> errors, @NotNull final Set<Long> invalidRequestIds) {
        if (!invalidRequestIds.isEmpty()) {
            errors.add(new Error(LocalizableString.of("unknown.request.ids"), Collections.emptySet(), invalidRequestIds));
        }
    }

    private static void validateTicketKeys(final List<Error> errors, @NotNull final Set<String> invalidTicketKeys) {
        if (!invalidTicketKeys.isEmpty()) {
            errors.add(new Error(LocalizableString.of("unknown.ticket.keys"), invalidTicketKeys, Collections.emptySet()));
        }
    }

    private static void checkIdentification(final Collection<Long> requestIds, final Collection<String> ticketKeys) {
        if (!requestIds.isEmpty() && !ticketKeys.isEmpty()) {
            throw SingleMessageException.illegalArgument("updates.must.contain.only.id.or.ticket.key");
        }
    }

    @NotNull
    private static Set<Long> getRequestIds(@NotNull final List<SetResourceAmountBodyOptional.Item> updatesFromBody) {
        return updatesFromBody.stream()
                .map(SetResourceAmountBodyOptional.Item::getRequestId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
    }

    @NotNull
    private static Set<String> getTicketKeys(@NotNull final List<SetResourceAmountBodyOptional.Item> updatesFromBody) {
        return updatesFromBody.stream()
                .map(SetResourceAmountBodyOptional.Item::getTicketKey)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());
    }

    @NotNull
    private static List<SetResourceAmountBody.ChangeBody> getMappedChanges(final SetResourceAmountBodyOptional.UpdateFor updateFor,
                                                                           final SetResourceAmountBodyOptional.ChangeBody changeBody,
                                                                           @NotNull final Collection<QuotaChangeRequest.Change> changes) {
        return changes.stream()
                .map(mappingToChangeBody(changeBody, updateFor))
                .collect(Collectors.toList());
    }

    @NotNull
    private static Function<QuotaChangeRequest.Change, SetResourceAmountBody.ChangeBody> mappingToChangeBody(
            @Nullable final SetResourceAmountBodyOptional.ChangeBody changeBody, final SetResourceAmountBodyOptional.UpdateFor updateFor) {
        if (changeBody == null || changeBody.getSegmentKeys() == null) {
            return change -> new SetResourceAmountBody.ChangeBody(
                    change.getResource().getService().getKey(),
                    change.getKey().getBigOrder() == null ? null : change.getKey().getBigOrder().getId(),
                    change.getResource().getPublicKey(),
                    change.getSegments().stream().map(Segment::getPublicKey).collect(Collectors.toSet()),
                    toDiAmountReady(change, updateFor), toDiAmountAllocated(change, updateFor));
        } else {
            return change -> new SetResourceAmountBody.ChangeBody(
                    change.getResource().getService().getKey(),
                    change.getKey().getBigOrder() == null ? null : change.getKey().getBigOrder().getId(),
                    change.getResource().getPublicKey(),
                    changeBody.getSegmentKeys(),
                    toDiAmountReady(change, updateFor), toDiAmountAllocated(change, updateFor));
        }
    }

    @Nullable
    private static DiAmount toDiAmountReady(final QuotaChangeRequest.Change change, final SetResourceAmountBodyOptional.UpdateFor updateFor) {
        if (updateFor == SetResourceAmountBodyOptional.UpdateFor.READY || updateFor == SetResourceAmountBodyOptional.UpdateFor.BOTH) {
            return DiAmount.of(change.getAmount(), change.getResource().getType().getBaseUnit());
        } else {
            return null;
        }
    }

    @Nullable
    private static DiAmount toDiAmountAllocated(final QuotaChangeRequest.Change change, final SetResourceAmountBodyOptional.UpdateFor updateFor) {
        if (updateFor == SetResourceAmountBodyOptional.UpdateFor.ALLOCATED || updateFor == SetResourceAmountBodyOptional.UpdateFor.BOTH) {
            return DiAmount.of(change.getAmount(), change.getResource().getType().getBaseUnit());
        } else {
            return null;
        }
    }

    private static class Error {
        private final LocalizableString message;
        private final Collection<String> problemTicketKeys;
        private final Collection<Long> problemRequestIds;

        Error(final LocalizableString message, final Collection<QuotaChangeRequest> problemRequests) {
            this.message = message;
            this.problemRequestIds = problemRequests.stream().map(LongIndexBase::getId).collect(Collectors.toList());
            this.problemTicketKeys = problemRequests.stream().map(QuotaChangeRequest::getTrackerIssueKey).collect(Collectors.toList());
        }

        Error(final LocalizableString message, final Collection<String> problemTicketKeys, final Collection<Long> problemRequestIds) {
            this.message = message;
            this.problemTicketKeys = problemTicketKeys;
            this.problemRequestIds = problemRequestIds;
        }

        public LocalizableString getMessage() {
            return message;
        }

        public Collection<String> getProblemTicketKeys() {
            return problemTicketKeys;
        }

        public Collection<Long> getProblemRequestIds() {
            return problemRequestIds;
        }

        public static Error of(final Collection<QuotaChangeRequest> requests, final String key, final Object... args) {
            return new Error(LocalizableString.of(key, args), requests);
        }
    }

}
