package ru.yandex.qe.dispenser.ws;

import java.util.ArrayList;
import java.util.Collection;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Inject;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;

import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;

import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeRequest;
import ru.yandex.qe.dispenser.api.v1.DiRequestPreOrder;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.request.BotPreOrderRequestDao;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.request.ExtendedPreOrderRequest;
import ru.yandex.qe.dispenser.domain.dao.service.ServiceReader;
import ru.yandex.qe.dispenser.domain.exception.SingleMessageException;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.ResourceWorkflow;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;

@ParametersAreNonnullByDefault
@Path("/v1/resource-preorder/{requestId}/preorders")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("request-preorders")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class ResourceRequestPreOrdersService {

    private final BotPreOrderRequestDao preOrderRequestDao;
    private final HierarchySupplier hierarchySupplier;

    @Inject
    public ResourceRequestPreOrdersService(final BotPreOrderRequestDao preOrderRequestDao,
                                           final HierarchySupplier hierarchySupplier) {
        this.preOrderRequestDao = preOrderRequestDao;
        this.hierarchySupplier = hierarchySupplier;
    }

    @Access
    @GET
    public DiListResponse<DiRequestPreOrder> getPreOrders(@PathParam("requestId") final QuotaChangeRequest request) {

        if (request.getType() != QuotaChangeRequest.Type.RESOURCE_PREORDER) {
            throw SingleMessageException.illegalArgument("preorders.exists.only.for.resource.preorder.request");
        }

        final Person performer = Session.WHOAMI.get();

        final boolean canUserViewMoney = ResourceWorkflow.canUserViewBotMoney(performer);

        final Collection<ExtendedPreOrderRequest> preOrderRequests = preOrderRequestDao.getRequestPreorders(request.getId());

        final ArrayList<DiRequestPreOrder> result = new ArrayList<>(preOrderRequests.size());

        final ServiceReader serviceReader = hierarchySupplier.get().getServiceReader();

        for (final ExtendedPreOrderRequest preOrderRequest : preOrderRequests) {

            final Service service = serviceReader.read(preOrderRequest.getServiceId());

            final DiRequestPreOrder.PreOrder preOrderView = new DiRequestPreOrder.PreOrder(preOrderRequest.getPreOrderId(),
                    preOrderRequest.getPreOrderName(),
                    preOrderRequest.getConfigurationName(),
                    preOrderRequest.getPreOrderServersQuantity(),
                    new DiQuotaChangeRequest.Service(service.getKey(), service.getName()));

            final DiRequestPreOrder diRequestPreOrder = new DiRequestPreOrder(preOrderView,
                    preOrderRequest.getServersQuantity(),
                    canUserViewMoney ? preOrderRequest.getCost() : null
            );
            result.add(diRequestPreOrder);
        }

        return new DiListResponse<>(result);
    }
}
