package ru.yandex.qe.dispenser.ws;

import java.util.Comparator;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.PATCH;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;

import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import ru.yandex.qe.dispenser.api.v1.DiSegment;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.domain.CampaignResource;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignResourceDao;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaDao;
import ru.yandex.qe.dispenser.domain.dao.resource.segmentation.ResourceSegmentationDao;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.param.SegmentParam;
import ru.yandex.qe.dispenser.ws.reqbody.SegmentBodyCreate;
import ru.yandex.qe.dispenser.ws.reqbody.SegmentBodyUpdate;

import static ru.yandex.qe.dispenser.ws.SegmentationService.SEGMENTATION_KEY;

@ParametersAreNonnullByDefault
@Controller
@Path("/v1/segmentations/{" + SEGMENTATION_KEY + "}/segments")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("segment")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class SegmentService extends ServiceBase {

    public static final String SEGMENT_KEY = "segment";

    @Autowired
    private SegmentDao segmentDao;

    @Autowired
    private ResourceSegmentationDao resourceSegmentationDao;

    @Autowired
    private QuotaDao quotaDao;

    @Autowired
    private CampaignResourceDao campaignResourceDao;

    @GET
    @NotNull
    public DiListResponse<DiSegment> getSegments(@PathParam(SEGMENTATION_KEY) final Segmentation segmentation) {
        final Set<Segment> segment = Hierarchy.get().getSegmentReader().get(segmentation);

        return new DiListResponse<>(segment.stream()
                .sorted(Comparator.comparing(Segment::getPriority))
                .map(Segment::toView)
                .collect(Collectors.toList()));
    }

    @GET
    @NotNull
    @Path("/{" + SEGMENT_KEY + "}")
    public DiSegment getSegment(@NotNull @PathParam(SEGMENTATION_KEY) final Segmentation segmentation,
                                @NotNull @PathParam(SEGMENT_KEY) final String segmentKey) {
        final SegmentParam segmentParam = new SegmentParam(segmentation, segmentKey);
        return segmentParam.get().toView();
    }

    @POST
    @Access(dispenserAdmin = true)
    @NotNull
    public DiSegment createSegment(@PathParam(SEGMENTATION_KEY) final Segmentation segmentation,
                                   @NotNull final SegmentBodyCreate body) {

        final Segment segment = segmentDao.create(new Segment.Builder(body.getKey(), segmentation)
                .name(body.getName())
                .description(body.getDescription())
                .priority(body.getPriority() == null ? (short) (segmentDao.getMaxPriority(segmentation) + 1) : body.getPriority())
                .build());

        quotaDao.updateQuotaSegments(resourceSegmentationDao.getResources(segmentation));

        return segment.toView();
    }

    @DELETE
    @Access(dispenserAdmin = true)
    @Path("/{" + SEGMENT_KEY + "}")
    public DiSegment deleteSegment(@NotNull @PathParam(SEGMENTATION_KEY) final Segmentation segmentation,
                                   @NotNull @PathParam(SEGMENT_KEY) final String segmentKey) {
        final SegmentParam segmentParam = new SegmentParam(segmentation, segmentKey);
        final Segment segment = segmentParam.get();
        validateCampaignResources(segment);
        final Set<Resource> affectedResources = resourceSegmentationDao.getResources(segment.getSegmentation());

        segmentDao.delete(segment);

        quotaDao.updateQuotaSegments(affectedResources);

        return segment.toView();
    }

    @PATCH
    @Access(dispenserAdmin = true)
    @Path("/{" + SEGMENT_KEY + "}")
    @NotNull
    public DiSegment updateSegment(@NotNull @PathParam(SEGMENTATION_KEY) final Segmentation segmentation,
                                   @NotNull @PathParam(SEGMENT_KEY) final String segmentKey,
                                   @NotNull final SegmentBodyUpdate body) {
        final SegmentParam segmentParam = new SegmentParam(segmentation, segmentKey);
        final long id = segmentParam.get().getId();

        final Segment segment = segmentDao.readForUpdate(id);

        if (body.isEmpty()) {
            return segment.toView();
        }

        final Segment modifiedSegment = new Segment.Builder(segmentParam.get().getKey())
                .name(body.getName() == null ? segment.getName() : body.getName())
                .description(body.getDescription() == null ? segment.getDescription() : body.getDescription())
                .priority(body.getPriority() == null ? segment.getPriority() : body.getPriority())
                .build();
        modifiedSegment.setId(id);

        segmentDao.update(modifiedSegment);

        return modifiedSegment.toView();
    }

    private void validateCampaignResources(@NotNull final Segment segment) {
        final Set<CampaignResource> allCampaignResources = campaignResourceDao.getAll();
        final boolean segmentIsUsed = allCampaignResources.stream().anyMatch(cr -> {
            final boolean hasSegmentation = cr.getSettings().getSegmentations().stream()
                    .anyMatch(s -> s.getSegments().stream().anyMatch(id -> segment.getId() == id));
            final boolean hasSegment = cr.getSettings().getSegmentsBigOrders().stream()
                    .anyMatch(s -> s.getSegments().stream().anyMatch(id -> segment.getId() == id));
            return hasSegmentation || hasSegment;
        });
        if (segmentIsUsed) {
            throw new IllegalArgumentException("This segment is referenced by one or more campaign settings.");
        }
    }

}
