package ru.yandex.qe.dispenser.ws;

import org.apache.commons.lang.math.LongRange;
import org.apache.commons.lang.math.Range;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Value;
import ru.yandex.qe.dispenser.api.v1.response.DiListPageResponse;
import ru.yandex.qe.dispenser.api.v1.response.DiListRelativePageResponse;
import ru.yandex.qe.dispenser.domain.index.LongIndexable;
import ru.yandex.qe.dispenser.domain.util.Page;
import ru.yandex.qe.dispenser.domain.util.PageInfo;
import ru.yandex.qe.dispenser.domain.util.RelativePage;
import ru.yandex.qe.dispenser.domain.util.RelativePageInfo;
import ru.yandex.qe.dispenser.ws.param.PageParam;
import ru.yandex.qe.dispenser.ws.param.PaginationParam;
import ru.yandex.qe.dispenser.ws.param.RelativePageParam;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.UriInfo;
import java.net.URI;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collector;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static ru.yandex.qe.dispenser.ws.param.PageParam.PAGE_PARAM;
import static ru.yandex.qe.dispenser.ws.param.RelativePageParam.FROM_ID;

@ParametersAreNonnullByDefault
public abstract class ServiceBase {
    public static final String CHARSET_UTF_8 = "charset=utf-8";
    public static final String APPLICATION_JSON_UTF_8 = MediaType.APPLICATION_JSON + ";" + CHARSET_UTF_8;
    public static final String TEXT_CSV_UTF_8 = "text/csv;" + CHARSET_UTF_8;
    public static final String TEXT_PLAIN_UTF_8 = MediaType.TEXT_PLAIN + ";" + CHARSET_UTF_8;
    public static final String NO_CACHE = "nocache";

    @Value("${dispenser.cluster.prefix}")
    private String clusterPrefix;

    protected <T, V> DiListPageResponse<V> collectToPage(final PaginationParam paginationParam, final Stream<T> stream,
                                                         final Function<T, V> toViewMapper) {
        return stream
                .sequential()
                .collect(toPageResult(paginationParam, toViewMapper));
    }

    protected <T, V> DiListPageResponse<V> toResponsePage(final PaginationParam paginationParam, final Page<T> page,
                                                          final Function<T, V> toViewMapper) {
        final List<V> items = page.getItems()
                .map(toViewMapper)
                .collect(Collectors.toList());

        return createResponsePage(items, page.getTotalCount(), paginationParam);
    }

    protected <T, V> DiListPageResponse<V> toResponsePageMulti(final PaginationParam paginationParam, final Page<T> page,
                                                               final Function<Collection<T>, Collection<V>> toViewMapper) {
        final Collection<V> items = toViewMapper.apply(page.getItems().collect(Collectors.toList()));

        return createResponsePage(items, page.getTotalCount(), paginationParam);
    }

    protected <T extends LongIndexable, V> DiListRelativePageResponse<V> toResponsePage(final RelativePageParam pageParam, final RelativePage<T> page,
                                                                                        final Function<T, V> toViewMapper) {
        final List<T> items = page.getItems();
        final List<V> views = items.stream()
                .map(toViewMapper)
                .collect(Collectors.toList());
        final Long lastId;
        if (items.isEmpty()) {
            lastId = null;
        } else {
            lastId = items.get(items.size() - 1).getId();
        }
        return createRelativeResponsePage(views, pageParam, lastId, page.isLast());
    }

    protected RelativePageInfo getPageInfo(final RelativePageParam relativePageParam) {
        return new RelativePageInfo(relativePageParam.getFromId(), relativePageParam.getPageSize());
    }

    protected PageInfo getPageInfo(final PageParam pageParam) {
        return new PageInfo(pageParam.getPageNumber(), pageParam.getPageSize());
    }

    private <T, V> Collector<T, PageSequentialResultsAccumulator<T, V>, DiListPageResponse<V>> toPageResult(
            final PaginationParam paginationParam, final Function<T, V> toViewMapper) {
        final long pageNumber = paginationParam.getPageParam().getPageNumber();
        final long pageSize = paginationParam.getPageParam().getPageSize();
        final Range resultsIndexRange = new LongRange((pageNumber - 1) * pageSize, pageNumber * pageSize - 1);


        return Collector.of(
                () -> new PageSequentialResultsAccumulator<>(resultsIndexRange, toViewMapper),
                PageSequentialResultsAccumulator::add,
                PageSequentialResultsAccumulator::merge,
                accumulator -> createResponsePage(accumulator.getPageResults(), accumulator.getCount(), paginationParam)
        );
    }

    private @NotNull <T> DiListRelativePageResponse<T> createRelativeResponsePage(final List<T> items,
                                                                                  final RelativePageParam pageParam,
                                                                                  final Long lastId,
                                                                                  final boolean isLast) {
        return DiListRelativePageResponse.<T>builder()
                .withResults(items)
                .withNextPageUrl(getNextPageUrl(pageParam, lastId, isLast))
                .build();
    }


    private @NotNull <T> DiListPageResponse<T> createResponsePage(final Collection<T> items, final long totalCount,
                                                                  final PaginationParam paginationParam) {
        final long totalPagesCount = calculatePagesCount(totalCount, paginationParam.getPageParam().getPageSize());

        return DiListPageResponse.<T>builder()
                .withResults(items)
                .withTotalResultsCount(totalCount)
                .withTotalPagesCount(totalPagesCount)
                .withNextPageUrl(getNextPageUrl(paginationParam, totalPagesCount))
                .withPreviousPageUrl(getPreviousPageUrl(paginationParam))
                .build();
    }

    private URI getNextPageUrl(final PaginationParam paginationParam, final long totalPagesCount) {
        final long pageNumber = paginationParam.getPageParam().getPageNumber();
        if (pageNumber >= totalPagesCount) {
            return null;
        }
        return getUriWithPage(paginationParam.getUriInfo(), pageNumber + 1);
    }

    private URI getNextPageUrl(final RelativePageParam pageParam, final Long lastId, final boolean isLast) {
        if (isLast) {
            return null;
        }
        return getUriWithFromId(pageParam.getUriInfo(), lastId + 1);
    }

    private URI getPreviousPageUrl(final PaginationParam paginationParam) {
        final long pageNumber = paginationParam.getPageParam().getPageNumber();
        if (pageNumber <= PageParam.FIRST_PAGE_NUMBER) {
            return null;
        }
        return getUriWithPage(paginationParam.getUriInfo(), pageNumber - 1);
    }

    private URI getUriWithPage(final UriInfo uriInfo, final long page) {
        final String originPath = uriInfo.getPath();

        return uriInfo.getRequestUriBuilder()
                .replacePath(clusterPrefix + uriInfo.getBaseUri().getPath() + "/" + originPath)
                .replaceQueryParam(PAGE_PARAM, page)
                .build();
    }

    private URI getUriWithFromId(final UriInfo uriInfo, final long fromId) {
        final String originPath = uriInfo.getPath();

        return uriInfo.getRequestUriBuilder()
                .replacePath(clusterPrefix + uriInfo.getBaseUri().getPath() + "/" + originPath)
                .replaceQueryParam(FROM_ID, fromId)
                .build();
    }

    private static long calculatePagesCount(final long resultsCount, final long pageSize) {
        long count = resultsCount / pageSize;
        if (resultsCount % pageSize > 0) {
            count += 1;
        }
        return count;
    }

    private static class PageSequentialResultsAccumulator<T, V> {

        @NotNull
        private final Range resultsIndexRange;

        private final List<V> pageResults = new ArrayList<>();
        private final Function<T, V> toViewMapper;
        private long count = 0;

        private PageSequentialResultsAccumulator(@NotNull final Range resultsIndexRange,
                                                 final Function<T, V> toViewMapper) {
            this.resultsIndexRange = resultsIndexRange;
            this.toViewMapper = toViewMapper;
        }


        public void add(final T indexedResult) {
            if (resultsIndexRange.containsLong(count)) {
                pageResults.add(toViewMapper.apply(indexedResult));
            }
            count += 1;
        }

        public PageSequentialResultsAccumulator<T, V> merge(final PageSequentialResultsAccumulator<T, V> other) {
            count += other.count;
            pageResults.addAll(other.pageResults);
            return this;
        }

        public long getCount() {
            return count;
        }

        public List<V> getPageResults() {
            return pageResults;
        }
    }
}
