package ru.yandex.qe.dispenser.ws;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;

import org.apache.cxf.message.Message;
import org.apache.cxf.phase.PhaseInterceptorChain;
import org.apache.cxf.transport.http.AbstractHTTPDestination;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.dispenser.api.v1.request.quota.Body;
import ru.yandex.qe.dispenser.api.v1.request.quota.ChangeBody;
import ru.yandex.qe.dispenser.domain.Entity;
import ru.yandex.qe.dispenser.domain.EntitySpec;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.QuotaSpec;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.ws.intercept.AccessLogFilter;
import ru.yandex.qe.dispenser.ws.param.QuotaChangeRequestFilterParam;
import ru.yandex.qe.dispenser.ws.param.QuotaGetParams;

public final class ServiceEndpointUtils {

    private static final Logger LOG = LoggerFactory.getLogger(ServiceEndpointUtils.class);

    public static void memoizeServiceForQuotas(@NotNull final QuotaGetParams quotaGetParams) {
        final List<String> serviceKeys = quotaGetParams.getResources()
                .stream()
                .map(Resource::getService)
                .map(Service::getKey)
                .collect(Collectors.toList());
        memoizeServiceKey(serviceKeys);
    }

    public static void memoizeServiceForChangeRequestFilter(@NotNull final QuotaChangeRequestFilterParam filterParams) {
        final List<String> serviceKeys = filterParams.getServices().stream().map(Service::getKey).collect(Collectors.toList());
        memoizeServiceKey(serviceKeys);
    }

    public static void memoizeServiceForChangeRequest(@NotNull final QuotaChangeRequest request) {
        memoizeServiceValidateKeys(request.getChanges().stream().map(c -> c.getResource().getService().getKey()).collect(Collectors.toSet()));
    }

    public static void memoizeServiceForChangeRequest(@NotNull final Body request) {
        memoizeServiceValidateKeys(request.getChanges().stream().map(ChangeBody::getServiceKey).collect(Collectors.toSet()));
    }

    public static void memoizeServiceForChangeRequests(@NotNull final Collection<Body> requests) {
        final List<String> serviceKeys = requests.stream()
                .filter(Objects::nonNull)
                .flatMap(b -> b.getChanges().stream())
                .map(ChangeBody::getServiceKey)
                .collect(Collectors.toList());
        memoizeServiceValidateKeys(serviceKeys);
    }

    public static void memoizeServiceForEntitySpec(@NotNull final EntitySpec entitySpec) {
        memoizeServiceKey(entitySpec.getService().getKey());
    }

    public static void memoizeServiceForQuotaSpec(@NotNull final QuotaSpec quotaSpec) {
        memoizeServiceKey(quotaSpec.getResource().getService().getKey());
    }

    public static void memoizeServiceForResource(@NotNull final Resource resource) {
        memoizeServiceKey(resource.getService().getKey());
    }

    public static void memoizeServiceForEntity(@NotNull final Entity entity) {
        memoizeServiceKey(entity.getSpec().getService().getKey());
    }

    public static void memoizeService(@NotNull final Collection<Service> services) {
        if (services.isEmpty()) {
            return;
        }
        final List<String> serviceKeys = services.stream().map(Service::getKey).collect(Collectors.toList());
        memoizeServiceKey(serviceKeys);
    }

    public static void memoizeService(@NotNull final Service service) {
        memoizeServiceKey(service.getKey());
    }

    public static void memoizeServiceValidateKeys(@NotNull final Collection<String> serviceKeys) {
        if (serviceKeys.isEmpty()) {
            return;
        }
        final Set<String> uniqueServiceKeys = serviceKeys.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        if (uniqueServiceKeys.size() != 1) {
            return;
        }
        final String serviceKey = uniqueServiceKeys.iterator().next();
        memoizeServiceValidateKey(serviceKey);
    }

    public static void memoizeServiceValidateKey(@NotNull final String serviceKey) {
        final Service service;
        try {
            service = Hierarchy.get().getServiceReader().readOrNull(serviceKey);
        } catch (final Exception e) {
            LOG.warn("Failed to load service " + serviceKey, e);
            return;
        }
        if (service == null) {
            return;
        }
        memoizeServiceKey(service.getKey());
    }

    private static void memoizeServiceKey(@NotNull final List<String> serviceKeys) {
        if (serviceKeys.isEmpty()) {
            return;
        }
        final Set<String> uniqueServiceKeys = serviceKeys.stream().filter(Objects::nonNull).collect(Collectors.toSet());
        if (uniqueServiceKeys.size() != 1) {
            return;
        }
        final String serviceKey = uniqueServiceKeys.iterator().next();
        memoizeServiceKey(serviceKey);
    }

    private static void memoizeServiceKey(@NotNull final String serviceKey) {
        final Message currentMessage = PhaseInterceptorChain.getCurrentMessage();
        if (currentMessage == null) {
            return;
        }
        final Object httpRequest = currentMessage.getContextualProperty(AbstractHTTPDestination.HTTP_REQUEST);
        if (httpRequest instanceof HttpServletRequest) {
            ((HttpServletRequest) httpRequest).setAttribute(AccessLogFilter.ENDPOINT_SERVICE, serviceKey);
        }
    }

    private ServiceEndpointUtils() {
    }

}
