package ru.yandex.qe.dispenser.ws;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;

import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;

import ru.yandex.qe.dispenser.api.v1.DiResource;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignResourceDao;
import ru.yandex.qe.dispenser.domain.dao.resource.ResourceDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.util.ValidationUtils;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.param.ResourceParam;
import ru.yandex.qe.dispenser.ws.reqbody.ResourceBody;

import static ru.yandex.qe.dispenser.ws.ServiceService.SERVICE_KEY;

@Path("/v1/services/{" + SERVICE_KEY + "}/resources")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("resource")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class ServiceResourceService extends ServiceBase {

    public static final String RESOURCE_KEY = "resource_key";
    public static final String GROUP_PARAM = "group";

    @Autowired
    private ResourceDao resourceDao;
    @Autowired
    private CampaignResourceDao campaignResourceDao;

    @GET
    public DiListResponse<DiResource> getResources(@PathParam(SERVICE_KEY) final @NotNull Service service,
                                                   @QueryParam(GROUP_PARAM) final List<String> groups) {
        ServiceEndpointUtils.memoizeService(service);
        Stream<Resource> resourcesByServiceStream = Hierarchy.get().getResourceReader().getByService(service).stream();
        if (!groups.isEmpty()) {
            resourcesByServiceStream = resourcesByServiceStream
                    .filter(resource -> resource.getGroup() != null)
                    .filter(resource -> groups.contains(resource.getGroup().getKey().getPublicKey()));
        }
        final List<DiResource> resources = resourcesByServiceStream
                .map(Resource::toView)
                .collect(Collectors.toList());
        return new DiListResponse<>(resources);
    }

    /*
      curl -X PUT -H "Content-Type: application/json" -d '{"name": "Yt Storage"}' "http://localhost:8082/api/v0/service/nirvana/yt_storage" -H "Authorization: sancho" -i | less
     */
    @PUT
    @Access(serviceAdmin = true)
    @Consumes(MediaType.APPLICATION_JSON)
    @Path("/{" + RESOURCE_KEY + "}")
    public DiResource createResource(@PathParam(SERVICE_KEY) final @NotNull Service service,
                                     @PathParam(RESOURCE_KEY) @NotNull final String resourceKey,
                                     @RequestBody @NotNull final ResourceBody body) {
        ServiceEndpointUtils.memoizeService(service);
        final Resource resource = new Resource.Builder(resourceKey, service)
                .name(body.getName())
                .description(body.getDescription())
                .type(body.getType())
                .mode(body.getQuotingMode())
                .group(body.getGroup())
                .priority(body.getPriority())
                .build();
        return resourceDao.create(resource).toView();
    }

    @GET
    @Path("/{" + RESOURCE_KEY + "}")
    public DiResource getResource(@PathParam(SERVICE_KEY) @NotNull final Service service,
                                  @PathParam(RESOURCE_KEY) @NotNull final String resourceKey) {
        final ResourceParam resourceParam = new ResourceParam(service, resourceKey);
        final Resource resource = resourceParam.get();
        ServiceEndpointUtils.memoizeServiceForResource(resource);
        return resource.toView();
    }

    @POST
    @Access(serviceAdmin = true)
    @Consumes(MediaType.APPLICATION_JSON)
    @Path("/{" + RESOURCE_KEY + "}")
    public DiResource updateResource(@PathParam(SERVICE_KEY) @NotNull final Service service,
                                     @PathParam(RESOURCE_KEY) @NotNull final String resourceKey,
                                     @RequestBody @NotNull final ResourceBody body) {
        final ResourceParam resourceParam = new ResourceParam(service, resourceKey);
        final Resource resource = resourceParam.get();
        ServiceEndpointUtils.memoizeServiceForResource(resource);

        ValidationUtils.requireEquals(resource.getType(), body.getType(), "Type can't be changed");
        ValidationUtils.requireEquals(resource.getMode(), body.getQuotingMode(), "Mode can't be changed");

        final Resource updatedResource = Resource.copyOf(resource)
                .name(body.getName())
                .description(body.getDescription())
                .group(body.getGroup())
                .priority(body.getPriority())
                .build();

        resourceDao.update(updatedResource);

        return updatedResource.toView();
    }

    /*
      curl -X DELETE "http://localhost:8082/api/v0/service/nirvana/yt_storage" -H ""Authorization: sancho" -i | less
     */
    @DELETE
    @Access(serviceAdmin = true)
    @Path("/{" + RESOURCE_KEY + "}")
    public DiResource deleteResource(@PathParam(SERVICE_KEY) @NotNull final Service service,
                                     @PathParam(RESOURCE_KEY) @NotNull final String resourceKey) {
        final ResourceParam resourceParam = new ResourceParam(service, resourceKey);
        final Resource resource = resourceParam.get();
        ServiceEndpointUtils.memoizeServiceForResource(resource);
        if (campaignResourceDao.existsByResource(resource)) {
            throw new IllegalArgumentException("This resource is referenced by one or more campaign settings.");
        }
        resourceDao.delete(resource);
        return resource.toView();
    }
}

