package ru.yandex.qe.dispenser.ws;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.ws.rs.Consumes;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.google.common.collect.Sets;
import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.RequestBody;

import ru.yandex.qe.dispenser.api.v1.DiQuotaMaxDeltaUpdate;
import ru.yandex.qe.dispenser.api.v1.DiQuotaMaxUpdate;
import ru.yandex.qe.dispenser.api.v1.DiQuotaMaxUpdateRequest;
import ru.yandex.qe.dispenser.api.v1.DiQuotingMode;
import ru.yandex.qe.dispenser.api.v1.DiService;
import ru.yandex.qe.dispenser.api.v1.request.DiActualQuotaUpdate;
import ru.yandex.qe.dispenser.api.v1.request.DiQuotaState;
import ru.yandex.qe.dispenser.api.v1.response.DiListResponse;
import ru.yandex.qe.dispenser.api.v1.response.DiQuotaGetResponse;
import ru.yandex.qe.dispenser.api.v1.response.DiResponse;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaMaxUpdate;
import ru.yandex.qe.dispenser.domain.QuotaSpec;
import ru.yandex.qe.dispenser.domain.QuotaView;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignResourceDao;
import ru.yandex.qe.dispenser.domain.dao.person.PersonDao;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaDao;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaUtils;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentUtils;
import ru.yandex.qe.dispenser.domain.dao.service.ServiceDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.domain.support.QuotaState;
import ru.yandex.qe.dispenser.domain.util.MathUtils;
import ru.yandex.qe.dispenser.domain.util.PropertyUtils;
import ru.yandex.qe.dispenser.domain.util.ValidationUtils;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.aspect.AccessAspect;
import ru.yandex.qe.dispenser.ws.aspect.ForbiddenException;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.PerformerContext;
import ru.yandex.qe.dispenser.ws.quota.update.QuotaUpdateManager;
import ru.yandex.qe.dispenser.ws.reqbody.ServiceBody;

import static java.util.stream.Collectors.toSet;

@Path("/v1/services")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@org.springframework.stereotype.Service("service")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class ServiceService extends ServiceBase {
    public static final String SERVICE_KEY = "service_key";

    @Autowired
    private ServiceDao serviceDao;

    @Autowired
    private PersonDao personDao;

    @Autowired
    private QuotaDao quotaDao;

    @Autowired
    private QuotaUpdateManager quotaUpdateManager;

    @Autowired
    private CampaignResourceDao campaignResourceDao;

    @GET
    @Path("/all")
    public DiListResponse<DiService> getAllServices() {
        final List<DiService> services = Hierarchy.get()
                .getServiceReader()
                .getAll()
                .stream()
                .map(Service::toVerboseView)
                .collect(Collectors.toList());
        return new DiListResponse<>(services);
    }


    @GET
    @Path("/{" + SERVICE_KEY + "}")
    public DiService getService(@PathParam(SERVICE_KEY) @NotNull final Service service) {
        ServiceEndpointUtils.memoizeService(service);
        return service.toVerboseView();
    }

    /*
      curl -X PUT -H "Content-Type: application/json" -d '{"name": "Metrics"}' "http://localhost:8082/api/v0/service/metrics" -i | less
     */
    @PUT
    @Access(dispenserAdmin = true)
    @Path("/{" + SERVICE_KEY + "}")
    @Consumes(MediaType.APPLICATION_JSON)
    public DiService registerService(@PathParam(SERVICE_KEY) @NotNull final String key,
                                     @RequestBody @NotNull final ServiceBody body) {
        ValidationUtils.requireNonNull(body.getAbcServiceId(), "ABC service ID is required");
        ValidationUtils.requireNonNull(body.getAdmins(), "Admins field is required");
        ValidationUtils.requireNonNull(body.getTrustees(), "Trustees field is required");

        final Service createdService = serviceDao.create(Service.withKey(key)
                .withName(body.getName())
                .withAbcServiceId(body.getAbcServiceId())
                .withPriority(body.getPriority())
                .build());
        serviceDao.attachAdmins(createdService, personDao.readPersonsByLogins(body.getAdmins()));
        serviceDao.attachTrustees(createdService, personDao.readPersonsByLogins(body.getTrustees()));
        ServiceEndpointUtils.memoizeService(createdService);
        return createdService.toVerboseView();
    }

    /*
      curl -X DELETE "http://localhost:8082/api/v0/service/metrics" -i | less
     */
    @DELETE
    @Access(dispenserAdmin = true)
    @Path("/{" + SERVICE_KEY + "}")
    public boolean deleteService(@PathParam(SERVICE_KEY) final @NotNull Service service) {
        ServiceEndpointUtils.memoizeService(service);
        if (campaignResourceDao.existsByService(service)) {
            throw new IllegalArgumentException("This service is referenced by one or more campaign settings.");
        }
        return serviceDao.delete(service);
    }

    @POST
    @Access(dispenserAdmin = true)
    @Path("/{" + SERVICE_KEY + "}")
    @Consumes(MediaType.APPLICATION_JSON)
    public DiService updateService(@PathParam(SERVICE_KEY) @NotNull final String key,
                                   @RequestBody @NotNull final ServiceBody body) {
        final Service service = serviceDao.read(key);
        ServiceEndpointUtils.memoizeService(service);

        if (service.getAbcServiceId() != null) {
            ValidationUtils.requireNonNull(body.getAbcServiceId(), "ABC service ID is required");
        }
        ValidationUtils.requireNonNull(body.getAdmins(), "Admins field is required");
        ValidationUtils.requireNonNull(body.getTrustees(), "Trustees field is required");

        final Service updatedService = Service.copyOf(service)
                .withName(body.getName())
                .withAbcServiceId(body.getAbcServiceId())
                .withPriority(body.getPriority())
                .build();

        serviceDao.detachAllAdmins(updatedService);
        serviceDao.attachAdmins(updatedService, personDao.readPersonsByLogins(body.getAdmins()));

        serviceDao.detachAllTrustees(updatedService);
        serviceDao.attachTrustees(updatedService, personDao.readPersonsByLogins(body.getTrustees()));


        serviceDao.update(updatedService);

        return serviceDao.read(key).toVerboseView();
    }

    /*
      curl -X POST -H "Content-Type: application/json" -d '["sancho"]' "http://localhost:8082/api/v0/service/nirvana/attach-admins" -H "Authorization: whistler" -i | less
     */
    @POST
    @Access(serviceAdmin = true)
    @Consumes(MediaType.APPLICATION_JSON)
    @Path("/{" + SERVICE_KEY + "}/attach-admins")
    public DiService attachAdmins(@PathParam(SERVICE_KEY) final @NotNull Service service,
                                  @RequestBody @NotNull final List<Person> admins) {
        ServiceEndpointUtils.memoizeService(service);
        serviceDao.attachAdmins(service, admins);
        return serviceDao.read(service.getKey()).toView(serviceDao, true);
    }

    /*
      curl -X POST -H "Content-Type: application/json" -d '["sancho"]' "http://localhost:8082/api/v0/service/nirvana/detach-admins" -H "Authorization: whistler" -i | less
     */
    @POST
    @Access(serviceAdmin = true)
    @Consumes(MediaType.APPLICATION_JSON)
    @Path("/{" + SERVICE_KEY + "}/detach-admins")
    public DiService detachAdmins(@PathParam(SERVICE_KEY) final @NotNull Service service,
                                  @RequestBody @NotNull final List<Person> admins) {
        ServiceEndpointUtils.memoizeService(service);
        serviceDao.detachAdmins(service, admins);
        return serviceDao.read(service.getKey()).toView(serviceDao, true);
    }

    @POST
    @Access(serviceAdmin = true)
    @Consumes(MediaType.APPLICATION_JSON)
    @Path("/{" + SERVICE_KEY + "}/attach-trustees")
    public DiService attachTrustees(@PathParam(SERVICE_KEY) final @NotNull Service service,
                                    @RequestBody @NotNull final List<Person> trustees) {
        ServiceEndpointUtils.memoizeService(service);
        serviceDao.attachTrustees(service, trustees);
        return serviceDao.read(service.getKey()).toView(serviceDao, true);
    }

    @POST
    @Access(serviceAdmin = true)
    @Consumes(MediaType.APPLICATION_JSON)
    @Path("/{" + SERVICE_KEY + "}/detach-trustees")
    public DiService detachTrustees(@PathParam(SERVICE_KEY) final @NotNull Service service,
                                    @RequestBody @NotNull final List<Person> trustees) {
        ServiceEndpointUtils.memoizeService(service);
        serviceDao.detachTrustees(service, trustees);
        return serviceDao.read(service.getKey()).toView(serviceDao, true);
    }

    @POST
    @Access
    @Path("/{" + SERVICE_KEY + "}/sync-state/quotas/set")
    @Consumes(MediaType.APPLICATION_JSON)
    public DiQuotaGetResponse syncQuotas(@PathParam(SERVICE_KEY) @NotNull final Service service,
                                         @RequestBody @NotNull final List<DiQuotaState> quotaStates) {
        ServiceEndpointUtils.memoizeService(service);
        final Person performer = Session.WHOAMI.get();

        final Quota.ChangeHolder changeHolder = new Quota.ChangeHolder();

        for (final DiQuotaState diState : quotaStates) {
            final QuotaState state = QuotaState.from(diState, service);

            if (state.getMax() != null) {
                if (state.getQuotaKey().getProject().isRemoved()) {
                    throw new IllegalArgumentException("Quota max value can't be updated if project is removed!");
                }
                checkCanUpdateMax(performer, state.getQuotaKey());
                checkCanChangeQuotaMaxWithOldStyleMethod(state.getQuotaKey());
                changeHolder.setMax(state.getQuotaKey(), state.getMax());
            }
            if (state.getOwnMax() != null) {
                if (state.getQuotaKey().getProject().isRemoved()) {
                    throw new IllegalArgumentException("Quota own max value can't be updated if project is removed!");
                }
                checkCanUpdateOwnMax(performer, state.getQuotaKey());
                checkCanChangeQuotaMaxWithOldStyleMethod(state.getQuotaKey());
                changeHolder.setOwnMax(state.getQuotaKey(), state.getOwnMax());
            }

            if (state.getActual() != null) {
                checkCanUpdateActual(performer, state);
                changeHolder.setActual(state.getQuotaKey(), state.getActual());
            }
        }

        final PerformerContext ctx = new PerformerContext(performer);
        final Collection<QuotaView> modifiedQuotasByIds = quotaUpdateManager.updateQuota(changeHolder, ctx, null);

        return new DiQuotaGetResponse(modifiedQuotasByIds.stream().map(QuotaUtils::toView).collect(Collectors.toList()));
    }

    @POST
    @Access
    @Path("/{" + SERVICE_KEY + "}/update-max/quotas")
    @Consumes(MediaType.APPLICATION_JSON)
    public DiQuotaGetResponse changeMax(@PathParam(SERVICE_KEY) @NotNull final Service service,
                                        @RequestBody @NotNull final DiQuotaMaxUpdateRequest<DiQuotaMaxUpdate> maxUpdateRequest) {
        ServiceEndpointUtils.memoizeService(service);
        final Person performer = Session.WHOAMI.get();
        final Quota.ChangeHolder changeHolder = new Quota.ChangeHolder();

        for (final DiQuotaMaxUpdate update : maxUpdateRequest.getUpdates()) {
            final QuotaMaxUpdate maxUpdate = QuotaMaxUpdate.from(update, service);

            if (maxUpdate.getQuotaKey().getProject().isRemoved()) {
                throw new IllegalArgumentException("Quota max value can't be updated if project is removed!");
            }
            if (maxUpdate.getMax() != null) {
                checkCanUpdateMax(performer, maxUpdate.getQuotaKey());
                changeHolder.setMax(maxUpdate.getQuotaKey(), maxUpdate.getMax());
            }
            if (maxUpdate.getOwnMax() != null) {
                checkCanUpdateOwnMax(performer, maxUpdate.getQuotaKey());
                changeHolder.setOwnMax(maxUpdate.getQuotaKey(), maxUpdate.getOwnMax());
            }
        }

        final PerformerContext ctx = new PerformerContext(performer);
        ctx.setComment(maxUpdateRequest.getDescription());
        final Collection<QuotaView> modifiedQuotasByIds = quotaUpdateManager.updateQuota(changeHolder, ctx,
                maxUpdateRequest.getTicketKey());

        return new DiQuotaGetResponse(modifiedQuotasByIds.stream().map(QuotaUtils::toView).collect(Collectors.toList()));
    }

    @POST
    @Access
    @Idempotent
    @Path("/{" + SERVICE_KEY + "}/update-max/deltas")
    @Consumes(MediaType.APPLICATION_JSON)
    public DiQuotaGetResponse changeMaxByDeltas(@PathParam(SERVICE_KEY) @NotNull final Service service,
                                                @RequestBody @NotNull final DiQuotaMaxUpdateRequest<DiQuotaMaxDeltaUpdate> maxUpdateRequest) {
        ServiceEndpointUtils.memoizeService(service);
        final Person performer = Session.WHOAMI.get();
        final Map<Quota.Key, Long> maxes = new HashMap<>();
        final Map<Quota.Key, Long> ownMaxes = new HashMap<>();

        for (final DiQuotaMaxDeltaUpdate update : maxUpdateRequest.getUpdates()) {
            final QuotaMaxUpdate maxUpdate = QuotaMaxUpdate.from(update, service);

            if (maxUpdate.getQuotaKey().getProject().isRemoved()) {
                throw new IllegalArgumentException("Quota max value can't be updated if project is removed!");
            }
            if (maxUpdate.getMax() != null) {
                checkCanUpdateMax(performer, maxUpdate.getQuotaKey());
                MathUtils.increment(maxes, maxUpdate.getQuotaKey(), maxUpdate.getMax());
            }
            if (maxUpdate.getOwnMax() != null) {
                checkCanUpdateOwnMax(performer, maxUpdate.getQuotaKey());
                MathUtils.increment(ownMaxes, maxUpdate.getQuotaKey(), maxUpdate.getOwnMax());
            }
        }

        final Sets.SetView<Quota.Key> changedQuotaKeys = Sets.union(maxes.keySet(), ownMaxes.keySet());
        final Set<Quota> quotasForUpdate = quotaDao.getQuotasForUpdate(changedQuotaKeys);

        for (final Quota quota : quotasForUpdate) {
            maxes.computeIfPresent(quota.getKey(), (quotaKey, quotaDelta) -> quota.getMax() + quotaDelta);
            ownMaxes.computeIfPresent(quota.getKey(), (quotaKey, quotaDelta) -> quota.getOwnMax() + quotaDelta);
        }

        maxes.forEach(ServiceService::checkValueIsPositive);
        ownMaxes.forEach(ServiceService::checkValueIsPositive);

        final Quota.ChangeHolder changeHolder = new Quota.ChangeHolder();
        changeHolder.setMaxes(maxes);
        changeHolder.setOwnMaxes(ownMaxes);

        final PerformerContext ctx = new PerformerContext(performer);
        ctx.setComment(maxUpdateRequest.getDescription());
        final Collection<QuotaView> modifiedQuotasByIds = quotaUpdateManager.updateQuota(changeHolder, ctx, maxUpdateRequest.getTicketKey());

        return new DiQuotaGetResponse(modifiedQuotasByIds.stream().map(QuotaUtils::toView).collect(Collectors.toList()));
    }

    @POST
    @Access
    @Path("/{" + SERVICE_KEY + "}/sync-raw-state/quotas/set")
    @Consumes(MediaType.APPLICATION_JSON)
    public DiResponse syncRawQuotas(@PathParam(SERVICE_KEY) @NotNull final Service service,
                                    @RequestBody @NotNull final List<DiQuotaState> quotaStates) {
        ServiceEndpointUtils.memoizeService(service);
        final Person performer = Session.WHOAMI.get();
        validateRequiredRawFields(quotaStates);
        final Map<Quota.Key, Long> max = new HashMap<>();
        final Map<Quota.Key, Long> ownActual = new HashMap<>();
        final Map<Quota.Key, Long> ownMax = new HashMap<>();
        validateRawUpdates(service, quotaStates, performer, max, ownActual, ownMax);
        quotaDao.applyChanges(max, ownActual, ownMax);
        return new DiResponse(Response.Status.OK.getStatusCode());
    }

    @POST
    @Access
    @Path("/{" + SERVICE_KEY + "}/sync-state/actual-quotas/set")
    @Consumes(MediaType.APPLICATION_JSON)
    public DiResponse syncActualQuotas(@PathParam(SERVICE_KEY) @NotNull final Service service,
                                       @RequestBody @NotNull final List<DiActualQuotaUpdate> updates) {
        ServiceEndpointUtils.memoizeService(service);
        final Person performer = Session.WHOAMI.get();
        validateRequiredFields(updates);
        final Map<Quota.Key, Long> ownActual = validateUpdates(service, updates, performer);
        quotaDao.applyChanges(Collections.emptyMap(), ownActual, Collections.emptyMap());
        return new DiResponse(Response.Status.OK.getStatusCode());
    }

    @NotNull
    private Map<Quota.Key, Long> validateUpdates(@NotNull final Service service, @NotNull final List<DiActualQuotaUpdate> updates,
                                                 @NotNull final Person performer) {
        final Map<Quota.Key, Long> result = new HashMap<>();
        final Set<String> validationErrors = new HashSet<>();
        for (final DiActualQuotaUpdate update : updates) {
            final Resource resource = Hierarchy.get().getResourceReader().read(new Resource.Key(update.getResourceKey(), service));
            final Project project = Hierarchy.get().getProjectReader().read(update.getProjectKey());
            final QuotaSpec quotaSpec = Hierarchy.get().getQuotaSpecReader().read(resource, update.getQuotaSpecKey());
            final Set<Segment> segments = SegmentUtils.getCompleteSegmentSet(resource, update.getSegments() != null
                    ? update.getSegments() : new HashSet<>());
            final Quota.Key key = new Quota.Key(quotaSpec, project, segments);
            final long actual = resource.getType().getBaseUnit().convert(update.getActual());
            if (actual < 0) {
                validationErrors.add("Quota actual values can not be negative.");
            }
            AccessAspect.checkServiceTrustee(performer, key.getSpec().getResource().getService());
            if (key.getSpec().getResource().getMode() != DiQuotingMode.SYNCHRONIZATION) {
                validationErrors.add("Quota actual values can be synced only for resources with synchronization quoting mode.");
            }
            if (key.isAggregation()) {
                validationErrors.add("Quota actual values can not be synced for quotas with aggregation segments.");
            }
            result.put(key, actual);
        }
        if (!validationErrors.isEmpty()) {
            throw new IllegalArgumentException("Invalid request. " + String.join(" ", validationErrors));
        }
        return result;
    }

    private void validateRequiredFields(@NotNull final List<DiActualQuotaUpdate> updates) {
        final Set<String> validationErrors = new HashSet<>();
        for (final DiActualQuotaUpdate update : updates) {
            if (update == null) {
                validationErrors.add("Quota update may not be null.");
                continue;
            }
            if (update.getProjectKey() == null) {
                validationErrors.add("Project key is required.");
            }
            if (update.getResourceKey() == null) {
                validationErrors.add("Resource key is required.");
            }
            if (update.getActual() == null) {
                validationErrors.add("Actual quota value is required.");
            }
            if (update.getSegments() != null && update.getSegments().stream().anyMatch(Objects::isNull)) {
                validationErrors.add("Segment keys may not be null.");
            }
        }
        if (!validationErrors.isEmpty()) {
            throw new IllegalArgumentException("Invalid request. " + String.join(" ", validationErrors));
        }
    }

    @NotNull
    private void validateRawUpdates(@NotNull final Service service, @NotNull final List<DiQuotaState> quotaStates,
                                    @NotNull final Person performer, @NotNull final Map<Quota.Key, Long> max,
                                    @NotNull final Map<Quota.Key, Long> ownActual, @NotNull final Map<Quota.Key, Long> ownMax) {
        if (!PropertyUtils.readBoolean(RAW_QUOTA_IMPORT_ENTITY, ENDPOINT_ENABLED_PROPERTY, false)) {
            throw new ForbiddenException("Raw quota import is forbidden");
        }
        final Set<String> validationErrors = new HashSet<>();
        for (final DiQuotaState quotaState : quotaStates) {
            final Resource resource = Hierarchy.get().getResourceReader().read(new Resource.Key(quotaState.getResourceKey(), service));
            final Project project = Hierarchy.get().getProjectReader().read(quotaState.getProjectKey());
            final QuotaSpec quotaSpec = Hierarchy.get().getQuotaSpecReader().read(resource, quotaState.getQuotaSpecKey());
            final Set<Segment> segments = SegmentUtils.getCompleteSegmentSet(resource, quotaState.getSegmentKeys() != null
                    ? quotaState.getSegmentKeys() : new HashSet<>());
            final Quota.Key key = new Quota.Key(quotaSpec, project, segments);
            if (quotaState.getMax() != null) {
                final long maxValue = resource.getType().getBaseUnit().convert(quotaState.getMax());
                if (maxValue < 0) {
                    validationErrors.add("Quota max values can not be negative.");
                }
                if (project.isRemoved()) {
                    validationErrors.add("Quota max values can not be updated if project is removed!");
                }
                if (!service.getSettings().usesProjectHierarchy()) {
                    validationErrors.add("Can not change quota max values for service that does not use project hierarchy");
                }
                max.put(key, maxValue);
            }
            if (quotaState.getActual() != null) {
                final long actualValue = resource.getType().getBaseUnit().convert(quotaState.getActual());
                if (actualValue < 0) {
                    validationErrors.add("Quota actual values can not be negative.");
                }
                if (key.getSpec().getResource().getMode() != DiQuotingMode.SYNCHRONIZATION) {
                    validationErrors.add("Quota actual values can be synced only for resources with synchronization quoting mode.");
                }
                if (key.isAggregation()) {
                    validationErrors.add("Quota actual values can not be synced for quotas with aggregation segments.");
                }
                ownActual.put(key, actualValue);
            }
            if (quotaState.getOwnMax() != null) {
                final long ownMaxValue = resource.getType().getBaseUnit().convert(quotaState.getOwnMax());
                if (ownMaxValue < 0) {
                    validationErrors.add("Quota own max values can not be negative.");
                }
                if (project.isRemoved()) {
                    validationErrors.add("Quota own max values can not be updated if project is removed!");
                }
                ownMax.put(key, ownMaxValue);
            }
            AccessAspect.checkServiceTrustee(performer, key.getSpec().getResource().getService());
        }
        if (!validationErrors.isEmpty()) {
            throw new IllegalArgumentException("Invalid request. " + String.join(" ", validationErrors));
        }
    }

    private void validateRequiredRawFields(@NotNull final List<DiQuotaState> quotaStates) {
        final Set<String> validationErrors = new HashSet<>();
        for (final DiQuotaState quotaState : quotaStates) {
            if (quotaState == null) {
                validationErrors.add("Quota state may not be null.");
                continue;
            }
            if (quotaState.getProjectKey() == null) {
                validationErrors.add("Project key is required.");
            }
            if (quotaState.getResourceKey() == null) {
                validationErrors.add("Resource key is required.");
            }
            if (quotaState.getActual() == null && quotaState.getMax() == null && quotaState.getOwnMax() == null) {
                validationErrors.add("At least one of actual, max, own max quota value is required.");
            }
            if (quotaState.getSegmentKeys() != null && quotaState.getSegmentKeys().stream().anyMatch(Objects::isNull)) {
                validationErrors.add("Segment keys may not be null.");
            }
        }
        if (!validationErrors.isEmpty()) {
            throw new IllegalArgumentException("Invalid request. " + String.join(" ", validationErrors));
        }
    }

    private static void checkValueIsPositive(final Quota.Key key, final long value) {
        if (value < 0) {
            throw new IllegalArgumentException("Result value for quota '" + key + "' is negative: '" + value + "'");
        }
    }

    public static final String OLD_STYLE_QUOTA_MAX_CHANGE_ENTITY = "oldStyleQuotaMaxChange";
    public static final String FORBIDDEN_SERVICES_PROPERTY = "forbiddenServices";
    public static final String RAW_QUOTA_IMPORT_ENTITY = "rawQuotaImport";
    public static final String ENDPOINT_ENABLED_PROPERTY = "forbiddenServices";

    public static void checkCanChangeQuotaMaxWithOldStyleMethod(final Quota.Key quotaKey) {
        PropertyUtils.readStringOptional(OLD_STYLE_QUOTA_MAX_CHANGE_ENTITY, FORBIDDEN_SERVICES_PROPERTY)
                .ifPresent(forbiddenServicesString ->
                        {
                            final Set<String> services = Arrays.stream(StringUtils.split(forbiddenServicesString, ",")).collect(toSet());
                            final Service quotaService = quotaKey.getSpec().getResource().getService();
                            if (services.contains(quotaService.getKey())) {
                                throw new IllegalArgumentException("Max values of '" + quotaService.getName() + "' quotas can't be changed using this API method");
                            }
                        }
                );
    }

    public static void checkCanUpdateMax(@NotNull final Person person, @NotNull final Quota.Key quotaKey) {
        if (quotaKey.getProject().isRoot()) {
            AccessAspect.checkServiceAdmin(person, quotaKey.getSpec().getResource().getService());
        } else {
            AccessAspect.checkServiceAdminOrProjectResponsible(person, quotaKey.getSpec().getResource().getService(),
                    quotaKey.getProject().getParent());
        }

        final Service service = quotaKey.getSpec().getResource().getService();
        final Service.Settings serviceSettings = service.getSettings();
        if (!serviceSettings.usesProjectHierarchy()) {
            throw new IllegalArgumentException("Can't change quota max for service that doesn't use project hierarchy: " + service.getKey());
        }
    }

    private void checkCanUpdateOwnMax(@NotNull final Person person, @NotNull final Quota.Key quotaKey) {
        AccessAspect.checkServiceAdminOrProjectResponsible(person, quotaKey.getSpec().getResource().getService(), quotaKey.getProject());
    }

    private void checkCanUpdateActual(@NotNull final Person person, @NotNull final QuotaState state) {
        final Quota.Key quotaKey = state.getQuotaKey();
        AccessAspect.checkServiceTrustee(person, quotaKey.getSpec().getResource().getService());
        if (quotaKey.getSpec().getResource().getMode() != DiQuotingMode.SYNCHRONIZATION) {
            throw new IllegalArgumentException("Quota actual values can be synced only for resources with quoting mode " +
                    DiQuotingMode.SYNCHRONIZATION);
        }
        if (state.isAggregation() && state.getActual() != null && state.getActual() > 0) {
            throw new IllegalArgumentException("Quota actual values cannot be synced for quotas with aggregation segments");
        }
    }
}
