package ru.yandex.qe.dispenser.ws;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.inject.Inject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.GaugeInt64;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.qe.dispenser.domain.QuotaView;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.solomon.SolomonHolder;

@Component("servicesOverCommitValueMetrics")
@Profile("secondary")
public class ServicesOverCommitValueMetrics {
    private static final Logger LOG = LoggerFactory.getLogger(ServicesOverCommitValueMetrics.class);

    private final HierarchySupplier hierarchySupplier;
    private final MetricRegistry metricRegistry;
    private final ConcurrentMap<Resource.Key, GaugeInt64> metricsByResource = new ConcurrentHashMap<>();

    @Inject
    public ServicesOverCommitValueMetrics(final HierarchySupplier hierarchySupplier,
                                          final SolomonHolder solomonHolder) {
        this.hierarchySupplier = hierarchySupplier;
        this.metricRegistry = solomonHolder.getRootRegistry();
    }

    public void countOverCommit() {
        LOG.info("Counting over commit...");
        hierarchySupplier.renewThreadHierarchy();

        final Set<Resource.Key> resourceKeys = new HashSet<>();
        final Map<Resource.Key, Long> overCommitByResourceKey = new HashMap<>();

        for (final QuotaView quota : Hierarchy.get().getQuotaCache().getAll()) {
            resourceKeys.add(quota.getResource().getKey());

            if (quota.getOwnActual() > quota.getOwnMax()) {
                final Resource.Key resourceKey = quota.getResource().getKey();
                final Long previous = overCommitByResourceKey.getOrDefault(resourceKey, 0L);
                overCommitByResourceKey.put(resourceKey, previous + quota.getOwnActual() - quota.getOwnMax());
            }
        }

        for (final Resource.Key key : resourceKeys) {
            GaugeInt64 gauge = metricsByResource.computeIfAbsent(key,
                    k -> metricRegistry.gaugeInt64("quotas.over_commit",
                            Labels.of("provider_key", k.getService().getKey(), "resource_key", k.getPublicKey())));

            final long currentOverCommitValue = overCommitByResourceKey.getOrDefault(key, 0L);
            final long oldOverCommitValue = gauge.get();
            final long diff = currentOverCommitValue - oldOverCommitValue;
            gauge.add(diff);
        }
        LOG.info("Successfully counted over commit");
    }
}
