package ru.yandex.qe.dispenser.ws;

import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Optional;
import java.util.concurrent.Callable;

public class TvmIdentity {

    private final String serviceTicket;
    private final String userTicket;

    private static final ThreadLocal<Deque<TvmIdentity>> stack = ThreadLocal.withInitial(ArrayDeque::new);

    private TvmIdentity(final TvmIdentity other) {
        this(other.serviceTicket, other.userTicket);
    }

    public TvmIdentity(final String serviceTicket, final String userTicket) {
        this.serviceTicket = serviceTicket;
        this.userTicket = userTicket;
    }

    public TvmIdentity(final String serviceTicket) {
        this.serviceTicket = serviceTicket;
        this.userTicket = null;
    }

    public <T> T callWith(final Callable<T> supplier) throws Exception {
        enter();
        try {
            return supplier.call();
        } finally {
            exit();
        }
    }

    public void runWith(final Runnable runnable) {
        enter();
        try {
            runnable.run();
        } finally {
            exit();
        }
    }

    public Optional<String> getServiceTicket() {
        return Optional.ofNullable(serviceTicket);
    }

    public Optional<String> getUserTicket() {
        return Optional.ofNullable(userTicket);
    }

    public static TvmIdentity get() {
        final TvmIdentity current = stack.get().peek();
        if (current == null) {
            return new TvmIdentity(null, null);
        } else {
            return current;
        }
    }

    public static void reset() {
        stack.get().clear();
    }

    private static void exit() {
        stack.get().pop();
    }

    private void enter() {
        stack.get().push(new TvmIdentity(this));
    }

}
