package ru.yandex.qe.dispenser.ws.abc;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Inject;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.qe.bus.Robot;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.abc.AbcApi;
import ru.yandex.qe.dispenser.domain.abc.AbcPerson;
import ru.yandex.qe.dispenser.domain.abc.AbcRole;
import ru.yandex.qe.dispenser.domain.abc.AbcService;
import ru.yandex.qe.dispenser.domain.abc.AbcServiceGradient;
import ru.yandex.qe.dispenser.domain.abc.AbcServiceMember;
import ru.yandex.qe.dispenser.domain.abc.AbcServicePersonHolder;
import ru.yandex.qe.dispenser.domain.abc.AbcServiceReference;
import ru.yandex.qe.dispenser.domain.abc.AbcServiceState;
import ru.yandex.qe.dispenser.ws.TvmDestination;

@Component
@ParametersAreNonnullByDefault
public class AbcApiHelper {

    public static final String MISSING_NESTED_ABC_SERVICES_ERROR_MESSAGE = "New project must include existing nested abc services: ";
    public static final String NON_SPECIFIC_PARENT_ABC_SERVICE_ERROR_MESSAGE = "Tree has more specific parent abc service: ";
    public static final String INVALID_DESCENDANTS_ABC_SERVICES_ERROR_MESSAGE = "Invalid descendant abc services: ";
    public static final String INVALID_ANCESTOR_ERROR_MESSAGE = "Invalid ancestor abc service: ";
    public static final String INVALID_ABC_SERVICE_ID_ERROR_MESSAGE = "Invalid ABC service id";
    public static final int SERVICE_COUNT_IN_COLLECT_PERSON_REQUEST = 200;

    @Nonnull
    private final AbcApi abcApi;
    @Nonnull
    private final Robot robot;
    @Nonnull
    private final TvmDestination abcTvmDestination;
    private final boolean skipProjectAbcServiceValidation;

    @Inject
    public AbcApiHelper(final AbcApi abcApi,
                        @Qualifier("abc-robot") final Robot robot,
                        @Qualifier("abc-tvm") final TvmDestination abcTvmDestination,
                        @Value("${dispenser.skip.project.abc.service.validation}") final boolean skipProjectAbcServiceValidation) {
        this.abcApi = abcApi;
        this.robot = robot;
        this.abcTvmDestination = abcTvmDestination;
        this.skipProjectAbcServiceValidation = skipProjectAbcServiceValidation;
    }

    @Nonnull
    public Stream<AbcService> getServicesWithAncestors(final Collection<Integer> ids) {
        return new ServiceRequestBuilder(abcApi, abcTvmDestination)
                .ids(ids)
                .fields("id", "ancestors")
                .pageSize(Math.min(AbcApi.MAX_PAGE_SIZE, ids.size()))
                .stream();
    }

    @Nonnull
    public Stream<AbcService> getServicesByParentWithDescendants(final Integer id) {
        return new ServiceRequestBuilder(abcApi, abcTvmDestination)
                .fields("id", "ancestors")
                .parentWithDescendants(id)
                .pageSize(AbcApi.MAX_PAGE_SIZE)
                .stream();
    }

    @Nonnull
    public Stream<AbcPerson> getServiceResponsibles(final int serviceId) {
        return createResponsiblesRequestBuilder()
                .serviceId(serviceId)
                .fields("person.login")
                .stream()
                .map(AbcServicePersonHolder::getPerson);
    }

    @Nonnull
    public Stream<AbcPerson> getServiceMembers(final int serviceId, final int roleId) {
        return createMembersRequestBuilder()
                .serviceId(serviceId)
                .role(roleId)
                .fields("id", "person.login")
                .stream()
                .map(AbcServicePersonHolder::getPerson);
    }

    public Stream<AbcPerson> getServiceMembers(final int serviceId) {
        return createMembersRequestBuilder()
                .serviceId(serviceId)
                .fields("id", "person.login")
                .stream()
                .map(AbcServicePersonHolder::getPerson);
    }

    public Stream<AbcServiceMember> getServiceMembersByUser(final Set<String> logins, final AbcServiceState... states) {
        return createMembersByUsersRequestBuilder()
                .logins(logins)
                .states(states)
                .fields("person.login", "service.id", "role.id", "role.service.id")
                .stream();
    }

    public Stream<AbcServiceMember> getServiceMembersWithRoleByUser(final Set<String> logins, final long roleId, final AbcServiceState... states) {
        return createMembersByUsersRequestBuilder()
                .logins(logins)
                .states(states)
                .role(roleId)
                .fields("person.login", "service.id", "role.id", "role.service.id")
                .stream();
    }

    @Nonnull
    public ServiceRequestBuilder createServiceRequestBuilder() {
        return new ServiceRequestBuilder(abcApi, abcTvmDestination);
    }

    public void validateAbcServiceHierarchy(final Project projectToUpdate) {
        validateAbcServiceHierarchy(projectToUpdate.isRoot() ? null :
                        projectToUpdate.getParent(), projectToUpdate.getAbcServiceId(), projectToUpdate.getSubProjects(),
                Collections.singletonMap(projectToUpdate, projectToUpdate.getAbcServiceId())
        );
    }

    public void validateAbcServiceHierarchy(@Nullable final Project parent, @Nullable final Integer abcServiceId) {
        validateAbcServiceHierarchy(parent, abcServiceId, Collections.emptyList());
    }

    public void validateAbcServiceHierarchy(@Nullable final Project parent, @Nullable final Integer abcServiceId,
                                            final Collection<Project> subProjects) {
        validateAbcServiceHierarchy(parent, abcServiceId, subProjects, Collections.emptyMap());
    }

    public void validateAbcServiceHierarchy(@Nullable final Project parent, @Nullable final Integer abcServiceId,
                                            final Collection<Project> subProjects, final Map<Project, Integer> abcServiceIdOverride) {
        if (abcServiceId == null || skipProjectAbcServiceValidation) {
            return;
        }

        final Set<Integer> dispenserSplitAncestors;
        Optional<Project> firstAncestorWithDifferentAbcServiceId = Optional.empty();

        Stream<Project> subProjectsDescendants = getSubs(subProjects, abcServiceId);

        final Multimap<Integer, Project> ancestorProjectsByAbcIds = HashMultimap.create();

        if (parent == null) {
            dispenserSplitAncestors = Collections.emptySet();
        } else {
            final List<Project> ancestors = parent.getPathToRoot();
            firstAncestorWithDifferentAbcServiceId = Optional.of(ancestors.get(ancestors.size() - 1));

            int abcPrefixProjectCount = 0;

            for (int i = 0; i < ancestors.size(); i += 1) {
                final Project ancestor = ancestors.get(i);
                if (ancestor.getAbcServiceId() == null) {
                    continue;
                }
                if (abcServiceId.equals(ancestor.getAbcServiceId())) {
                    abcPrefixProjectCount = i + 1;
                } else {
                    firstAncestorWithDifferentAbcServiceId = Optional.of(ancestor);
                    break;
                }
            }

            dispenserSplitAncestors = ancestors.stream()
                    .skip(abcPrefixProjectCount)
                    .peek(p -> ancestorProjectsByAbcIds.put(p.getAbcServiceId(), p))
                    .map(Project::getAbcServiceId)
                    .filter(Objects::nonNull)
                    .collect(Collectors.toSet());

            if (abcPrefixProjectCount > 0) {
                final Project lastAncestorWithSameAbcServiceId = ancestors.get(abcPrefixProjectCount - 1);
                subProjectsDescendants = Stream.concat(subProjectsDescendants, getSubs(lastAncestorWithSameAbcServiceId.getSubProjects(), abcServiceId));
            }
        }

        final Multimap<Integer, Project> descendantProjectsByAbcIds = HashMultimap.create();

        final Set<Integer> dispenserSplitDescendants = subProjectsDescendants
                .peek(p -> descendantProjectsByAbcIds.put(p.getAbcServiceId(), p))
                .map(Project::getAbcServiceId)
                .filter(Objects::nonNull)
                .collect(Collectors.toSet());

        final Set<Integer> abcSplitAncestors = new HashSet<>();
        final Set<Integer> abcSplitDescendants = new HashSet<>();

        final Consumer<AbcService> saveAbcServiceSplit = s -> {
            final List<Integer> ancestorsIds = s.getAncestors().stream().map(AbcServiceReference::getId)
                    .collect(Collectors.toList());

            if (abcServiceId.equals(s.getId())) {
                abcSplitAncestors.addAll(ancestorsIds);
                return;
            }

            final int splitPoint = ancestorsIds.indexOf(abcServiceId);

            if (splitPoint >= 0) {
                abcSplitAncestors.addAll(ancestorsIds.subList(0, splitPoint));

                abcSplitDescendants.addAll(ancestorsIds.subList(splitPoint + 1, ancestorsIds.size()));
                abcSplitDescendants.add(s.getId());
            }

        };

        final long childCount = getServicesByParentWithDescendants(abcServiceId)
                .peek(saveAbcServiceSplit)
                .count();

        if (childCount <= 0) {
            final long servicesWithNewAbcId = getServicesWithAncestors(Collections.singleton(abcServiceId))
                    .peek(saveAbcServiceSplit)
                    .count();

            if (servicesWithNewAbcId <= 0) {
                throw new IllegalArgumentException(INVALID_ABC_SERVICE_ID_ERROR_MESSAGE);
            }
        }

        final Sets.SetView<Integer> invalidAncestors = Sets.difference(dispenserSplitAncestors, abcSplitAncestors);

        if (!invalidAncestors.isEmpty()) {
            final Project mostClosestParent = getMostClosestParent(ancestorProjectsByAbcIds, invalidAncestors);
            throw new IllegalArgumentException(INVALID_ANCESTOR_ERROR_MESSAGE + mostClosestParent);
        }

        final Sets.SetView<Integer> invalidDescendants = Sets.difference(dispenserSplitDescendants, abcSplitDescendants);

        if (!invalidDescendants.isEmpty()) {
            final List<Project> invalidChildren = getMostClosestChildren(descendantProjectsByAbcIds, invalidDescendants);
            throw new IllegalArgumentException(INVALID_DESCENDANTS_ABC_SERVICES_ERROR_MESSAGE +
                    StringUtils.join(invalidChildren, ", "));
        }

        final Multimap<Integer, Project> projectsByAbcIds = HashMultimap.create();

        final Set<Integer> parentAbcServiceDescendants = firstAncestorWithDifferentAbcServiceId.map(project -> {
            return getSubs(project.getSubProjects(), project.getAbcServiceId())
                    .peek(p -> projectsByAbcIds.put(p.getAbcServiceId(), p))
                    .map(p -> abcServiceIdOverride.getOrDefault(p, p.getAbcServiceId()))
                    .filter(Objects::nonNull)
                    .filter(id -> !abcServiceId.equals(id))
                    .collect(Collectors.toSet());
        }).orElse(Collections.emptySet());

        final Set<Integer> neighborhoodAbcServiceIds = Sets.difference(parentAbcServiceDescendants, dispenserSplitDescendants);

        final Sets.SetView<Integer> possibleParentsInNeighborhood = Sets.intersection(neighborhoodAbcServiceIds, abcSplitAncestors);

        if (!possibleParentsInNeighborhood.isEmpty()) {
            final Project mostClosestParent = getMostClosestParent(projectsByAbcIds, possibleParentsInNeighborhood);
            throw new IllegalArgumentException(NON_SPECIFIC_PARENT_ABC_SERVICE_ERROR_MESSAGE + mostClosestParent);
        }

        final Sets.SetView<Integer> possibleChildrenInNeighborhood = Sets.intersection(neighborhoodAbcServiceIds, abcSplitDescendants);

        if (!possibleChildrenInNeighborhood.isEmpty()) {
            final List<Project> mostClosestChildren = getMostClosestChildren(projectsByAbcIds, possibleChildrenInNeighborhood);
            throw new IllegalArgumentException(MISSING_NESTED_ABC_SERVICES_ERROR_MESSAGE +
                    StringUtils.join(mostClosestChildren, ", "));
        }
    }

    @NotNull
    private List<Project> getMostClosestChildren(final Multimap<Integer, Project> projectsByAbcIds,
                                                 final Sets.SetView<Integer> abcServiceIds) {
        final List<Pair<Project, List<Project>>> projectWithPathToRoot = abcServiceIds.stream()
                .flatMap(id -> projectsByAbcIds.get(id).stream())
                .map(p -> Pair.of(p, p.getPathToRoot()))
                .collect(Collectors.toList());

        final List<List<Project>> mostCommonAncestors = new ArrayList<>();

        for (final Pair<Project, List<Project>> projectListPair : projectWithPathToRoot) {
            boolean isNewBranch = true;
            for (final List<Project> mostCommonAncestor : mostCommonAncestors) {
                if (mostCommonAncestor.contains(projectListPair.getLeft())) {
                    mostCommonAncestors.remove(mostCommonAncestor);
                    mostCommonAncestor.add(projectListPair.getLeft());

                    isNewBranch = false;
                    break;

                } else if (projectListPair.getRight().containsAll(mostCommonAncestor)) {
                    isNewBranch = false;
                    break;
                }
            }
            if (isNewBranch) {
                mostCommonAncestors.add(projectListPair.getRight());
            }
        }

        return mostCommonAncestors.stream()
                .map(list -> list.get(0))
                .collect(Collectors.toList());
    }

    @NotNull
    private Project getMostClosestParent(final Multimap<Integer, Project> projectsByAbcIds, final Sets.SetView<Integer> abcServiceIds) {
        return abcServiceIds.stream()
                .flatMap(id -> projectsByAbcIds.get(id).stream())
                .map(p -> Pair.of(p, p.getPathToRoot().size()))
                .max(Comparator.comparingInt(Pair::getRight))
                .map(Pair::getLeft)
                .get();
    }

    @NotNull
    private Stream<Project> getSubs(final Collection<Project> projects, @Nullable final Integer abcServiceIdToIgnore) {
        return projects.stream()
                .flatMap(p -> {
                    if (abcServiceIdToIgnore != null) {
                        if (abcServiceIdToIgnore.equals(p.getAbcServiceId()) || p.getAbcServiceId() == null) {
                            return getSubs(p.getSubProjects(), abcServiceIdToIgnore);
                        }
                    }
                    return Stream.concat(Stream.of(p), getSubs(p.getSubProjects(), null));
                });
    }

    @Nonnull
    public MembersRequestBuilder createMembersRequestBuilder() {
        return new MembersRequestBuilder(abcApi, abcTvmDestination);
    }

    @Nonnull
    public ResponsiblesRequestBuilder createResponsiblesRequestBuilder() {
        return new ResponsiblesRequestBuilder(abcApi, abcTvmDestination);
    }

    @Nonnull
    public MembersByUsersRequestBuilder createMembersByUsersRequestBuilder() {
        return new MembersByUsersRequestBuilder(abcApi, abcTvmDestination);
    }

    @NotNull
    public Stream<AbcRole> getServiceRoles(final int serviceId) {
        return new RoleRequestBuilder(abcApi, abcTvmDestination)
                .serviceId(serviceId)
                .stream();
    }

    @NotNull
    public Stream<AbcPerson> getServiceMembersWithRoles(final int abcServiceId, @NotNull final Set<String> roles) {
        final Set<Integer> roleIds = roles.stream()
                .map(Integer::parseInt)
                .collect(Collectors.toSet());

        final Set<Integer> existingRoleIds = new RoleRequestBuilder(abcApi, abcTvmDestination)
                .ids(roleIds)
                .stream()
                .map(AbcRole::getId)
                .collect(Collectors.toSet());

        final Sets.SetView<Integer> invalidRoles = Sets.difference(roleIds, existingRoleIds);
        if (!invalidRoles.isEmpty()) {
            throw new IllegalArgumentException("Invalid abc roles: " + StringUtils.join(invalidRoles, ", "));
        }

        return roles.stream()
                .map(Integer::parseInt)
                .flatMap(roleId -> getServiceMembers(abcServiceId, roleId));
    }

    @NotNull
    private Map<Integer, ? extends Collection<? extends AbcServicePersonHolder>> collectPersonsByService(
            final BasePageServiceRelationRequestBuilder<? extends AbcServicePersonHolder, ?> provider, final Collection<Integer> serviceIds) {
        return Lists.partition(Lists.newArrayList(serviceIds), SERVICE_COUNT_IN_COLLECT_PERSON_REQUEST)
                .stream()
                .flatMap(ids -> provider
                        .serviceId(ids)
                        .fields("person.login", "service.id")
                        .stream())
                .collect(Collectors.groupingBy(personHolder -> personHolder.getServiceReference().getId()));
    }

    @Nonnull
    public Map<Integer, ? extends Collection<? extends AbcServicePersonHolder>> getServiceResponsibles(
            final Collection<Integer> serviceIds) {
        return collectPersonsByService(createResponsiblesRequestBuilder(), serviceIds);
    }

    @NotNull
    public Map<Integer, ? extends Collection<? extends AbcServicePersonHolder>> getServiceMembers(final Collection<Integer> serviceIds) {
        return collectPersonsByService(createMembersRequestBuilder(), serviceIds);
    }

    @NotNull
    public Map<Integer, List<AbcServiceMember>> getServiceMembers(final Collection<Integer> serviceIds,
                                                                  final Collection<Integer> roleIds) {
        return createMembersRequestBuilder()
                .serviceId(serviceIds)
                .role(roleIds)
                .fields("id", "person.login", "service.id", "role.id")
                .stream()
                .collect(Collectors.groupingBy(member -> member.getServiceReference().getId()));
    }

    public Stream<AbcServiceGradient> getServiceGradients() {
        return new CursorItemIterator<>(
                (cursor) -> robot.runAuthorized(() -> abcApi.getServicesGradient(cursor))
        ).stream();
    }
}
