package ru.yandex.qe.dispenser.ws.abc;

import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.abc.AbcService;
import ru.yandex.qe.dispenser.domain.abc.AbcServiceGradient;
import ru.yandex.qe.dispenser.domain.abc.AbcServiceReference;
import ru.yandex.qe.dispenser.domain.abc.AbcServiceState;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectDao;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectManager;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.util.MoreCollectors;
import ru.yandex.qe.dispenser.domain.util.MoreFunctions;

@ParametersAreNonnullByDefault
public class ProjectTreeSync {
    private static final Logger LOG = LoggerFactory.getLogger(ProjectTreeSync.class);

    @NotNull
    private final AbcApiHelper abcApiHelper;
    @NotNull
    private final ProjectDao projectDao;
    @NotNull
    private final UpdateProjectMembers updateProjectMembers;
    @NotNull
    private final ProjectManager projectManager;

    public ProjectTreeSync(final AbcApiHelper abcApiHelper,
                           final ProjectDao projectDao,
                           final UpdateProjectMembers updateProjectMembers,
                           final ProjectManager projectManager) {
        this.abcApiHelper = abcApiHelper;
        this.projectDao = projectDao;
        this.updateProjectMembers = updateProjectMembers;
        this.projectManager = projectManager;
    }

    private static class Context {
        final Multimap<Integer, AbcService> abcServicesByParentAbcService;
        final Map<Integer, Project> projectsByAbcServiceId;
        private final Map<Integer, Long> valueStreamIdByAbcServiceId;

        private Context(final Multimap<Integer, AbcService> abcServicesByParentAbcService,
                        final Map<Integer, Project> projectsByAbcServiceId,
                        final Map<Integer, Long> valueStreamIdByAbcServiceId) {
            this.abcServicesByParentAbcService = abcServicesByParentAbcService;
            this.projectsByAbcServiceId = projectsByAbcServiceId;
            this.valueStreamIdByAbcServiceId = valueStreamIdByAbcServiceId;
        }
    }

    private Stream<Project> syncForParent(@Nullable final Integer parentServiceId, final Project parentProject,
                                          final Context context) {
        final Collection<AbcService> childAbcIds = context.abcServicesByParentAbcService.get(parentServiceId);

        return childAbcIds.stream()
                .flatMap(abcService -> {
                    final Integer abcServiceId = abcService.getId();
                    final Project.Builder projectBuilder = getSyncedProjectBuilder(abcService, parentProject, context.valueStreamIdByAbcServiceId.get(abcServiceId));
                    final Project syncedProject = Optional.ofNullable(context.projectsByAbcServiceId.get(abcServiceId))
                            .map(project -> syncFields(project, projectBuilder.id(project.getId()).build()))
                            .orElseGet(() -> createProjectForAbcServiceWithParent(projectBuilder.build()));

                    return Stream.concat(Stream.of(syncedProject), syncForParent(abcServiceId, syncedProject, context));
                });
    }

    @NotNull
    private Project syncFields(final Project project, final Project syncedProject) {
        final Project newParent = syncedProject.getParent();

        if (!Objects.equals(project.getParent(), newParent)) {
            projectManager.moveQuotas(project, newParent);
        }

        projectDao.update(syncedProject);

        return syncedProject;
    }

    @NotNull
    private Project createProjectForAbcServiceWithParent(final Project syncedProject) {
        return projectDao.create(syncedProject);
    }

    @NotNull
    private Project.Builder getSyncedProjectBuilder(final AbcService abcService, final Project parentProject, @Nullable final Long valueStreamId) {
        final String slug = getSlug(abcService);
        return Project.withKey(slug)
                .parent(parentProject)
                .name(abcService.getName().getEn())
                .description(abcService.getDescription().getEn())
                .abcServiceId(abcService.getId())
                .valueStreamAbcServiceId(valueStreamId);
    }

    @NotNull
    private Project reduceProjects(final List<Project> projects) {

        if (projects.size() > 1) {
            projects.sort(Comparator.comparingLong(Project::getId));
        }
        return projects.get(0);
    }

    @NotNull
    private static String getSlug(final AbcService abcService) {
        return abcService.getSlug();
    }

    public void perform(final UpdateProjectMembers.RoleChangesHolder roleChangesHolder) {
        final Map<Integer, Long> valueStreamIdByAbcServiceId = getValueStreamIdByAbcServiceId();

        final Set<Project> syncedProjects = syncProjects(valueStreamIdByAbcServiceId);

        if (!syncedProjects.isEmpty()) {
            LOG.info("Started members sync");
            updateProjectMembers.update(syncedProjects, roleChangesHolder);
            LOG.info("Members sync finished");
        }
    }

    private Map<Integer, Long> getValueStreamIdByAbcServiceId() {
        return abcApiHelper.getServiceGradients()
                .reduce(new HashMap<>(), (m, sg) -> {
                    final int id = (int) sg.getId();
                    if (sg.getValueStream() != null) {
                        m.put(id, sg.getValueStream());
                    } else if (AbcServiceGradient.TYPE_VALUE_STREAM.equals(sg.getType())) {
                        m.put(id, sg.getId());
                    }
                    return m;
                }, MoreFunctions.leftProjection());
    }

    private Set<Project> syncProjects(final Map<Integer, Long> valueStreamIdByAbcServiceId) {
        final Multimap<Integer, AbcService> abcServicesByParentAbcService = abcApiHelper.createServiceRequestBuilder()
                .fields("id", "parent.id", "slug", "name", "description")
                .state(AbcServiceState.DEVELOP, AbcServiceState.SUPPORTED, AbcServiceState.NEED_INFO)
                .stream()
                .collect(MoreCollectors.toLinkedMultimap(s -> s.getParent() == null ? null : s.getParent().getId(), Function.identity()));

        LOG.info("Found {} services", abcServicesByParentAbcService.values().size());

        final Set<Project> projects = Hierarchy.get().getProjectReader().getAll()
                .stream()
                .filter(p -> !p.isPersonal())
                .collect(Collectors.toSet());

        final Map<String, Integer> idBySlug = abcServicesByParentAbcService.values().stream()
                .collect(Collectors.toMap(ProjectTreeSync::getSlug, AbcServiceReference::getId));

        final Map<Integer, Project> projectsByAbcId = projects.stream()
                .filter(p -> idBySlug.containsKey(p.getPublicKey()))
                .collect(Collectors.toMap(p -> idBySlug.get(p.getPublicKey()), Function.identity()));

        LOG.info("Already mapped {} projects", projectsByAbcId.size());

        final Project rootProject = projects.stream()
                .filter(p -> p.getPublicKey().equals("yandex"))
                .findFirst()
                .get();

        final Optional<Project> trashProject = projects.stream()
                .filter(p -> p.getPublicKey().equals(Project.TRASH_PROJECT_KEY))
                .findFirst();

        final Context context = new Context(abcServicesByParentAbcService, projectsByAbcId, valueStreamIdByAbcServiceId);
        final Set<Project> syncedProjects = syncForParent(null, rootProject, context)
                .collect(Collectors.toSet());

        final HashSet<Project> syncableProjects = Sets.newHashSet(projects);
        syncableProjects.remove(rootProject);

        trashProject.ifPresent(syncableProjects::remove);

        final Set<Project> projectsToRemove = Sets.difference(syncableProjects, syncedProjects);

        removeProjects(projectsToRemove);
        LOG.info("Removed {} projects", projectsToRemove.size());

        return syncedProjects;
    }

    private void removeProjects(final Set<Project> projects) {
        final Set<Project> mostCommonAncestors = projects.stream()
                .filter(p -> {
                    if (p.isRoot()) {
                        return true;
                    }
                    final List<Project> pathToRoot = p.getPathToRoot();
                    final Set<Project> ancestors = Sets.newHashSet(pathToRoot.subList(1, pathToRoot.size()));
                    return Sets.intersection(projects, ancestors).isEmpty();
                })
                .collect(Collectors.toSet());

        final Map<Boolean, List<Project>> projectsByRetainableState = mostCommonAncestors.stream()
                .collect(Collectors.groupingBy(this::isRetainableProject));

        projectDao.deleteAll(projectsByRetainableState.getOrDefault(false, Collections.emptyList()));
        moveProjectsToTrash(projectsByRetainableState.getOrDefault(true, Collections.emptyList()));
    }

    private boolean isRetainableProject(final Project project) {
        return true; //TODO create criteria for project complete removal
    }

    @NotNull
    private Project createIfAbsentTrashProject() {
        return projectDao.createIfAbsent(Project.withKey(Project.TRASH_PROJECT_KEY)
                .parent(Hierarchy.get().getProjectReader().read("yandex"))
                .name("Trash")
                .description("Trash parent project")
                .build());
    }

    @NotNull
    private Project moveProject(final Project project, final Project parentProject) {
        if (project.getParent().equals(parentProject)) {
            return project;
        }

        projectManager.moveQuotas(project, parentProject);

        final Project movedProject = Project.copyOf(project)
                .parent(parentProject)
                .build();

        projectDao.update(movedProject);

        return movedProject;
    }

    private void moveProjectsToTrash(final Collection<Project> projects) {
        if (projects.isEmpty()) {
            return;
        }

        final Project trashProject = createIfAbsentTrashProject();

        projects.forEach(project -> moveProject(project, trashProject));
    }
}
