package ru.yandex.qe.dispenser.ws.abc;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.inject.Inject;

import com.google.common.base.Stopwatch;
import com.google.common.base.Ticker;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Sets;
import com.google.common.collect.Table;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.ProjectRole;
import ru.yandex.qe.dispenser.domain.abc.AbcServiceMember;
import ru.yandex.qe.dispenser.domain.abc.AbcServicePersonHolder;
import ru.yandex.qe.dispenser.domain.dao.person.StaffCache;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectDao;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectReader;
import ru.yandex.qe.dispenser.domain.dao.project.role.ProjectRoleCache;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.index.NormalizedPrimaryKeyBase;
import ru.yandex.qe.dispenser.solomon.SolomonHolder;
import ru.yandex.qe.dispenser.domain.util.MoreCollectors;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

@Component("updateProjectMembers")
@ParametersAreNonnullByDefault
public class UpdateProjectMembers {
    private static final Logger LOG = LoggerFactory.getLogger(UpdateProjectMembers.class);

    public static final String SENSOR_PREFIX = "abc_members_sync_task.";
    public static final String ELAPSED_TIME_SENSOR = SENSOR_PREFIX + "elapsed_time";
    public static final String ERROR_RATE_SENSOR = SENSOR_PREFIX + "error_rate";
    public static final String LAST_START_SENSOR = SENSOR_PREFIX + "time_since_last_start";
    public static final String LAST_SUCCESS_SENSOR = SENSOR_PREFIX + "time_since_last_success_end";

    public static final String LAST_ADDED_ROLES_SENSOR = SENSOR_PREFIX + "added_roles";
    public static final String LAST_REMOVED_ROLES_SENSOR = SENSOR_PREFIX + "removed_roles";
    public static final String LAST_ROLE_ADD_FAILURES_SENSOR = SENSOR_PREFIX + "role_add_failures";

    @NotNull
    private final StaffCache staffCache;
    @NotNull
    private final ProjectDao projectDao;
    @NotNull
    private final AbcApiHelper abcApiHelper;
    @NotNull
    private final ProjectRoleCache projectRoleCache;
    @NotNull
    private final Histogram elapsedTime;
    @NotNull
    private final Rate errorRate;
    @NotNull
    private final Ticker ticker = Ticker.systemTicker();

    private volatile long lastStart;
    private volatile long lastSuccessEnd;

    private volatile long lastAddedRoles;
    private volatile long lastRemovedRoles;
    private volatile long lastRoleAdditionFailures;

    @Inject
    public UpdateProjectMembers(final ProjectDao projectDao, final StaffCache staffCache, final AbcApiHelper abcApiHelper,
                                final SolomonHolder solomonHolder, @NotNull final ProjectRoleCache roleCache) {
        this.projectDao = projectDao;
        this.staffCache = staffCache;
        this.abcApiHelper = abcApiHelper;
        this.projectRoleCache = roleCache;
        final MetricRegistry rootRegistry = solomonHolder.getRootRegistry();
        this.elapsedTime = rootRegistry.histogramRate(ELAPSED_TIME_SENSOR, Labels.of(), Histograms.exponential(22, 2, 1.0d));
        this.errorRate = rootRegistry.rate(ERROR_RATE_SENSOR, Labels.of());
        this.lastStart = TimeUnit.NANOSECONDS.toMillis(ticker.read());
        rootRegistry.lazyGaugeInt64(LAST_START_SENSOR, Labels.of(), () -> TimeUnit.NANOSECONDS.toMillis(ticker.read()) - lastStart);
        this.lastSuccessEnd = TimeUnit.NANOSECONDS.toMillis(ticker.read());
        rootRegistry.lazyGaugeInt64(LAST_SUCCESS_SENSOR, Labels.of(), () -> TimeUnit.NANOSECONDS.toMillis(ticker.read()) - lastSuccessEnd);

        rootRegistry.lazyGaugeInt64(LAST_ADDED_ROLES_SENSOR, Labels.of(), () -> lastAddedRoles);
        rootRegistry.lazyGaugeInt64(LAST_REMOVED_ROLES_SENSOR, Labels.of(), () -> lastRemovedRoles);
        rootRegistry.lazyGaugeInt64(LAST_ROLE_ADD_FAILURES_SENSOR, Labels.of(), () -> lastRoleAdditionFailures);
    }

    public void update() {
        LOG.info("Syncing service members with ABC...");
        final Stopwatch stopwatch = Stopwatch.createStarted();
        lastStart = TimeUnit.NANOSECONDS.toMillis(ticker.read());
        boolean success = false;
        final RoleChangesHolder roleChangesHolder = new RoleChangesHolder();
        try {
            final Set<Project> projectsWithAbcMembersSync = Hierarchy.get().getProjectReader().getAll().stream()
                    .filter(project -> Objects.nonNull(project.getAbcServiceId()))
                    .filter(Project::isSyncedWithAbc)
                    .collect(Collectors.toSet());
            update(projectsWithAbcMembersSync, roleChangesHolder);
            success = true;
        } catch (Throwable e) {
            LOG.error("Failed to sync service members with ABC", e);
            throw e;
        } finally {
            stopwatch.stop();
            final long elapsed = stopwatch.elapsed(TimeUnit.MILLISECONDS);
            if (success) {
                lastSuccessEnd = TimeUnit.NANOSECONDS.toMillis(ticker.read());
                LOG.info("Successfully synced service members with ABC in {} seconds", TimeUnit.MILLISECONDS.toSeconds(elapsed));
            } else {
                errorRate.inc();
                LOG.info("Failed to sync service members with ABC in {} seconds", TimeUnit.MILLISECONDS.toSeconds(elapsed));
            }
            elapsedTime.record(elapsed);
            lastAddedRoles = roleChangesHolder.getAddedRoles();
            lastRemovedRoles = roleChangesHolder.getRemovedRoles();
            lastRoleAdditionFailures = roleChangesHolder.getAdditionFailures();
        }
    }

    public void update(final Set<Project> projectsWithAbcMembersSync, final RoleChangesHolder roleChangesHolder) {
        LOG.info("Syncing members for {} projects...", projectsWithAbcMembersSync.size());
        final ProjectReader projectReader = Hierarchy.get().getProjectReader();

        final Set<ProjectRole> projectRoles = projectRoleCache.getAll();

        final Set<ProjectRole> rolesWithAbcMemberSync = new HashSet<>();
        final Set<ProjectRole> rolesWithAbcResponsibleSync = new HashSet<>();
        final Map<Integer, ProjectRole> projectRoleByAbcRoleId = new HashMap<>();

        for (final ProjectRole projectRole : projectRoles) {
            if (projectRole.getAbcRoleSyncType() == ProjectRole.AbcRoleSyncType.MEMBER) {
                rolesWithAbcMemberSync.add(projectRole);
            }
            if (projectRole.getAbcRoleSyncType() == ProjectRole.AbcRoleSyncType.RESPONSIBLE) {
                rolesWithAbcResponsibleSync.add(projectRole);
            }
            if (projectRole.getAbcRoleSyncType() == ProjectRole.AbcRoleSyncType.ROLE) {
                projectRoleByAbcRoleId.put(projectRole.getAbcRoleId().intValue(), projectRole);
            }
        }

        final Table<Project, ProjectRole, Set<String>> projectMembersWithRole = getPersonWithRoles(projectsWithAbcMembersSync, projectRoleByAbcRoleId);

        if (!rolesWithAbcMemberSync.isEmpty()) {
            final Map<Project, Set<String>> projectMemberByProject = getPersons(projectsWithAbcMembersSync, abcApiHelper::getServiceMembers);
            for (final ProjectRole projectRole : rolesWithAbcMemberSync) {
                projectMembersWithRole.column(projectRole).putAll(projectMemberByProject);
            }
        }

        if (!rolesWithAbcResponsibleSync.isEmpty()) {
            final Map<Project, Set<String>> projectMemberByProject = getPersons(projectsWithAbcMembersSync, abcApiHelper::getServiceResponsibles);
            for (final ProjectRole projectRole : rolesWithAbcResponsibleSync) {
                projectMembersWithRole.column(projectRole).putAll(projectMemberByProject);
            }
        }

        final HashSet<ProjectRole> syncedRoles = new HashSet<>();
        syncedRoles.addAll(rolesWithAbcMemberSync);
        syncedRoles.addAll(rolesWithAbcResponsibleSync);
        syncedRoles.addAll(projectRoleByAbcRoleId.values());

        projectsWithAbcMembersSync.forEach(project -> {
            LOG.debug("Updating membership for {}", project.getPublicKey());

            syncedRoles.forEach(role -> {
                Set<String> logins = projectMembersWithRole.get(project, role);
                if (logins == null) {
                    logins = Collections.emptySet();
                }

                final Set<Person> newMembers = staffCache.tryGetPersonsByLogins(logins);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("New members with role {} before staff resolution for {} are: {}", role.getKey(), project.getPublicKey(),
                            String.join(", ", logins));
                    LOG.debug("New members with role {} after staff resolution for {} are: {}", role.getKey(), project.getPublicKey(),
                            String.join(", ", newMembers.stream().map(NormalizedPrimaryKeyBase::getKey).collect(Collectors.toSet())));
                }
                final Set<Person> oldMembers = projectReader.getLinkedPersons(project, role.getKey());
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Old members with role {} for {} are: {}", role.getKey(), project.getPublicKey(),
                            String.join(", ", oldMembers.stream().map(NormalizedPrimaryKeyBase::getKey).collect(Collectors.toSet())));
                }

                updateProjectPersons(project, newMembers, oldMembers, role, roleChangesHolder);
            });

        });
        LOG.info("Synced members for {} projects", projectsWithAbcMembersSync.size());
    }

    public void updateProjectPersons(final Project project, final Set<Person> newPersons, final Set<Person> oldPersons,
                                     final ProjectRole role, final RoleChangesHolder roleChangesHolder) {
        final Set<Person> toAttach = Sets.newHashSet(Sets.difference(newPersons, oldPersons));
        final Set<Person> toDetach = Sets.newHashSet(Sets.difference(oldPersons, newPersons));

        if (!toDetach.isEmpty()) {
            LOG.info("Detaching role {} in project {} for users: {}", role, project.getPublicKey(),
                    String.join(", ", toDetach.stream().map(NormalizedPrimaryKeyBase::getKey).collect(Collectors.toSet())));
            projectDao.detachAll(toDetach, Collections.emptySet(), project, role.getId());
            roleChangesHolder.incRemoved(toDetach.size());
        }

        if (!toAttach.isEmpty()) {
            try {
                LOG.info("Attaching role {} in project {} for users: {}", role, project.getPublicKey(),
                        String.join(", ", toAttach.stream().map(NormalizedPrimaryKeyBase::getKey).collect(Collectors.toSet())));
                projectDao.attachAll(toAttach, Collections.emptySet(), project, role.getId());
                roleChangesHolder.incAdded(toAttach.size());
            } catch (RuntimeException e) {
                LOG.error("Can't attach person for project " + project.getPublicKey(), e);
                roleChangesHolder.incAdditionFailures();
            }
        }
    }

    private Table<Project, ProjectRole, Set<String>> getPersonWithRoles(final Set<Project> projects, final Map<Integer, ProjectRole> projectRoleByAbcRoleId) {

        final MembersRequestBuilder requestBuilder = abcApiHelper.createMembersRequestBuilder();

        final Multimap<Integer, Project> projectByAbcServiceId = projects.stream()
                .collect(MoreCollectors.toLinkedMultimap(Project::getAbcServiceId, Function.identity()));
        final Set<Integer> serviceIds = projectByAbcServiceId.keySet();

        if (projects.size() <= 100) {
            requestBuilder
                    .serviceId(serviceIds);
        }

        final Table<Project, ProjectRole, Set<String>> projectMembersByRole = HashBasedTable.create();

        requestBuilder
                .role(projectRoleByAbcRoleId.keySet())
                .fields("person.login", "service.id", "role.id")
                .stream()
                .filter(member -> serviceIds.contains(member.getServiceReference().getId()))
                .forEach(member -> {
                    final Integer serviceId = member.getServiceReference().getId();
                    final String login = member.getPerson().getLogin();
                    final ProjectRole projectRole = projectRoleByAbcRoleId.get(member.getRole().getId());
                    if (projectRole != null) {
                        projectByAbcServiceId.get(serviceId).forEach(project -> {
                            Set<String> logins = projectMembersByRole.get(project, projectRole);
                            if (logins == null) {
                                logins = new HashSet<>();
                                projectMembersByRole.put(project, projectRole, logins);
                            }
                            logins.add(login);
                        });
                    }
                });
        if (LOG.isDebugEnabled()) {
            projectMembersByRole.cellSet().forEach(c -> {
                LOG.debug("Project members for project {} and role {} from ABC: {}", c.getRowKey().getPublicKey(), c.getColumnKey().getKey(), String.join(", ", c.getValue()));
            });
        }
        return projectMembersByRole;
    }

    private Map<Project, Set<String>> getPersons(final Set<Project> projects,
                                                 final Function<Collection<Integer>, Map<Integer, ? extends Collection<? extends AbcServicePersonHolder>>> customPersonHolderProvider) {
        final Map<Project, Set<String>> loginsByProject;
        if (!projects.isEmpty()) {
            loginsByProject = getPersonsFromAbcToProjectRole(projects, customPersonHolderProvider);
        } else {
            loginsByProject = Collections.emptyMap();
        }
        if (LOG.isDebugEnabled()) {
            loginsByProject.keySet().forEach(p -> {
                LOG.debug("Members for project {} from ABC: {}", p.getPublicKey(),
                        String.join(", ", loginsByProject.get(p)));
            });
        }
        return loginsByProject;
    }

    private Map<Project, Set<String>> getPersonsFromAbcToProjectRole(final Set<Project> projects,
                                                                     final Function<Collection<Integer>, Map<Integer, ? extends Collection<? extends AbcServicePersonHolder>>> personHoldersProvider) {
        final Map<Project, Set<String>> loginsByProject = new HashMap<>();

        final Multimap<Integer, Project> projectByAbcServiceId = projects.stream()
                .collect(MoreCollectors.toLinkedMultimap(Project::getAbcServiceId, Function.identity()));

        personHoldersProvider.apply(projectByAbcServiceId.keySet())
                .forEach((abcServiceId, personHolders) -> {

                    final Set<String> personLogins = personHolders.stream()
                            .map(personHolder -> personHolder.getPerson().getLogin())
                            .collect(Collectors.toSet());

                    projectByAbcServiceId.get(abcServiceId).forEach(project -> {
                        loginsByProject.put(project, personLogins);
                    });
                });
        if (LOG.isDebugEnabled()) {
            loginsByProject.keySet().forEach(p -> {
                LOG.debug("Members with default roles for project {} from ABC: {}", p.getPublicKey(),
                        String.join(", ", loginsByProject.get(p)));
            });
        }
        return loginsByProject;
    }


    private Multimap<Project, String> getPersonsFromAbcWithRolesToProjectRole(final Multimap<Project, Integer> linkedRoles) {

        final Multimap<Project, String> loginsByProject = HashMultimap.create();

        final Multimap<Integer, Project> projectByAbcServiceId = linkedRoles.keySet().stream()
                .collect(MoreCollectors.toLinkedMultimap(Project::getAbcServiceId, Function.identity()));

        abcApiHelper.getServiceMembers(projectByAbcServiceId.keySet(), linkedRoles.values())
                .forEach((abcServiceId, members) -> {

                    final Map<Integer, List<AbcServiceMember>> membersByRole = members.stream()
                            .collect(Collectors.groupingBy(member -> member.getRole().getId()));

                    projectByAbcServiceId.get(abcServiceId).forEach(project -> {
                        final Stream<AbcServiceMember> membersStream = linkedRoles.get(project)
                                .stream()
                                .flatMap(roleId -> membersByRole.getOrDefault(roleId, Collections.emptyList()).stream());

                        final Set<String> memberLogins = membersStream
                                .map(member -> member.getPerson().getLogin())
                                .collect(Collectors.toSet());

                        loginsByProject.putAll(project, memberLogins);
                    });
                });
        if (LOG.isDebugEnabled()) {
            loginsByProject.keySet().forEach(p -> {
                LOG.debug("Members with custom roles for project {} from ABC: {}", p.getPublicKey(),
                        String.join(", ", loginsByProject.get(p)));
            });
        }
        return loginsByProject;
    }

    public static class RoleChangesHolder {

        private long addedRoles;
        private long removedRoles;
        private long additionFailures;

        public void incAdded(final long inc) {
            addedRoles += inc;
        }

        public void incRemoved(final long inc) {
            removedRoles += inc;
        }

        public void incAdditionFailures() {
            additionFailures++;
        }

        public long getAddedRoles() {
            return addedRoles;
        }

        public long getRemovedRoles() {
            return removedRoles;
        }

        public long getAdditionFailures() {
            return additionFailures;
        }

    }

}
