package ru.yandex.qe.dispenser.ws.abc.validator;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nonnull;
import javax.inject.Inject;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.springframework.context.annotation.Primary;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;

import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.ProjectRole;
import ru.yandex.qe.dispenser.domain.abc.AbcServiceMember;
import ru.yandex.qe.dispenser.domain.abc.AbcServiceState;
import ru.yandex.qe.dispenser.domain.dao.person.PersonDao;
import ru.yandex.qe.dispenser.domain.dao.project.role.ProjectRoleCache;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.ws.abc.AbcApiHelper;

@Component
@Profile("abc-sync")
@Primary
public class ProjectMembersValidatorImpl implements ProjectMembersValidator {

    private static final long VALIDATION_PAGE_SIZE = 100L;

    @NotNull
    private final PersonDao personDao;
    @NotNull
    private final AbcApiHelper abcApiHelper;
    @NotNull
    private final ProjectRoleCache projectRoleCache;

    @Inject
    public ProjectMembersValidatorImpl(@NotNull final PersonDao personDao, @NotNull final AbcApiHelper abcApiHelper, @NotNull final ProjectRoleCache projectRoleCache) {
        this.personDao = personDao;
        this.abcApiHelper = abcApiHelper;
        this.projectRoleCache = projectRoleCache;
    }

    public ProjectMembersValidationResult validate() {

        final Map<String, Map<String, Set<Long>>> missingRolesByRole = new HashMap<>();
        final Map<String, Map<String, Set<Long>>> excessRolesByRole = new HashMap<>();
        final Map<String, ProjectRole> roleByKey = new HashMap<>();

        for (final ProjectRole projectRole : projectRoleCache.getAll()) {
            if (projectRole.getAbcRoleSyncType() != ProjectRole.AbcRoleSyncType.NONE) {
                missingRolesByRole.put(projectRole.getKey(), new HashMap<>());
                excessRolesByRole.put(projectRole.getKey(), new HashMap<>());
                roleByKey.put(projectRole.getKey(), projectRole);
            }
        }

        final Set<ProjectRole> roles = projectRoleCache.getAll();

        // Expectations:
        // 1) For each user known to Dispenser for every role of this user in active services as returned by ABC there is always matching role in Dispenser
        // 2) For each user known to Dispenser there is no roles of this user in Dispenser other than those returned by ABC in active and inactive services
        Long from = null;
        while (true) {
            final PersonDao.PersonMembershipPage page = personDao.getPersonMembershipsPage(from, VALIDATION_PAGE_SIZE);
            if (page.getServiceRolesByLogin().isEmpty() || !page.getLastId().isPresent()) {
                break;
            }
            from = page.getLastId().orElse(null);
            final Set<String> logins = page.getServiceRolesByLogin().keySet();

            final Map<String, Map<String, Set<Long>>> servicePersonByLoginByRoleId = new HashMap<>();
            final Map<ProjectRole, Set<AbcServiceMember>> servicePersonByRoleId = new HashMap<>();

            for (final ProjectRole role : roles) {
                switch (role.getAbcRoleSyncType()) {
                    case MEMBER:
                        servicePersonByRoleId.put(role, abcApiHelper.getServiceMembersByUser(logins,
                                AbcServiceState.DEVELOP, AbcServiceState.SUPPORTED, AbcServiceState.NEED_INFO).collect(Collectors.toSet())
                        );
                        servicePersonByLoginByRoleId.put(role.getKey(), new HashMap<>());
                        break;
                    case ROLE:
                        servicePersonByRoleId.put(role, abcApiHelper.getServiceMembersWithRoleByUser(logins, role.getAbcRoleId(),
                                AbcServiceState.DEVELOP, AbcServiceState.SUPPORTED, AbcServiceState.NEED_INFO).collect(Collectors.toSet())
                        );
                        servicePersonByLoginByRoleId.put(role.getKey(), new HashMap<>());
                        break;
                    case RESPONSIBLE:
                    case NONE:
                    default:
                        continue;
                }

                findMissingRoles(page.getServiceRolesByLogin(), servicePersonByRoleId.get(role), role.getKey(), missingRolesByRole.get(role.getKey()), servicePersonByLoginByRoleId.get(role.getKey()));

            }

            findExcessRoles(page.getServiceRolesByLogin(), servicePersonByLoginByRoleId, excessRolesByRole);

        }
        // Roles in inactive projects are not excessive
        checkRolesWithDeletedServices(excessRolesByRole, roleByKey);
        // Ignore excessive roles when projects are in trash subtree
        // The reason is - if projects are non-exported than we are unable to get actual roles from ABC for them to verify those roles
        skipProjectsInTrash(excessRolesByRole);
        // Ignore missing roles when project is not known
        // Th reason is - exported projects with non-exported parents can not be synced at all but still their roles are available in ABC
        skipUnknownServices(missingRolesByRole);

        return new ProjectMembersValidationResult(missingRolesByRole, excessRolesByRole);
    }

    private void checkRolesWithDeletedServices(@Nonnull final Map<String, Map<String, Set<Long>>> excessRolesByRole, final Map<String, ProjectRole> roleByKey) {

        for (final String roleKey : excessRolesByRole.keySet()) {
            final ProjectRole role = roleByKey.get(roleKey);
            if (role.getAbcRoleSyncType() != ProjectRole.AbcRoleSyncType.MEMBER && role.getAbcRoleSyncType() != ProjectRole.AbcRoleSyncType.ROLE) {
                continue;
            }
            final Map<String, Set<Long>> excessRoles = excessRolesByRole.get(roleKey);
            Lists.partition(new ArrayList<>(excessRoles.keySet()), (int) VALIDATION_PAGE_SIZE).forEach(logins -> {
                final Set<AbcServiceMember> serviceMembers;
                if (role.getAbcRoleSyncType() == ProjectRole.AbcRoleSyncType.MEMBER) {
                    serviceMembers = abcApiHelper
                            .getServiceMembersByUser(new HashSet<>(logins), AbcServiceState.CLOSED, AbcServiceState.DELETED).collect(Collectors.toSet());
                } else {
                    serviceMembers = abcApiHelper.getServiceMembersWithRoleByUser(new HashSet<>(logins), role.getAbcRoleId(),
                            AbcServiceState.CLOSED, AbcServiceState.DELETED).collect(Collectors.toSet());
                }

                serviceMembers.forEach(responsible -> {
                    if (responsible.getPerson() != null && excessRoles.containsKey(responsible.getPerson().getLogin())) {
                        final Set<Long> services = excessRoles.get(responsible.getPerson().getLogin());
                        if (responsible.getServiceReference() != null && responsible.getServiceReference().getId() != null) {
                            services.remove(responsible.getServiceReference().getId().longValue());
                            if (services.isEmpty()) {
                                excessRoles.remove(responsible.getPerson().getLogin());
                            }
                        }
                    }
                });
            });
        }

    }

    private void skipProjectsInTrash(@Nonnull final Map<String, Map<String, Set<Long>>> excessRolesByRole) {
        excessRolesByRole.values().forEach(excessRoles -> {

            new HashMap<>(excessRoles).forEach((login, services) -> {
                final Map<Long, Project> projects = Hierarchy.get().getProjectReader()
                        .tryReadByAbcServiceIds(services.stream().map(Long::intValue).collect(Collectors.toSet()))
                        .stream().collect(Collectors.toMap(p -> p.getAbcServiceId().longValue(), p -> p, (l, r) -> l));
                new HashSet<>(services).forEach(service -> {
                    if (!projects.containsKey(service)) {
                        return;
                    }
                    final Project project = projects.get(service);
                    if (project.getPathToRoot().stream().anyMatch(p -> Project.TRASH_PROJECT_KEY.equals(p.getPublicKey()))) {
                        services.remove(service);
                        if (services.isEmpty()) {
                            excessRoles.remove(login);
                        }
                    }
                });
            });
        });
    }

    private void skipUnknownServices(@Nonnull final Map<String, Map<String, Set<Long>>> missingRolesByRole) {

        missingRolesByRole.values().forEach(missingRoles -> {

            new HashMap<>(missingRoles).forEach((login, services) -> {
                final Map<Long, Project> projects = Hierarchy.get().getProjectReader()
                        .tryReadByAbcServiceIds(services.stream().map(Long::intValue).collect(Collectors.toSet()))
                        .stream().collect(Collectors.toMap(p -> p.getAbcServiceId().longValue(), p -> p, (l, r) -> l));
                new HashSet<>(services).forEach(service -> {
                    if (!projects.containsKey(service)) {
                        services.remove(service);
                        if (services.isEmpty()) {
                            missingRoles.remove(login);
                        }
                    }
                });
            });

        });
    }

    private void findMissingRoles(final Map<String, Map<String, Set<Long>>> window, final Set<AbcServiceMember> serviceMembers, final String roleKey,
                                  final Map<String, Set<Long>> missingMemberRoles, final Map<String, Set<Long>> serviceMembershipByLogin) {
        serviceMembers.forEach(member -> {
            if (member.getPerson() == null || member.getServiceReference() == null || member.getServiceReference().getId() == null) {
                return;
            }
            final String login = member.getPerson().getLogin();
            final Long serviceId = member.getServiceReference().getId().longValue();
            if (window.containsKey(login)) {
                final Map<String, Set<Long>> roles = window.get(login);
                if (roles.containsKey(roleKey)) {
                    final Set<Long> projectAbcIds = roles.get(roleKey);
                    if (!projectAbcIds.contains(serviceId)) {
                        missingMemberRoles.computeIfAbsent(login, l -> new HashSet<>()).add(serviceId);
                    }
                } else {
                    missingMemberRoles.computeIfAbsent(login, l -> new HashSet<>()).add(serviceId);
                }
            } else {
                missingMemberRoles.computeIfAbsent(login, l -> new HashSet<>()).add(serviceId);
            }
            serviceMembershipByLogin.computeIfAbsent(login, l -> new HashSet<>()).add(serviceId);
        });
    }

    private void findExcessRoles(final Map<String, Map<String, Set<Long>>> window, final Map<String, Map<String, Set<Long>>> servicePersonsByLoginByRole, final Map<String, Map<String, Set<Long>>> excessMemberRolesByRole) {
        window.forEach((login, serviceRoles) -> {
            serviceRoles.forEach((roleKey, services) -> {
                if (servicePersonsByLoginByRole.containsKey(roleKey)) {
                    fillExcessRoles(login, services, servicePersonsByLoginByRole.get(roleKey), excessMemberRolesByRole.get(roleKey));
                }

            });
        });
    }

    private void fillExcessRoles(final String login, final Set<Long> services, final Map<String, Set<Long>> serviceMembershipByLogin,
                                 final Map<String, Set<Long>> excessMemberRoles) {
        if (serviceMembershipByLogin.containsKey(login)) {
            final Set<Long> receivedServices = serviceMembershipByLogin.get(login);
            final Set<Long> excessServices = Sets.difference(services, receivedServices);
            if (!excessServices.isEmpty()) {
                excessMemberRoles.computeIfAbsent(login, l -> new HashSet<>()).addAll(excessServices);
            }
        } else if (!services.isEmpty()) {
            excessMemberRoles.computeIfAbsent(login, l -> new HashSet<>()).addAll(services);
        }
    }

}
