package ru.yandex.qe.dispenser.ws.admin;

import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiPredicate;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Table;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.QuotaView;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.YaGroup;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectUtils;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;
import ru.yandex.qe.dispenser.ws.HtmlPage;

public abstract class AdminPageBase extends HtmlPage {
    private static final Logger LOG = LoggerFactory.getLogger(AdminPageBase.class);

    @NotNull
    protected Table<Resource, Project, QuotaView> getQuotaTable() {
        final Set<QuotaView> allQuotas = Hierarchy.get().getQuotaCache().getAll();
        final Table<Resource, Project, QuotaView> table = HashBasedTable.create();
        allQuotas.stream()
                .filter(QuotaView::isTotal)
                .forEach(q -> table.put(q.getSpec().getKey().getResource(), q.getProject(), q));
        return table;
    }

    @NotNull
    @Override
    public Map<String, Object> getDataModel(@Nullable final String whoami) {
        final String login = Optional.ofNullable(Session.AUTH_LOGIN.get()).orElse(whoami);
        if (login == null) {
            throw new IllegalArgumentException("You are not authorized!");
        }
        final Person user = Hierarchy.get().getPersonReader().readPersonByLogin(login);

        final Set<Service> allServices = Hierarchy.get().getServiceReader().getAll();
        final Set<Project> allProjects = Hierarchy.get().getProjectReader().getAll();
        final Map<Project, Set<Person>> linkedMemberPersons = Hierarchy.get().getProjectReader().getLinkedMembers(allProjects);
        final Map<Project, Set<Person>> linkedResponsibles = Hierarchy.get().getProjectReader().getLinkedResponsibles(allProjects);
        final Map<Project, Set<YaGroup>> linkedMemberGroups = Hierarchy.get().getProjectReader().getLinkedMemberGroups(allProjects);
        final boolean dispenserAdmin = Hierarchy.get().getDispenserAdminsReader().getDispenserAdmins().contains(user);
        final Multimap<Service, Person> serviceAdmins = Hierarchy.get().getServiceReader().getAdmins(allServices);
        final Multimap<Project, Person> responsibles = HashMultimap.create();
        for (final Project project : allProjects) {
            for (final Project parent : project.getPathToRoot()) {
                responsibles.putAll(project, linkedResponsibles.get(parent));
            }
        }
        final Set<Resource> allResources = Hierarchy.get().getResourceReader().getAll();

        return ImmutableMap.<String, Object>builder()
                .put("root", ProjectUtils.root(allProjects))
                .put("allServices", allServices)
                .put("allResources", allResources)
                .put("serviceResources", CollectionUtils.toMultimap(allResources, Resource::getService))
                .put("quotaTable", getQuotaTable())
                .put("user", user)
                .put("linkedResponsibles", linkedResponsibles)
                .put("linkedMemberPersons", linkedMemberPersons)
                .put("linkedMemberGroups", linkedMemberGroups)
                .put("dispenserAdmin", dispenserAdmin)
                .put("isResponsible", (BiPredicate<Person, Project>) (u, p) -> dispenserAdmin || responsibles.get(p).contains(u))
                .put("isServiceAdmin", (BiPredicate<Person, Service>) (u, s) -> dispenserAdmin || serviceAdmins.get(s).contains(u))
                .build();
    }
}
