package ru.yandex.qe.dispenser.ws.admin;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.cli.PosixParser;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiPerformer;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.api.v1.request.DiEntity;
import ru.yandex.qe.dispenser.api.v1.request.DiEntityUsage;
import ru.yandex.qe.dispenser.client.v1.DiOAuthToken;
import ru.yandex.qe.dispenser.client.v1.Dispenser;
import ru.yandex.qe.dispenser.client.v1.builder.BatchQuotaChangeRequestBuilder;
import ru.yandex.qe.dispenser.client.v1.impl.DispenserConfig;
import ru.yandex.qe.dispenser.client.v1.impl.RemoteDispenserFactory;

public enum EntityUploader {
    ;

    private static final Logger LOG = LoggerFactory.getLogger(EntityUploader.class);

    private static final String[] RESOURCES = {"yt-disk", "yt-node", "yt-chunk"};
    private static final String PROJECT = "default";
    private static final DiPerformer PERFORMER = DiPerformer.login("amosov-f").chooses(PROJECT);

    private static final int BATCH_SIZE = 1000;
    private static final int MAX_PARSED_ENTITIES = 10000;
    private static final int MAX_CONCURRENT_REQUESTS = 10;

    private static final Set<String> createdIdSet = new HashSet<>();
    private static final int RECIEVE_TIMEOUT = 100000;

    private static final int PRODUCTION_ENTITY_COUNT = 15_000_000;

    private static final AtomicLong DISK = new AtomicLong(0);
    private static final AtomicLong CHUNK = new AtomicLong(0);
    private static final AtomicLong NODE = new AtomicLong(0);

    @SuppressWarnings("OverlyLongMethod")
    public static void main(@NotNull final String[] args) throws IOException, InterruptedException, ParseException {
        final long start = System.currentTimeMillis();

        final Options options = new Options();
        options.addOption(new Option("h", "host", true, "target host"));
        options.addOption(new Option("t", "token", true, "zombie oauth token"));
        options.addOption(new Option("u", "url", true, "table url"));
        final CommandLineParser clp = new PosixParser();
        final CommandLine cl = clp.parse(options, args);

        final String host = cl.getOptionValue('h');
        final String oauthToken = cl.getOptionValue('t');
        final String url = cl.getOptionValue('u');

        final DispenserConfig config = new DispenserConfig();
        config.setDispenserHost(host);
        config.setServiceZombieOAuthToken(DiOAuthToken.of(oauthToken));
        config.setReceiveTimeout(RECIEVE_TIMEOUT);
        final Dispenser dispenser = new RemoteDispenserFactory(config).get();

        final AtomicInteger totalCounter = new AtomicInteger();
        final AtomicInteger counter = new AtomicInteger();
        final AtomicReference<BatchQuotaChangeRequestBuilder> ref = new AtomicReference<>(dispenser.quotas().changeInService("nirvana"));
        try (BufferedReader reader = new BufferedReader(new FileReader(url))) {
            reader.lines().filter(s -> !s.isEmpty()).forEach(line -> {
                final BatchQuotaChangeRequestBuilder builder = ref.get();

                parse(line, builder);
                counter.incrementAndGet();
                totalCounter.incrementAndGet();

                if (counter.get() > BATCH_SIZE) {
                    boolean isOk = false;
                    ref.get().withReqId("load-" + totalCounter.get());
                    while (!isOk) {
                        try {
                            builder.perform();
                            isOk = true;
                        } catch (Throwable t) {
                            isOk = false;
                            System.out.println(t);
                            System.out.println("sleep for 1 min before retry");
                            try {
                                Thread.sleep(10L);
                            } catch (InterruptedException e) {
                                e.printStackTrace();
                            }
                        }
                    }
                    ref.set(dispenser.quotas().changeInService("nirvana"));
                    counter.set(0);
                    System.out.println(totalCounter + " lines");
                    System.out.println("totl time: " + ((System.currentTimeMillis() - start) / 60000.) + " min");
                    System.out.println("speed: " + (totalCounter.get() * 1. / ((System.currentTimeMillis() - start) / 60000.)));
                    System.out.println("CHUNK " + CHUNK.get());
                    System.out.println("NODE " + NODE.get());
                    System.out.println("DISK " + DISK.get());
                }
            });

            boolean isOk = false;
            ref.get().withReqId("last");
            while (!isOk) {
                try {
                    ref.get().perform();
                    isOk = true;
                } catch (Throwable t) {
                    isOk = false;
                    System.out.println(t);
                    System.out.println("sleep for 1 min before retry");
                    try {
                        Thread.sleep(10L);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                }
            }
        }

        System.out.println("totl time: " + ((System.currentTimeMillis() - start) / 60000.) + " min");
        System.out.println("CHUNK " + CHUNK.get());
        System.out.println("NODE " + NODE.get());
        System.out.println("DISK " + DISK.get());
    }


    private static void parse(@NotNull final String line, @NotNull final BatchQuotaChangeRequestBuilder builder) {
        final String[] parts = line.split("\\s+");
        //uid \t path \t disk_space \t chunk_count \t node_count \t usage_count \t used_by
        final String key = parts[0];
        final int count = Integer.parseInt(parts[5]);
        final long disk = Long.parseLong(parts[2]);
        final long chunks = Long.parseLong(parts[3]);
        final long nodes = Long.parseLong(parts[4]);

        final String user = parts[6].split(",")[0].trim();

        if (createdIdSet.contains(key)) {
            builder.shareEntity(
                    DiEntityUsage.of(DiEntity.withKey(key)
                                    .bySpecification("yt-file")
                                    .occupies("yt-disk", DiAmount.of(disk, DiUnit.BYTE))
                                    .occupies("yt-chunk", DiAmount.of(chunks, DiUnit.COUNT))
                                    .occupies("yt-node", DiAmount.of(nodes, DiUnit.COUNT))
                                    .build(),
                            count), DiPerformer.login(user).chooses("default")
            );
        } else {
            DISK.addAndGet(disk);
            CHUNK.addAndGet(chunks);
            NODE.addAndGet(nodes);

            createdIdSet.add(key);
            builder.createEntity(
                    DiEntity.withKey(key)
                            .bySpecification("yt-file")
                            .occupies("yt-disk", DiAmount.of(disk, DiUnit.BYTE))
                            .occupies("yt-chunk", DiAmount.of(chunks, DiUnit.COUNT))
                            .occupies("yt-node", DiAmount.of(nodes, DiUnit.COUNT))
                            .build(),
                    DiPerformer.login(user).chooses("default")
            );
            if (count > 1) {
                builder.shareEntity(
                        DiEntityUsage.of(DiEntity.withKey(key)
                                        .bySpecification("yt-file")
                                        .occupies("yt-disk", DiAmount.of(disk, DiUnit.BYTE))
                                        .occupies("yt-chunk", DiAmount.of(chunks, DiUnit.COUNT))
                                        .occupies("yt-node", DiAmount.of(nodes, DiUnit.COUNT))
                                        .build(),
                                count - 1), DiPerformer.login(user).chooses("default")
                );
            }
        }
    }


    private static class EntityInfo {
        private final DiEntity entity;
        private final String owner;
        private final Map<String, Long> usages;

        private EntityInfo(final DiEntity entity, final String owner, final Map<String, Long> usages) {
            this.entity = entity;
            this.owner = owner;
            this.usages = usages;
        }

        public String getOwner() {
            return owner;
        }

        public Map<String, Long> getUsages() {
            return usages;
        }

        public DiEntity getEntity() {
            return entity;
        }
    }
}
