package ru.yandex.qe.dispenser.ws.admin;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.ws.rs.DefaultValue;
import javax.ws.rs.POST;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import io.swagger.annotations.Api;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.web.bind.annotation.RequestBody;

import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectReader;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestFilterImpl;
import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.Access;
import ru.yandex.qe.dispenser.ws.base_resources.impl.BaseResourceChangesRefreshManager;
import ru.yandex.qe.dispenser.ws.quota.request.QuotaChangeRequestManager;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.QuotaChangeOwningCostRefreshManager;
import ru.yandex.qe.dispenser.ws.quota.request.owning_cost.campaignOwningCost.CampaignOwningCostRefreshManager;
import ru.yandex.qe.dispenser.ws.quota.request.ticket.QuotaChangeRequestTicketManager;
import ru.yandex.qe.dispenser.ws.quota.request.unbalanced.QuotaChangeUnbalanceRefreshManager;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.PerformerContext;

@Service
@Path("/admin/requests")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class RequestAdminService {
    private static final Logger LOG = LoggerFactory.getLogger(RequestAdminService.class);

    @Autowired
    private ProjectReader projectReader;

    @Autowired
    private QuotaChangeRequestDao quotaChangeRequestDao;

    @Autowired
    private QuotaChangeRequestManager quotaChangeRequestManager;

    @Autowired
    private QuotaChangeRequestTicketManager quotaChangeRequestTicketManager;

    @Autowired
    private BaseResourceChangesRefreshManager baseResourceChangesRefreshManager;

    @Autowired
    private QuotaChangeOwningCostRefreshManager quotaChangeOwningCostRefreshManager;

    @Autowired
    private QuotaChangeUnbalanceRefreshManager quotaChangeUnbalanceRefreshManager;

    @Autowired
    private CampaignOwningCostRefreshManager campaignOwningCostRefreshManager;

    @Autowired
    private CampaignDao campaignDao;

    @POST
    @Path("_refresh/{key}")
    @Access(dispenserAdmin = true)
    public Response refresh(@PathParam("key") @NotNull final String ticketKey,
                            @QueryParam("summonAssignee") @DefaultValue("true") final boolean summonAssignee,
                            @QueryParam("suppressSummon") @DefaultValue("false") final boolean suppressSummon) {

        final QuotaChangeRequest quotaChangeRequest = quotaChangeRequestDao.findByTicketKey(ticketKey)
                .orElseThrow(() -> new IllegalArgumentException("No quotaChangeRequest with ticketKey = " + ticketKey));

        if (quotaChangeRequest.getType() != QuotaChangeRequest.Type.RESOURCE_PREORDER) {
            throw new IllegalArgumentException("Unsupported request type");
        }
        final boolean answer = quotaChangeRequestTicketManager.refreshTicket(quotaChangeRequest, summonAssignee,
                suppressSummon);

        return Response.ok(Collections.singletonMap("success", answer))
                .type(MediaType.APPLICATION_JSON_TYPE)
                .build();
    }

    @POST
    @Path("_moveToProject")
    @Access(dispenserAdmin = true)
    public Response moveRequestsToProject(@QueryParam("suppressSummon") @DefaultValue("false") final boolean suppressSummon,
                                          @RequestBody final MoveRequestsToProjectParams params) {
        final List<QuotaChangeRequest> requests = new ArrayList<>();

        if (params.fromTicketKey != null) {
            quotaChangeRequestDao.findByTicketKey(params.fromTicketKey)
                    .ifPresent(requests::add);
        }

        if (params.fromProject != null) {
            final Project projectFrom = projectReader.readByAbcServiceId(params.fromProject);

            final QuotaChangeRequestFilterImpl filter = new QuotaChangeRequestFilterImpl();
            filter.setProjects(Collections.singleton(projectFrom));

            requests.addAll(quotaChangeRequestDao.read(filter));
        }

        if (requests.isEmpty()) {
            return Response.noContent().build();
        }

        final Person performer = Session.WHOAMI.get();

        final List<QuotaChangeRequestManager.MoveResult> answer = quotaChangeRequestManager.moveToProject(requests, params.toProject,
                new PerformerContext(performer), suppressSummon);
        return Response.ok(SerializationUtils.writeValueAsString(answer))
                .type(MediaType.APPLICATION_JSON_TYPE)
                .build();
    }

    @POST
    @Path("_changeStatus/ids")
    @Access(dispenserAdmin = true)
    public Response changeStatusIds(@QueryParam("toStatus") @NotNull final QuotaChangeRequest.Status toStatus,
                                    @QueryParam("historyComment") @Nullable final String historyComment,
                                    @QueryParam("suppressSummon") @DefaultValue("false") final boolean suppressSummon,
                                    @NotNull final Set<Long> requestIds) {
        if (requestIds == null || requestIds.isEmpty()) {
            throw new IllegalArgumentException("Request ids must be not empty!");
        }
        return changeStatuses(toStatus, historyComment, requestIds, ids -> quotaChangeRequestDao.readForUpdate(ids),
                QuotaChangeRequest::getId, suppressSummon);
    }

    @POST
    @Path("_changeStatus/tickets")
    @Access(dispenserAdmin = true)
    public Response changeStatusTickets(@QueryParam("toStatus") @NotNull final QuotaChangeRequest.Status toStatus,
                                        @QueryParam("historyComment") @Nullable final String historyComment,
                                        @QueryParam("suppressSummon") @DefaultValue("false") final boolean suppressSummon,
                                        @NotNull final Set<String> ticketKeys) {
        if (ticketKeys == null || ticketKeys.isEmpty()) {
            throw new IllegalArgumentException("Ticket keys must be not empty!");
        }
        return changeStatuses(toStatus, historyComment, ticketKeys, keys -> quotaChangeRequestDao.findByTicketKeys(keys),
                QuotaChangeRequest::getTrackerIssueKey, suppressSummon);
    }

    @POST
    @Path("_refreshBaseResourceChanges")
    @Access(dispenserAdmin = true)
    public Response refreshBaseResourceChanges(@QueryParam("campaignIds") Set<Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return Response.noContent().build();
        }
        campaignDao.read(campaignIds);
        baseResourceChangesRefreshManager.refreshBaseResourceChanges(campaignIds);
        return Response.noContent().build();
    }

    @POST
    @Path("_refreshChangesOwningCost")
    @Access(dispenserAdmin = true)
    public Response _refreshChangesOwningCost() {
        new Thread(() -> {
            LOG.info("Refreshing owning cost of changes in admin controller...");
            quotaChangeOwningCostRefreshManager.refresh();
            LOG.info("Successfully refreshed owning cost of changes in admin controller");
        }).start();
        return Response.noContent().build();
    }

    @POST
    @Path("_refreshUnbalanced")
    @Access(dispenserAdmin = true)
    public Response _refreshUnbalanced() {
        new Thread(() -> {
            LOG.info("Refreshing unbalance of quota request in admin controller...");
            quotaChangeUnbalanceRefreshManager.refresh();
            LOG.info("Successfully unbalance of quota request in admin controller");
        }).start();
        return Response.noContent().build();
    }

    @POST
    @Path("_refreshCampaignOwningCost")
    @Access(dispenserAdmin = true)
    public Response _refreshCampaignOwningCost() {
        new Thread(() -> {
            LOG.info("Refreshing campaign owning cost in admin controller...");
            campaignOwningCostRefreshManager.refresh();
            LOG.info("Successfully campaign refreshed owning cost in admin controller");
        }).start();
        return Response.noContent().build();
    }

    private <T> Response changeStatuses(@NotNull final QuotaChangeRequest.Status toStatus,
                                        @Nullable final String historyComment,
                                        @NotNull final Set<T> requests,
                                        @NotNull final Function<Set<T>, Map<T, QuotaChangeRequest>> requestExtractor,
                                        @NotNull final Function<QuotaChangeRequest, T> idExtractor,
                                        final boolean suppressSummon) {
        final Map<T, QuotaChangeRequest> requestsById = requestExtractor.apply(requests);

        final Map<Boolean, List<QuotaChangeRequest>> requestsByStatusChange = requestsById.values()
                .stream()
                .collect(Collectors.partitioningBy(qr -> qr.getStatus() != toStatus));

        final PerformerContext ctx = new PerformerContext(Session.WHOAMI.get());
        ctx.setComment(historyComment);
        quotaChangeRequestManager.setStatusBatch(requestsByStatusChange.get(true), toStatus, ctx, suppressSummon);

        final Set<T> missing = Sets.difference(requests, requestsById.keySet());
        return Response.ok(ImmutableMap.builder()
                .put("missed", missing)
                .put("skipped", requestsByStatusChange.get(false).stream().map(idExtractor).collect(Collectors.toList()))
                .build())
                .type(MediaType.APPLICATION_JSON_TYPE)
                .build();
    }

    public static class MoveRequestsToProjectParams {
        private final Integer fromProject;
        private final String fromTicketKey;
        private final int toProject;

        @JsonCreator
        public MoveRequestsToProjectParams(
                @JsonProperty(value = "fromProject") final Integer fromProject,
                @JsonProperty("fromTicketKey") final String fromTicketKey,
                @JsonProperty(value = "toProject", required = true) final int toProject) {
            this.fromProject = fromProject;
            this.fromTicketKey = fromTicketKey;
            this.toProject = toProject;
        }

        public Integer getFromProject() {
            return fromProject;
        }

        public String getFromTicketKey() {
            return fromTicketKey;
        }

        public int getToProject() {
            return toProject;
        }
    }
}
