package ru.yandex.qe.dispenser.ws.api.domain.distribution;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.ws.api.model.distribution.ResourceDistributionAlgorithm;

public class DistributeQuotaParams {

    @NotNull
    private final ResourceDistributionAlgorithm algorithm;
    @NotNull
    private final Service service;
    @NotNull
    private final BigOrder bigOrder;
    @NotNull
    private final Campaign campaign;
    @Nullable
    private final String comment;
    private final boolean allocate;
    @NotNull
    private final List<Change> changes;

    private DistributeQuotaParams(@NotNull final ResourceDistributionAlgorithm algorithm,
                                  @NotNull final Service service,
                                  @NotNull final BigOrder bigOrder,
                                  @NotNull final Campaign campaign,
                                  @Nullable final String comment,
                                  final boolean allocate,
                                  @NotNull final List<Change> changes) {
        this.algorithm = algorithm;
        this.service = service;
        this.bigOrder = bigOrder;
        this.campaign = campaign;
        this.comment = comment;
        this.allocate = allocate;
        this.changes = changes;
    }

    @NotNull
    public static Builder builder() {
        return new Builder();
    }

    @NotNull
    public ResourceDistributionAlgorithm getAlgorithm() {
        return algorithm;
    }

    @NotNull
    public Service getService() {
        return service;
    }

    @NotNull
    public BigOrder getBigOrder() {
        return bigOrder;
    }

    @NotNull
    public Campaign getCampaign() {
        return campaign;
    }

    @Nullable
    public String getComment() {
        return comment;
    }

    public boolean isAllocate() {
        return allocate;
    }

    @NotNull
    public List<Change> getChanges() {
        return changes;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final DistributeQuotaParams that = (DistributeQuotaParams) o;
        return allocate == that.allocate &&
                algorithm == that.algorithm &&
                service.equals(that.service) &&
                bigOrder.equals(that.bigOrder) &&
                campaign.equals(that.campaign) &&
                Objects.equals(comment, that.comment) &&
                changes.equals(that.changes);
    }

    @Override
    public int hashCode() {
        return Objects.hash(algorithm, service, bigOrder, campaign, comment, allocate, changes);
    }

    @Override
    public String toString() {
        return "DistributeQuotaParams{" +
                "algorithm=" + algorithm +
                ", service=" + service +
                ", bigOrder=" + bigOrder +
                ", campaign=" + campaign +
                ", comment='" + comment + '\'' +
                ", allocate=" + allocate +
                ", changes=" + changes +
                '}';
    }

    public static final class Change {

        @NotNull
        private final Resource resource;
        @NotNull
        private final Set<Segment> segments;
        private final long amountReady;

        private Change(@NotNull final Resource resource,
                      @NotNull final Set<Segment> segments,
                      final long amountReady) {
            this.resource = resource;
            this.segments = segments;
            this.amountReady = amountReady;
        }

        @NotNull
        public static Builder builder() {
            return new Builder();
        }

        @NotNull
        public Resource getResource() {
            return resource;
        }

        @NotNull
        public Set<Segment> getSegments() {
            return segments;
        }

        public long getAmountReady() {
            return amountReady;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final Change change = (Change) o;
            return amountReady == change.amountReady &&
                    resource.equals(change.resource) &&
                    segments.equals(change.segments);
        }

        @Override
        public int hashCode() {
            return Objects.hash(resource, segments, amountReady);
        }

        @Override
        public String toString() {
            return "Change{" +
                    "resource=" + resource +
                    ", segments=" + segments +
                    ", amountReady=" + amountReady +
                    '}';
        }

        public static final class Builder {

            @Nullable
            private Resource resource;
            @NotNull
            private Set<Segment> segments = new HashSet<>();
            @Nullable
            private Long amountReady;

            private Builder() {
            }

            @NotNull
            public Builder resource(@NotNull final Resource resource) {
                Objects.requireNonNull(resource, "Resource is required.");
                this.resource = resource;
                return this;
            }

            @NotNull
            public Builder segments(@NotNull final Set<Segment> segments) {
                Objects.requireNonNull(segments, "Segments are required.");
                this.segments = segments;
                return this;
            }

            @NotNull
            public Builder amountReady(final long amountReady) {
                this.amountReady = amountReady;
                return this;
            }

            @Nullable
            public Resource getResource() {
                return resource;
            }

            @NotNull
            public Set<Segment> getSegments() {
                return segments;
            }

            @Nullable
            public Long getAmountReady() {
                return amountReady;
            }

            @NotNull
            public Change build() {
                Objects.requireNonNull(resource, "Resource is required.");
                Objects.requireNonNull(amountReady, "Amount ready is required.");
                return new Change(resource, segments, amountReady);
            }

            @Override
            public boolean equals(final Object o) {
                if (this == o) {
                    return true;
                }
                if (o == null || getClass() != o.getClass()) {
                    return false;
                }
                final Builder builder = (Builder) o;
                return Objects.equals(resource, builder.resource) &&
                        segments.equals(builder.segments) &&
                        Objects.equals(amountReady, builder.amountReady);
            }

            @Override
            public int hashCode() {
                return Objects.hash(resource, segments, amountReady);
            }

            @Override
            public String toString() {
                return "Builder{" +
                        "resource=" + resource +
                        ", segments=" + segments +
                        ", amountReady=" + amountReady +
                        '}';
            }
        }

    }

    public static final class Builder {

        @Nullable
        private ResourceDistributionAlgorithm algorithm;
        @Nullable
        private Service service;
        @Nullable
        private BigOrder bigOrder;
        @Nullable
        private Campaign campaign;
        @Nullable
        private String comment;
        private boolean allocate = false;
        @NotNull
        private final List<Change> changes = new ArrayList<>();

        private Builder() {
        }

        @NotNull
        public Builder algorithm(@NotNull final ResourceDistributionAlgorithm algorithm) {
            Objects.requireNonNull(algorithm, "Algorithm is required.");
            this.algorithm = algorithm;
            return this;
        }

        @NotNull
        public Builder service(@NotNull final Service service) {
            Objects.requireNonNull(service, "Service is required.");
            this.service = service;
            return this;
        }

        @NotNull
        public Builder bigOrder(@NotNull final BigOrder bigOrder) {
            Objects.requireNonNull(bigOrder, "Big order is required.");
            this.bigOrder = bigOrder;
            return this;
        }

        @NotNull
        public Builder campaign(@NotNull final Campaign campaign) {
            Objects.requireNonNull(campaign, "Campaign is required.");
            this.campaign = campaign;
            return this;
        }

        @NotNull
        public Builder comment(@Nullable final String comment) {
            this.comment = comment;
            return this;
        }

        @NotNull
        public Builder allocate(final boolean allocate) {
            this.allocate = allocate;
            return this;
        }

        @NotNull
        public Builder addChange(@NotNull final Change change) {
            Objects.requireNonNull(change, "Change is required.");
            this.changes.add(change);
            return this;
        }

        @Nullable
        public ResourceDistributionAlgorithm getAlgorithm() {
            return algorithm;
        }

        @Nullable
        public Service getService() {
            return service;
        }

        @Nullable
        public BigOrder getBigOrder() {
            return bigOrder;
        }

        @Nullable
        public Campaign getCampaign() {
            return campaign;
        }

        @Nullable
        public String getComment() {
            return comment;
        }

        public boolean isAllocate() {
            return allocate;
        }

        @NotNull
        public List<Change> getChanges() {
            return changes;
        }

        @NotNull
        public DistributeQuotaParams build() {
            Objects.requireNonNull(algorithm, "Algorithm is required.");
            Objects.requireNonNull(service, "Service is required.");
            Objects.requireNonNull(bigOrder, "Big order is required.");
            Objects.requireNonNull(campaign, "Campaign is required.");
            return new DistributeQuotaParams(algorithm, service, bigOrder, campaign, comment, allocate, changes);
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final Builder builder = (Builder) o;
            return allocate == builder.allocate &&
                    algorithm == builder.algorithm &&
                    Objects.equals(service, builder.service) &&
                    Objects.equals(bigOrder, builder.bigOrder) &&
                    Objects.equals(campaign, builder.campaign) &&
                    Objects.equals(comment, builder.comment) &&
                    changes.equals(builder.changes);
        }

        @Override
        public int hashCode() {
            return Objects.hash(algorithm, service, bigOrder, campaign, comment, allocate, changes);
        }

        @Override
        public String toString() {
            return "Builder{" +
                    "algorithm=" + algorithm +
                    ", service=" + service +
                    ", bigOrder=" + bigOrder +
                    ", campaign=" + campaign +
                    ", comment='" + comment + '\'' +
                    ", allocate=" + allocate +
                    ", changes=" + changes +
                    '}';
        }

    }

}
