package ru.yandex.qe.dispenser.ws.api.impl

import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.ExperimentalCoroutinesApi
import kotlinx.coroutines.FlowPreview
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.asFlow
import kotlinx.coroutines.flow.flatMapMerge
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.toList
import kotlinx.coroutines.runBlocking
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.context.MessageSource
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Propagation
import org.springframework.transaction.annotation.Transactional
import ru.yandex.qe.dispenser.api.v1.DiAmount
import ru.yandex.qe.dispenser.domain.Person
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest
import ru.yandex.qe.dispenser.domain.d.DeliveryOperationStatusDto
import ru.yandex.qe.dispenser.domain.d.DeliveryStatusDto
import ru.yandex.qe.dispenser.domain.dao.delivery.DeliveryDao
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier
import ru.yandex.qe.dispenser.domain.hierarchy.Role
import ru.yandex.qe.dispenser.domain.i18n.LocalizableString
import ru.yandex.qe.dispenser.domain.resources_model.QuotaRequestDelivery
import ru.yandex.qe.dispenser.domain.resources_model.QuotaRequestDeliveryResolveStatus
import ru.yandex.qe.dispenser.domain.util.LocalizationUtils
import ru.yandex.qe.dispenser.ws.bot.BigOrderManager
import ru.yandex.qe.dispenser.ws.common.domain.errors.ErrorCollection
import ru.yandex.qe.dispenser.ws.common.domain.errors.TypedError
import ru.yandex.qe.dispenser.ws.common.domain.result.Result
import ru.yandex.qe.dispenser.ws.d.DApiHelper
import ru.yandex.qe.dispenser.ws.intercept.SessionInitializer
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.PerformerContext
import ru.yandex.qe.dispenser.ws.reqbody.DeliveryStatusRequest
import ru.yandex.qe.dispenser.ws.reqbody.DeliveryStatusResponse
import ru.yandex.qe.dispenser.ws.reqbody.QuotaRequestDeliveryAmount
import ru.yandex.qe.dispenser.ws.reqbody.QuotaRequestDeliveryOperation
import ru.yandex.qe.dispenser.ws.reqbody.QuotaRequestDeliveryOperationStatus
import ru.yandex.qe.dispenser.ws.reqbody.QuotaRequestDeliverySegment
import ru.yandex.qe.dispenser.ws.reqbody.QuotaRequestDeliveryStatus
import ru.yandex.qe.dispenser.ws.reqbody.QuotaRequestDeliveryStatusPerDelivery
import ru.yandex.qe.dispenser.ws.reqbody.QuotaRequestPendingDeliveryResponse
import ru.yandex.qe.dispenser.ws.reqbody.QuotaRequestPendingDeliveryStatus
import java.time.format.DateTimeFormatter
import java.util.*
import javax.inject.Inject

@Component
open class AllocationStatusManager @Inject constructor(
    @Qualifier("errorMessageSource") private val errorMessageSource: MessageSource,
    private val quotaChangeRequestDao: QuotaChangeRequestDao,
    private val dApiHelper: DApiHelper,
    private val deliveryDao: DeliveryDao,
    private val hierarchySupplier: HierarchySupplier,
    private val bigOrderManager: BigOrderManager
) {

    @OptIn(ExperimentalCoroutinesApi::class)
    private val ioDispatcher: CoroutineDispatcher = Dispatchers.IO.limitedParallelism(15)

    @Transactional(propagation = Propagation.REQUIRED)
    open fun getDeliveryStatuses(
        request: DeliveryStatusRequest?,
        performerContext: PerformerContext,
        locale: Locale
    ): Result<DeliveryStatusResponse, ErrorCollection<String, TypedError<String>>> {
        return validate(request, locale).apply {validated -> applyGetStatus(validated, locale)}
    }

    @Transactional(propagation = Propagation.REQUIRED)
    open fun getPendingDeliveries(
        limit: Long,
        from: String?,
        performerContext: PerformerContext,
        locale: Locale
    ): Result<QuotaRequestPendingDeliveryResponse, ErrorCollection<String, TypedError<String>>> {
        if (!hasPermissionsForPending(hierarchySupplier.get(), performerContext.person)) {
            return Result.failure(ErrorCollection.typedStringBuilder()
                .addError(TypedError.forbidden(localize(locale, "not.enough.permissions")))
                .build())
        }
        if (limit > 1000 || limit <= 0) {
            return Result.failure(ErrorCollection.typedStringBuilder()
                .addError(TypedError.badRequest(localize(locale, "invalid.limit.value")))
                .build())
        }
        if (from != null && !isValidUuid(from)) {
            return Result.failure(ErrorCollection.typedStringBuilder()
                .addError(TypedError.badRequest(localize(locale, "invalid.delivery.id")))
                .build())
        }
        return Result.success(applyGetPendingDeliveries(limit, from, locale))
    }

    private fun applyGetPendingDeliveries(
        limit: Long,
        from: String?,
        locale: Locale
    ): QuotaRequestPendingDeliveryResponse {
        val hierarchy = hierarchySupplier.get()
        val deliveries = deliveryDao.getUnresolved(limit + 1, from?.let { UUID.fromString(it) })
        val statuses = loadAllStatuses(deliveries.map { d -> d.id.toString() }, toLanguage(locale))
        val statusesByDelivery = statuses.groupBy { s -> s.deliveryId }
        val result = mutableListOf<QuotaRequestPendingDeliveryStatus>()
        val (page, hasNext) = if (deliveries.size.toLong() == limit + 1) {
            Pair(deliveries.dropLast(1), true)
        } else {
            Pair(deliveries, false)
        }
        page.forEach {d ->
            val amountsResult = mutableListOf<QuotaRequestDeliveryAmount>()
            val operationsResult = mutableListOf<QuotaRequestDeliveryOperation>()
            d.internalResources.forEach { ir ->
                val resource = hierarchy.resourceReader.read(ir.resourceId)
                val segments = hierarchy.segmentReader.readByIds(ir.segmentIds)
                    .map { QuotaRequestDeliverySegment(it.publicKey, it.name, it.segmentation.key.publicKey,
                        it.segmentation.name) }
                val bigOrder = bigOrderManager.getByIdCached(ir.bigOrderId)!!
                amountsResult.add(QuotaRequestDeliveryAmount(resource.publicKey, resource.name,
                    resource.service.key, resource.service.name, segments, ir.bigOrderId,
                    bigOrder.date.format(DateTimeFormatter.ISO_LOCAL_DATE), DiAmount.of(ir.amount, resource.type.baseUnit)))
            }
            (statusesByDelivery[d.id.toString()] ?: listOf()).forEach { s ->
                s.operations.forEach { op ->
                    operationsResult.add(QuotaRequestDeliveryOperation(op.operationId,
                        toOperationStatus(op.status), op.errorMessage))
                }
            }
            result.add(QuotaRequestPendingDeliveryStatus(d.id.toString(), d.quotaRequestId, d.resolveStatus != QuotaRequestDeliveryResolveStatus.RESOLVED,
                amountsResult.toList(), operationsResult.toList()))
        }
        return QuotaRequestPendingDeliveryResponse(result.toList(),
            if (hasNext) { page.last().id.toString() } else { null })
    }

    private fun applyGetStatus(request: ValidatedDeliveryStatusRequest, locale: Locale): DeliveryStatusResponse {
        val hierarchy = hierarchySupplier.get()
        val deliveries = deliveryDao.getByRequestIds(request.quotaChangeRequests.map { r -> r.id }.toSet())
        val statuses = loadAllStatuses(deliveries.map { d -> d.id.toString() }, toLanguage(locale))
        val deliveriesByRequestId = deliveries.groupBy { d -> d.quotaRequestId }
        val statusesByDelivery = statuses.groupBy { s -> s.deliveryId }
        val result = mutableListOf<QuotaRequestDeliveryStatus>()
        request.quotaChangeRequests.forEach { r ->
            val deliveriesResult = mutableListOf<QuotaRequestDeliveryStatusPerDelivery>()
            (deliveriesByRequestId[r.id] ?: listOf<QuotaRequestDelivery>()).forEach { d ->
                val amountsResult = mutableListOf<QuotaRequestDeliveryAmount>()
                val operationsResult = mutableListOf<QuotaRequestDeliveryOperation>()
                d.internalResources.forEach { ir ->
                    val resource = hierarchy.resourceReader.read(ir.resourceId)
                    val segments = hierarchy.segmentReader.readByIds(ir.segmentIds)
                        .map { QuotaRequestDeliverySegment(it.publicKey, it.name, it.segmentation.key.publicKey,
                            it.segmentation.name) }
                    val bigOrder = bigOrderManager.getByIdCached(ir.bigOrderId)!!
                    amountsResult.add(QuotaRequestDeliveryAmount(resource.publicKey, resource.name,
                        resource.service.key, resource.service.name, segments, ir.bigOrderId,
                        bigOrder.date.format(DateTimeFormatter.ISO_LOCAL_DATE), DiAmount.of(ir.amount, resource.type.baseUnit)))
                }
                (statusesByDelivery[d.id.toString()] ?: listOf()).forEach { s ->
                    s.operations.forEach { op ->
                        operationsResult.add(QuotaRequestDeliveryOperation(op.operationId,
                            toOperationStatus(op.status), op.errorMessage))
                    }
                }
                deliveriesResult.add(QuotaRequestDeliveryStatusPerDelivery(d.id.toString(), d.resolveStatus != QuotaRequestDeliveryResolveStatus.RESOLVED,
                    amountsResult.toList(), operationsResult.toList()))
            }
            result.add(QuotaRequestDeliveryStatus(r.id, deliveriesResult.toList()))
        }
        return DeliveryStatusResponse(result.toList())
    }

    private fun toOperationStatus(status: DeliveryOperationStatusDto): QuotaRequestDeliveryOperationStatus {
        return when (status) {
            DeliveryOperationStatusDto.UNKNOWN_STATUS_VALUE -> QuotaRequestDeliveryOperationStatus.UNKNOWN_STATUS_VALUE
            DeliveryOperationStatusDto.IN_PROGRESS -> QuotaRequestDeliveryOperationStatus.IN_PROGRESS
            DeliveryOperationStatusDto.SUCCESS -> QuotaRequestDeliveryOperationStatus.SUCCESS
            DeliveryOperationStatusDto.FAILURE -> QuotaRequestDeliveryOperationStatus.FAILURE
        }
    }

    @OptIn(FlowPreview::class)
    private fun loadAllStatuses(deliveryIds: List<String>, language: String): List<DeliveryStatusDto> {
        if (deliveryIds.isEmpty()) {
            return listOf()
        }
        return runBlocking (ioDispatcher) {
            deliveryIds.chunked(1000).asFlow().flatMapMerge(concurrency = 10) { p -> loadStatusesPage(p, language) }
                .flowOn(ioDispatcher).toList().flatten()
        }
    }

    private fun loadStatusesPage(deliveryIds: List<String>, language: String): Flow<List<DeliveryStatusDto>> = flow {
        emit(dApiHelper.getStatuses(language, deliveryIds).deliveries ?: listOf())
    }.flowOn(ioDispatcher)

    private fun validate(
        request: DeliveryStatusRequest?,
        locale: Locale
    ): Result<ValidatedDeliveryStatusRequest, ErrorCollection<String, TypedError<String>>> {
        val errors = ErrorCollection.typedStringBuilder()
        var quotaRequests = listOf<QuotaChangeRequest>()
        if (request?.quotaRequestIds == null || request.quotaRequestIds!!.isEmpty()) {
            addFieldValidationError(errors, "quotaRequestIds", locale, "field.is.required")
        } else {
            for (i in request.quotaRequestIds!!.indices) {
                if (request.quotaRequestIds!![i] == null) {
                    addFieldValidationError(errors, "quotaRequestIds.$i", locale, "field.is.required")
                }
            }
            if (request.quotaRequestIds!!.size > 1000) {
                addFieldValidationError(errors, "quotaRequestIds", locale, "request.is.too.large")
            } else {
                val loadedQuotaRequests = quotaChangeRequestDao.read(request.quotaRequestIds!!.filterNotNull().distinct())
                val missingIds = request.quotaRequestIds!!.toHashSet().minus(loadedQuotaRequests.keys)
                if (missingIds.isNotEmpty()) {
                    for (i in request.quotaRequestIds!!.indices) {
                        if (missingIds.contains(request.quotaRequestIds!![i])) {
                            addFieldValidationError(errors, "quotaRequestIds.$i", locale,
                                "quota.change.request.not.found")
                        }
                    }
                } else {
                    quotaRequests = loadedQuotaRequests.values.toList()
                }
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build())
        }
        return Result.success(ValidatedDeliveryStatusRequest(quotaRequests))
    }

    private fun addFieldValidationError(
        errorsBuilder: ErrorCollection.Builder<String, TypedError<String>>,
        fieldKey: String,
        locale: Locale,
        messageKey: String,
        vararg args: Any
    ) {
        errorsBuilder.addError(fieldKey, TypedError.invalid(LocalizationUtils
            .resolveWithDefaultAsKey(errorMessageSource, LocalizableString.of(messageKey, *args), locale)))
    }

    private fun localize(
        locale: Locale,
        messageKey: String,
        vararg args: Any): String {
        return LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
            LocalizableString.of(messageKey, *args), locale)
    }

    private fun toLanguage(locale: Locale): String {
        return if (locale == SessionInitializer.RU) {
            "ru_RU"
        } else {
            "en_US"
        }
    }

    private fun isValidUuid(value: String?): Boolean {
        return if (value == null || value.isBlank()) {
            false
        } else try {
            UUID.fromString(value)
            true
        } catch (e: IllegalArgumentException) {
            false
        }
    }

    private fun hasPermissionsForPending(hierarchy: Hierarchy, person: Person): Boolean {
        return isUserProcessResponsible(hierarchy, person) || isProvidersAdmin(hierarchy, person)
            || isDispenserAdmin(hierarchy, person)
    }

    private fun isUserProcessResponsible(hierarchy: Hierarchy, person: Person): Boolean {
        return hierarchy.projectReader.hasRole(person, hierarchy.projectReader.root, Role.PROCESS_RESPONSIBLE)
    }

    private fun isProvidersAdmin(hierarchy: Hierarchy, person: Person): Boolean {
        return hierarchy.serviceReader.getAdminServices(person).isNotEmpty()
    }

    private fun isDispenserAdmin(hierarchy: Hierarchy, person: Person): Boolean {
        return hierarchy.dispenserAdminsReader.dispenserAdmins.contains(person)
    }

    data class ValidatedDeliveryStatusRequest(
        val quotaChangeRequests: List<QuotaChangeRequest>
    )

}
