package ru.yandex.qe.dispenser.ws.api.impl;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.CannotAcquireLockException;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeRequest;
import ru.yandex.qe.dispenser.api.v1.DiQuotaRequestHistoryEventType;
import ru.yandex.qe.dispenser.api.v1.DiResourceType;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequestHistoryEvent;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.domain.dao.history.request.QuotaChangeRequestHistoryDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeInRequest;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestFilterImpl;
import ru.yandex.qe.dispenser.domain.dao.quota.request.ResourceSegments;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.util.BigOrderLegacy;
import ru.yandex.qe.dispenser.quartz.trigger.QuartzTrackerComment;
import ru.yandex.qe.dispenser.ws.ResourcePreorderChangeManager;
import ru.yandex.qe.dispenser.ws.ResourcePreorderRequestUtils;
import ru.yandex.qe.dispenser.ws.api.domain.distribution.DistributeQuotaParams;
import ru.yandex.qe.dispenser.ws.api.domain.distribution.QuotaDistribution;
import ru.yandex.qe.dispenser.ws.api.model.distribution.DistributableQuota;
import ru.yandex.qe.dispenser.ws.api.model.distribution.DistributedQuota;
import ru.yandex.qe.dispenser.ws.api.model.distribution.DistributedQuotaDeltas;
import ru.yandex.qe.dispenser.ws.api.model.distribution.QuotaDistributionPlan;
import ru.yandex.qe.dispenser.ws.api.model.distribution.QuotaDistributionRemainder;
import ru.yandex.qe.dispenser.ws.api.model.distribution.ResourceDistributionAlgorithm;
import ru.yandex.qe.dispenser.ws.common.domain.errors.ErrorCollection;
import ru.yandex.qe.dispenser.ws.common.domain.errors.TypedError;
import ru.yandex.qe.dispenser.ws.common.domain.exceptions.TooManyRequestsException;
import ru.yandex.qe.dispenser.ws.common.domain.result.Result;
import ru.yandex.qe.dispenser.ws.quota.request.QuotaChangeRequestManager;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.PerformerContext;

@Component
public class QuotaDistributionManager {

    private static final Logger LOG = LoggerFactory.getLogger(QuotaDistributionManager.class);

    private static final Map<DiResourceType, DiUnit> BASE_UNITS = ImmutableMap.<DiResourceType, DiUnit>builder()
            .put(DiResourceType.ENUMERABLE, DiUnit.COUNT)
            .put(DiResourceType.PROCESSOR, DiUnit.CORES)
            .put(DiResourceType.MEMORY, DiUnit.MEBIBYTE)
            .put(DiResourceType.STORAGE, DiUnit.MEBIBYTE)
            .put(DiResourceType.STORAGE_BASE, DiUnit.GIBIBYTE_BASE)
            .put(DiResourceType.MEMORY_BASE, DiUnit.GIBIBYTE_BASE)
            .build();
    private static final Map<String, Map<DiResourceType, DiUnit>> SERVICE_BASE_UNITS = ImmutableMap.<String, Map<DiResourceType, DiUnit>>builder()
            .put("logbroker", ImmutableMap.<DiResourceType, DiUnit>builder()
                    .put(DiResourceType.TRAFFIC, DiUnit.MBPS)
                    .build())
            .put("dbaas", ImmutableMap.<DiResourceType, DiUnit>builder()
                    .put(DiResourceType.STORAGE, DiUnit.GIBIBYTE)
                    .put(DiResourceType.MEMORY, DiUnit.GIBIBYTE)
                    .build())
            .build();
    public static final int UPDATE_PARTITION_SIZE = 300;
    public static final int LOAD_PARTITION_SIZE = 300;
    @NotNull
    private final QuotaChangeRequestDao quotaChangeRequestDao;
    @NotNull
    private final QuotaChangeRequestHistoryDao quotaChangeRequestHistoryDao;
    @NotNull
    private final ResourcePreorderChangeManager preorderChangeManager;
    @NotNull
    private final QuotaChangeRequestManager quotaChangeRequestManager;
    @NotNull
    private final ResourcePreorderRequestUtils resourcePreorderRequestUtils;
    @NotNull
    private final QuartzTrackerComment quartzTrackerCommentTrigger;

    @Inject
    public QuotaDistributionManager(@NotNull final QuotaChangeRequestDao quotaChangeRequestDao,
                                    @NotNull final QuotaChangeRequestHistoryDao quotaChangeRequestHistoryDao,
                                    @NotNull final ResourcePreorderChangeManager preorderChangeManager,
                                    @NotNull final QuotaChangeRequestManager quotaChangeRequestManager,
                                    @NotNull final ResourcePreorderRequestUtils resourcePreorderRequestUtils,
                                    @NotNull final QuartzTrackerComment quartzTrackerCommentTrigger) {
        this.quotaChangeRequestDao = quotaChangeRequestDao;
        this.quotaChangeRequestHistoryDao = quotaChangeRequestHistoryDao;
        this.preorderChangeManager = preorderChangeManager;
        this.quotaChangeRequestManager = quotaChangeRequestManager;
        this.resourcePreorderRequestUtils = resourcePreorderRequestUtils;
        this.quartzTrackerCommentTrigger = quartzTrackerCommentTrigger;
    }

    @NotNull
    @Transactional(propagation = Propagation.MANDATORY)
    public Result<QuotaDistribution, ErrorCollection<String, TypedError<String>>> planAndLockDistribution(
            @NotNull final DistributeQuotaParams distributeQuotaParams) {
        return planDistribution(distributeQuotaParams, true);
    }

    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public Result<QuotaDistribution, ErrorCollection<String, TypedError<String>>> planDistribution(
            @NotNull final DistributeQuotaParams distributeQuotaParams) {
        return planDistribution(distributeQuotaParams, false);
    }

    @NotNull
    public QuotaDistributionPlan showDistributionPlan(@NotNull final QuotaDistribution quotaDistribution,
                                                      @NotNull final DistributeQuotaParams distributeQuotaParams) {
        return prepareQuotaDistributionPlan(quotaDistribution, distributeQuotaParams);
    }

    @Transactional(propagation = Propagation.MANDATORY)
    public void distributeQuota(@NotNull final QuotaDistribution quotaDistribution,
                                @NotNull final PerformerContext performerContext,
                                final boolean suppressSummon) {
        final List<QuotaDistribution.Change> changes = quotaDistribution.getChanges().stream()
                .filter(c -> c.getAmountAllocatedIncrement() > 0 || c.getAmountReadyIncrement() > 0)
                .collect(Collectors.toList());
        final List<QuotaDistribution.Change> incrementReady = changes.stream()
                .filter(c -> c.getAmountReadyIncrement() > 0L && c.getAmountAllocatedIncrement() == 0L)
                .collect(Collectors.toList());
        final List<QuotaDistribution.Change> incrementAllocated = changes.stream()
                .filter(c -> c.getAmountReadyIncrement() == 0L && c.getAmountAllocatedIncrement() > 0L)
                .collect(Collectors.toList());
        final List<QuotaDistribution.Change> incrementReadyAndAllocated = changes.stream()
                .filter(c -> c.getAmountReadyIncrement() > 0L && c.getAmountAllocatedIncrement() > 0L)
                .collect(Collectors.toList());
        final List<Long> requestIdsToLoadChanges = changes.stream()
                .map(QuotaDistribution.Change::getRequestId).distinct().collect(Collectors.toList());
        final Map<Long, List<QuotaChangeRequest.Change>> originalChanges = new HashMap<>();
        final Map<Long, Pair<QuotaChangeRequest.Status, String>> originalRequests = new HashMap<>();
        Lists.partition(requestIdsToLoadChanges, LOAD_PARTITION_SIZE).forEach(partition -> {
            originalChanges.putAll(quotaChangeRequestDao.selectChangesByRequestIds(partition));
            originalRequests.putAll(quotaChangeRequestDao.selectStatusAndIssueByRequestIds(partition));
        });
        if (!incrementReady.isEmpty()) {
            Lists.partition(incrementReady, UPDATE_PARTITION_SIZE).forEach(partition -> {
                final Map<Long, Long> increments = partition.stream().collect(Collectors
                        .toMap(QuotaDistribution.Change::getId, QuotaDistribution.Change::getAmountReadyIncrement));
                handleLockingFailures(() -> quotaChangeRequestDao.incrementChangesReadyAmount(increments));
            });
        }
        if (!incrementAllocated.isEmpty()) {
            Lists.partition(incrementAllocated, UPDATE_PARTITION_SIZE).forEach(partition -> {
                final Map<Long, Long> increments = partition.stream().collect(Collectors
                        .toMap(QuotaDistribution.Change::getId, QuotaDistribution.Change::getAmountAllocatedIncrement));
                handleLockingFailures(() -> quotaChangeRequestDao.incrementChangesAllocatedAmount(increments));
            });
        }
        if (!incrementReadyAndAllocated.isEmpty()) {
            Lists.partition(incrementReadyAndAllocated, UPDATE_PARTITION_SIZE).forEach(partition -> {
                final Map<Long, Pair<Long, Long>> increments = partition.stream().collect(Collectors
                        .toMap(QuotaDistribution.Change::getId,
                                c -> Pair.of(c.getAmountReadyIncrement(), c.getAmountAllocatedIncrement())));
                handleLockingFailures(() -> quotaChangeRequestDao.incrementChangesReadyAndAllocatedAmount(increments));
            });
        }
        final Map<Long, Set<QuotaDistribution.Change>> changesByRequest = changes.stream()
                .collect(Collectors.groupingBy(QuotaDistribution.Change::getRequestId, Collectors.toSet()));
        final List<QuotaChangeRequestHistoryEvent> historyEvents = prepareHistoryEvents(quotaDistribution, changesByRequest,
                originalChanges, performerContext);
        if (!historyEvents.isEmpty()) {
            quotaChangeRequestHistoryDao.create(historyEvents);
        }
        // Do not reset 'ready for allocation' flag.
        // This flag should be set to false for requests with updated 'amount allocated' when service supports manual allocation.
        // But allocation during automatic distribution is forbidden for service with manual allocation.
        updateRequestCompletion(quotaDistribution, changesByRequest, originalChanges, originalRequests, performerContext,
                suppressSummon);
        addIssueComments(quotaDistribution, changesByRequest, originalRequests, performerContext);
    }

    private void addIssueComments(@NotNull final QuotaDistribution quotaDistribution,
                                  @NotNull final Map<Long, Set<QuotaDistribution.Change>> changesByRequest,
                                  @NotNull final Map<Long, Pair<QuotaChangeRequest.Status, String>> originalRequests,
                                  @NotNull PerformerContext performerContext) {
        changesByRequest.forEach((requestId, changes) -> {
            final String issueKey = originalRequests.get(requestId).getRight();
            if (issueKey == null) {
                return;
            }
            final List<QuotaChangeRequest.Change> changesReady = new ArrayList<>();
            final List<QuotaChangeRequest.Change> changesAllocated = new ArrayList<>();
            final List<QuotaChangeRequest.Change> diffChangesAllocated = new ArrayList<>();
            changes.forEach(change -> {
                final long newAllocated = change.getAmountAllocated() + change.getAmountAllocatedIncrement();
                final long newReady = change.getAmountReady() + change.getAmountReadyIncrement();
                final QuotaChangeRequest.Change requestChange = QuotaChangeRequest.Change.builder()
                        .id(change.getId())
                        .resource(change.getResource())
                        .segments(change.getSegments())
                        .amount(change.getAmount())
                        .amountReady(newReady)
                        .amountAllocated(newAllocated)
                        .amountAllocating(newAllocated)
                        .owningCost(BigDecimal.ZERO)
                        .build();
                if (change.getAmountReadyIncrement() > 0 && change.getAmountAllocatedIncrement() > 0) {
                    if (newAllocated != newReady) {
                        changesReady.add(requestChange);
                    }
                    changesAllocated.add(requestChange);
                    diffChangesAllocated.add(requestChange.copyBuilder()
                            .amountAllocated(change.getAmountAllocatedIncrement()).build());
                } else if (change.getAmountReadyIncrement() > 0) {
                    changesReady.add(requestChange);
                } else if (change.getAmountAllocatedIncrement() > 0) {
                    changesAllocated.add(requestChange);
                    diffChangesAllocated.add(requestChange.copyBuilder()
                            .amountAllocated(change.getAmountAllocatedIncrement()).build());
                }
            });
            final String commentText = preorderChangeManager.getCommentForRequest(originalRequests.get(requestId).getLeft(), changesReady,
                    changesAllocated, diffChangesAllocated, quotaDistribution.getComment(), performerContext);
            quartzTrackerCommentTrigger.run(issueKey, commentText);
        });
    }

    private void updateRequestCompletion(@NotNull final QuotaDistribution quotaDistribution,
                                         @NotNull final Map<Long, Set<QuotaDistribution.Change>> changesByRequest,
                                         @NotNull final Map<Long, List<QuotaChangeRequest.Change>> originalChanges,
                                         @NotNull final Map<Long, Pair<QuotaChangeRequest.Status, String>> originalRequests,
                                         @NotNull final PerformerContext performerContext,
                                         final boolean suppressSummon) {
        final Set<Long> completedRequestIds = new HashSet<>();
        changesByRequest.forEach((requestId, changes) -> {
            final boolean modifiedChangesCompleted = changes.stream().filter(change -> preorderChangeManager
                            .isResourceRequiredForCompletion(quotaDistribution.getService(), change.getResource()))
                    .allMatch(change -> change.getAmountAllocatedIncrement()
                            + change.getAmountAllocated() >= change.getAmount()
                            && change.getAmountReadyIncrement() + change.getAmountReady() >= change.getAmount());
            final Set<Long> modifiedChangeIds = changes.stream().map(QuotaDistribution.Change::getId).collect(Collectors.toSet());
            final boolean unmodifiedChangesCompleted = originalChanges.get(requestId).stream()
                    .filter(c -> !modifiedChangeIds.contains(c.getId()))
                    .filter(c -> preorderChangeManager
                            .isResourceRequiredForCompletion(quotaDistribution.getService(), c.getResource()))
                    .allMatch(change -> change.getAmountAllocated() >= change.getAmount()
                            && change.getAmountReady() >= change.getAmount());
            final QuotaChangeRequest.Status currentStatus = originalRequests.get(requestId).getLeft();
            if (modifiedChangesCompleted && unmodifiedChangesCompleted
                    && currentStatus == QuotaChangeRequest.Status.CONFIRMED) {
                completedRequestIds.add(requestId);
            }
        });
        // Quite ineffective, but optimized rewrite would take too much time...
        if (!completedRequestIds.isEmpty()) {
            final QuotaChangeRequestFilterImpl filter = new QuotaChangeRequestFilterImpl();
            filter.setChangeRequestIds(completedRequestIds);
            filter.setStatuses(Collections.singleton(QuotaChangeRequest.Status.CONFIRMED));
            quotaChangeRequestManager.setStatusBatch(filter, QuotaChangeRequest.Status.COMPLETED, performerContext,
                    suppressSummon);
        }
    }

    @NotNull
    private List<QuotaChangeRequestHistoryEvent> prepareHistoryEvents(@NotNull final QuotaDistribution quotaDistribution,
                                                                      @NotNull final Map<Long, Set<QuotaDistribution.Change>> changesByRequest,
                                                                      @NotNull final Map<Long, List<QuotaChangeRequest.Change>> originalChanges,
                                                                      @NotNull final PerformerContext performerContext) {
        final List<QuotaChangeRequestHistoryEvent> result = new ArrayList<>();
        final Instant now = Instant.now();
        changesByRequest.forEach((requestId, changes) -> {
            final List<QuotaChangeRequest.Change> originalRequestChanges = originalChanges.getOrDefault(requestId, Collections.emptyList());
            final Map<Long, QuotaChangeRequest.Change> originalRequestChangesById = originalRequestChanges.stream()
                    .collect(Collectors.toMap(QuotaChangeRequest.Change::getId, Function.identity()));
            final Map<Long, QuotaDistribution.Change> changesById = changes.stream()
                    .collect(Collectors.toMap(QuotaDistribution.Change::getId, Function.identity()));
            final Map<String, Object> oldFields = new HashMap<>();
            final Map<String, Object> newFields = new HashMap<>();
            fillRequestHistoryFields(originalRequestChangesById, changesById, oldFields, newFields, quotaDistribution.getService(), quotaDistribution.getBigOrder());
            result.add(new QuotaChangeRequestHistoryEvent(performerContext.getPerson().getId(),
                    performerContext.getTvmId(), now, quotaDistribution.getComment(), requestId,
                    DiQuotaRequestHistoryEventType.QUOTA_DISTRIBUTION, oldFields, newFields));
        });
        return result;
    }

    private void fillRequestHistoryFields(@NotNull final Map<Long, QuotaChangeRequest.Change> originalChanges,
                                          @NotNull final Map<Long, QuotaDistribution.Change> newChanges,
                                          @NotNull final Map<String, Object> oldFields,
                                          @NotNull final Map<String, Object> newFields, @NotNull Service service, @Nullable BigOrder bigOrder) {
        final List<DiQuotaChangeRequest.Change> oldChangeList = new ArrayList<>();
        final List<DiQuotaChangeRequest.Change> newChangeList = new ArrayList<>();
        newChanges.values().forEach(change -> {
            oldChangeList.add(change.toChangeViewBefore(service, bigOrder == null ? null : BigOrderLegacy.toView(bigOrder)));
            newChangeList.add(change.toChangeViewAfter(service, bigOrder == null ? null : BigOrderLegacy.toView(bigOrder)));
        });
        final Set<Long> unmodifiedChanges = Sets.difference(originalChanges.keySet(), newChanges.keySet());
        unmodifiedChanges.forEach(id -> {
            oldChangeList.add(originalChanges.get(id).toView());
            newChangeList.add(originalChanges.get(id).toView());
        });
        oldFields.put(QuotaChangeRequest.Field.CHANGES.getFieldName(), oldChangeList);
        newFields.put(QuotaChangeRequest.Field.CHANGES.getFieldName(), newChangeList);
    }

    @NotNull
    private Result<QuotaDistribution, ErrorCollection<String, TypedError<String>>> planDistribution(
            @NotNull final DistributeQuotaParams distributeQuotaParams, final boolean lock) {
        final QuotaDistribution.Builder quotaDistributionBuilder = QuotaDistribution.builder();
        quotaDistributionBuilder
                .comment(distributeQuotaParams.getComment())
                .service(distributeQuotaParams.getService())
                .bigOrder(distributeQuotaParams.getBigOrder());
        final List<QuotaChangeInRequest> quotaRequestsChanges = findRequestsForDistribution(distributeQuotaParams, lock);
        final Map<DistributionKey, DiUnit> baseUnitsByKey = getBaseUnits(distributeQuotaParams, quotaRequestsChanges);
        final Map<DistributionKey, Long> availableByKey = new HashMap<>();
        final Map<DistributionKey, Long> availableBaseUnitByKey = new HashMap<>();
        preprocessAvailable(distributeQuotaParams.getChanges(), availableByKey, baseUnitsByKey, availableBaseUnitByKey);
        final Set<DistributionKey> availableKeys = availableByKey.keySet();
        final Map<DistributionKey, BigInteger> pendingByKey = new HashMap<>();
        final Map<DistributionKey, BigInteger> pendingBaseUnitByKey = new HashMap<>();
        final Map<DistributionKey, Integer> pendingRequestCountByKey = new HashMap<>();
        preprocessPending(quotaRequestsChanges, baseUnitsByKey, pendingByKey, pendingBaseUnitByKey, pendingRequestCountByKey);
        final Map<DistributionKey, Set<QuotaChangeInRequest>> changesByKey = quotaRequestsChanges.stream()
                .collect(Collectors.groupingBy(DistributionKey::new, Collectors.toSet()));
        // Resource processing order does not matter here
        availableKeys.forEach(key -> {
            // No changes to distribute to
            if (!changesByKey.containsKey(key)) {
                return;
            }
            // No pending resource amount
            if (!pendingByKey.containsKey(key)) {
                return;
            }
            final DiUnit originalUnit = key.getResource().getType().getBaseUnit();
            final DiUnit baseUnit = baseUnitsByKey.get(key);
            if (BigInteger.valueOf(availableByKey.get(key)).compareTo(pendingByKey.get(key)) >= 0) {
                // There is enough of this resource to satisfy every request
                // So we satisfy every request, possibly with some remainders left
                final long remainingAvailable = distributeEnoughForAll(distributeQuotaParams.isAllocate(),
                        changesByKey.get(key), availableByKey.get(key), quotaDistributionBuilder);
                availableByKey.put(key, remainingAvailable);
            } else if (availableBaseUnitByKey.get(key) <= pendingRequestCountByKey.get(key)) {
                // There are fewer base units of this resource than there are request instances for it
                // So we divide the available resource unevenly
                final long remainingAvailable = distributeUnevenly(distributeQuotaParams.isAllocate(), originalUnit, baseUnit,
                        changesByKey.get(key), availableByKey.get(key), availableBaseUnitByKey.get(key), pendingByKey.get(key),
                        pendingBaseUnitByKey.get(key), quotaDistributionBuilder);
                availableByKey.put(key, remainingAvailable);
            } else {
                // Either proportional distribution or even distribution otherwise
                final long remainingAvailable;
                if (distributeQuotaParams.getAlgorithm() == ResourceDistributionAlgorithm.PROPORTIONAL_TO_READINESS_RATIO) {
                    remainingAvailable = distributeProportional(distributeQuotaParams.isAllocate(), originalUnit, baseUnit,
                            changesByKey.get(key), availableByKey.get(key), availableBaseUnitByKey.get(key), pendingByKey.get(key),
                            pendingBaseUnitByKey.get(key), quotaDistributionBuilder);
                } else {
                    remainingAvailable = distributeEvenly(distributeQuotaParams.isAllocate(), originalUnit, baseUnit,
                            changesByKey.get(key), availableByKey.get(key), availableBaseUnitByKey.get(key), pendingByKey.get(key),
                            pendingBaseUnitByKey.get(key), quotaDistributionBuilder);
                }
                availableByKey.put(key, remainingAvailable);
            }
        });
        if (availableByKey.values().stream().anyMatch(v -> v > 0L)) {
            return prepareRemaindersError(availableByKey, quotaDistributionBuilder.build(), distributeQuotaParams);
        }
        return Result.success(quotaDistributionBuilder.build());
    }

    private void preprocessPending(@NotNull final List<QuotaChangeInRequest> quotaRequestsChanges,
                                   @NotNull final Map<DistributionKey, DiUnit> baseUnits,
                                   @NotNull final Map<DistributionKey, BigInteger> pendingByKey,
                                   @NotNull final Map<DistributionKey, BigInteger> pendingBaseUnitByKey,
                                   @NotNull final Map<DistributionKey, Integer> pendingRequestCountByKey) {
        final Map<DistributionKey, Set<Long>> requestIdsByKey = new HashMap<>();
        quotaRequestsChanges.forEach(change -> {
            if (change.getAmountReady() >= change.getAmount()) {
                return;
            }
            final DistributionKey key = new DistributionKey(change);
            final BigInteger delta = BigInteger.valueOf(change.getAmount() - change.getAmountReady());
            final DiUnit baseUnit = baseUnits.get(key);
            final BigInteger converted = baseUnit.convertInteger(delta, key.getResource().getType().getBaseUnit());
            requestIdsByKey.computeIfAbsent(key, k -> new HashSet<>()).add(change.getRequestId());
            pendingByKey.compute(key, (k, v) -> v == null ? delta : v.add(delta));
            pendingBaseUnitByKey.compute(key,
                    (k, v) -> v == null ? converted : v.add(converted));
        });
        requestIdsByKey.forEach((k, v) -> {
            pendingRequestCountByKey.put(k, v.size());
        });
    }

    private void preprocessAvailable(@NotNull final List<DistributeQuotaParams.Change> changes,
                                     @NotNull final Map<DistributionKey, Long> availableByKey,
                                     @NotNull final Map<DistributionKey, DiUnit> baseUnits,
                                     @NotNull final Map<DistributionKey, Long> availableBaseUnitByKey) {
        changes.forEach(change -> {
            final DistributionKey key = new DistributionKey(change);
            final long value = change.getAmountReady();
            final DiUnit baseUnit = baseUnits.get(key);
            final long converted = baseUnit.convertInteger(value, key.getResource().getType().getBaseUnit());
            availableByKey.put(key, value);
            availableBaseUnitByKey.put(key, converted);
        });
    }

    private long distributeEnoughForAll(final boolean allocate,
                                        @NotNull final Set<QuotaChangeInRequest> changesForKey,
                                        final long availableForKey,
                                        @NotNull final QuotaDistribution.Builder quotaDistributionBuilder) {
        long remainingAvailableOriginalUnit = availableForKey;
        for (final QuotaChangeInRequest change : changesForKey) {
            if (change.getAmountReady() >= change.getAmount()) {
                continue;
            }
            final QuotaDistribution.Change.Builder distributionChangeBuilder = QuotaDistribution.Change.builder();
            distributionChangeBuilder
                    .id(change.getId())
                    .requestId(change.getRequestId())
                    .resource(change.getResource())
                    .segments(change.getSegments())
                    .amount(change.getAmount())
                    .amountReady(change.getAmountReady())
                    .amountAllocated(change.getAmountAllocated())
                    .amountAllocating(change.getAmountAllocating());
            distributionChangeBuilder.amountReadyIncrement(change.getAmount() - change.getAmountReady());
            if (allocate && change.getAmount() > change.getAmountAllocated()) {
                distributionChangeBuilder
                        .amountAllocatedIncrement(change.getAmount() - change.getAmountAllocated());
            } else {
                distributionChangeBuilder.amountAllocatedIncrement(0L);
            }
            remainingAvailableOriginalUnit -= change.getAmount() - change.getAmountReady();
            quotaDistributionBuilder.addChange(distributionChangeBuilder.build());
        }
        return remainingAvailableOriginalUnit;
    }

    private long distributeProportional(final boolean allocate, @NotNull final DiUnit originalUnit, @NotNull final DiUnit baseUnit,
                                        @NotNull final Set<QuotaChangeInRequest> changesForKey,
                                        final long available, final long availableBaseUnit,
                                        @NotNull final BigInteger pending, @NotNull final BigInteger pendingBaseUnit,
                                        @NotNull final QuotaDistribution.Builder quotaDistributionBuilder) {
        // Remaining amount of base units of pending resource per change
        final Map<Long, Long> pendingOriginalUnitByChange = new HashMap<>();
        // Remaining amount of base units of pending resource per change
        final Map<Long, Long> pendingBaseUnitByChange = new HashMap<>();
        // Group pending amounts per change
        changesForKey.forEach(change -> {
            final long pendingOriginalUnit = change.getAmount() > change.getAmountReady()
                    ? change.getAmount() - change.getAmountReady() : 0L;
            final long converted = baseUnit.convertInteger(pendingOriginalUnit, originalUnit);
            pendingOriginalUnitByChange.put(change.getId(), pendingOriginalUnit);
            pendingBaseUnitByChange.put(change.getId(), converted);
        });
        // Remaining amount of base units of available resource
        long remainingAvailableBaseUnit = availableBaseUnit;
        // Remaining amount of base units of pending resource
        BigInteger remainingPendingBaseUnit = pendingBaseUnit;
        // Remaining amount of available resource in original units
        long remainingAvailableOriginalUnit = available;
        // Remaining amount of pending resource in original units
        BigInteger remainingPendingOriginalUnit = pending;
        // Collect increments for each change in original units
        final Map<Long, Long> incrementByChange = new HashMap<>();
        for (final QuotaChangeInRequest change : changesForKey) {
            // Break the loop when the distribution is finished
            if (remainingAvailableBaseUnit == 0L || remainingPendingBaseUnit.compareTo(BigInteger.ZERO) == 0) {
                break;
            }
            // This change is already fully satisfied
            if (change.getAmountReady() >= change.getAmount()) {
                continue;
            }
            final long pendingChangeBaseUnit = pendingBaseUnitByChange.get(change.getId());
            // This change is already fully satisfied
            if (pendingChangeBaseUnit == 0) {
                continue;
            }
            final long appliedBaseUnit = Math.min(BigDecimal.valueOf(pendingChangeBaseUnit).multiply(BigDecimal.valueOf(availableBaseUnit))
                    .divideAndRemainder(new BigDecimal(pendingBaseUnit))[0].longValue(), pendingChangeBaseUnit);
            final long appliedOriginalUnit = originalUnit.convert(appliedBaseUnit, baseUnit);
            final long pendingChangeOriginalUnit = pendingOriginalUnitByChange.get(change.getId());
            remainingAvailableBaseUnit -= appliedBaseUnit;
            remainingPendingBaseUnit = remainingPendingBaseUnit.subtract(BigInteger.valueOf(appliedBaseUnit));
            remainingAvailableOriginalUnit -= appliedOriginalUnit;
            remainingPendingOriginalUnit = remainingPendingOriginalUnit.subtract(BigInteger.valueOf(appliedOriginalUnit));
            pendingBaseUnitByChange.put(change.getId(), pendingChangeBaseUnit - appliedBaseUnit);
            pendingOriginalUnitByChange.put(change.getId(), pendingChangeOriginalUnit - appliedOriginalUnit);
            incrementByChange.compute(change.getId(), (k, v) -> v == null ? appliedOriginalUnit : v + appliedOriginalUnit);
        }
        // Remainder of base units is no more then one base unit per request, so distribute them unevenly
        // Update pending amounts in base units per change
        remainingPendingBaseUnit = BigInteger.ZERO;
        for (final Map.Entry<Long, Long> entry : pendingOriginalUnitByChange.entrySet()) {
            final long converted = baseUnit.convertInteger(entry.getValue(), originalUnit);
            pendingBaseUnitByChange.put(entry.getKey(), converted);
            remainingPendingBaseUnit = remainingPendingBaseUnit.add(BigInteger.valueOf(converted));
        }
        // Update total available amount in base units
        remainingAvailableBaseUnit = baseUnit.convertInteger(remainingAvailableOriginalUnit, originalUnit);
        final long remainingAvailable = distributeUnevenlyImpl(originalUnit, baseUnit, changesForKey, remainingAvailableOriginalUnit,
                remainingAvailableBaseUnit, remainingPendingOriginalUnit, remainingPendingBaseUnit,
                pendingOriginalUnitByChange, pendingBaseUnitByChange, incrementByChange);
        buildAccumulatedChanges(allocate, quotaDistributionBuilder, changesForKey, incrementByChange);
        return remainingAvailable;
    }

    private long distributeEvenly(final boolean allocate,
                                  @NotNull final DiUnit originalUnit, @NotNull final DiUnit baseUnit,
                                  @NotNull final Set<QuotaChangeInRequest> changesForKey,
                                  @NotNull final Long available,
                                  @NotNull final Long availableBaseUnit,
                                  @NotNull final BigInteger pending,
                                  @NotNull final BigInteger pendingBaseUnit,
                                  @NotNull final QuotaDistribution.Builder quotaDistributionBuilder) {
        // Remaining amount of base units of pending resource per change
        final Map<Long, Long> pendingOriginalUnitByChange = new HashMap<>();
        // Remaining amount of base units of pending resource per change
        final Map<Long, Long> pendingBaseUnitByChange = new HashMap<>();
        // Group pending amounts per change
        changesForKey.forEach(change -> {
            final long pendingOriginalUnit = change.getAmount() > change.getAmountReady()
                    ? change.getAmount() - change.getAmountReady() : 0L;
            final long converted = baseUnit.convertInteger(pendingOriginalUnit, originalUnit);
            pendingOriginalUnitByChange.put(change.getId(), pendingOriginalUnit);
            pendingBaseUnitByChange.put(change.getId(), converted);
        });
        // Remaining amount of base units of available resource
        long remainingAvailableBaseUnit = availableBaseUnit;
        // Remaining amount of base units of pending resource
        BigInteger remainingPendingBaseUnit = pendingBaseUnit;
        // Remaining amount of available resource in original units
        long remainingAvailableOriginalUnit = available;
        // Remaining amount of pending resource in original units
        BigInteger remainingPendingOriginalUnit = pending;
        // Collect increments for each change in original units
        final Map<Long, Long> incrementByChange = new HashMap<>();
        final Map<Long, Set<QuotaChangeInRequest>> changesByRequest = changesForKey.stream()
                .collect(Collectors.groupingBy(QuotaChangeInRequest::getRequestId, Collectors.toSet()));
        final List<Long> sortedRequestIds = changesByRequest.keySet().stream().sorted().collect(Collectors.toList());
        // Distribute integer base units of the resource first
        // Repeat until either there are no more available base units or all base units requirements are satisfied
        while (remainingAvailableBaseUnit > 0L && remainingPendingBaseUnit.compareTo(BigInteger.ZERO) > 0) {
            // Count requests where at least one base unit of resource is required
            final long pendingRequestsCount = countPendingRequests(changesForKey, pendingBaseUnitByChange);
            if (pendingRequestsCount == 0) {
                break;
            }
            // Distribute available base units evenly between pending requests
            // Some requests may not need all those resources, so those extra resources will be distributed during the next iteration
            final long baseUnitsPerRequest = remainingAvailableBaseUnit / pendingRequestsCount;
            // Remainder is distributed separately
            final long remainderBaseUnits = remainingAvailableBaseUnit % pendingRequestsCount;
            final long remainderOriginalUnits = originalUnit.convert(remainderBaseUnits, baseUnit);
            if (baseUnitsPerRequest > 0L) {
                for (final long requestId : sortedRequestIds) {
                    // Break the loop when the distribution is finished
                    if (remainingAvailableBaseUnit == 0L || remainingPendingBaseUnit.compareTo(BigInteger.ZERO) == 0) {
                        break;
                    }
                    final List<QuotaChangeInRequest> changesForRequest = changesByRequest.get(requestId).stream()
                            .sorted(Comparator.comparing(QuotaChangeInRequest::getId)).collect(Collectors.toList());
                    BigInteger pendingBaseUnitsForRequest = sumPendingRequests(changesForRequest, pendingBaseUnitByChange);
                    if (pendingBaseUnitsForRequest.compareTo(BigInteger.ZERO) == 0) {
                        break;
                    }
                    long availableBaseUnitsForRequest = baseUnitsPerRequest;
                    for (final QuotaChangeInRequest change : changesForRequest) {
                        // Break the loop when the distribution is finished
                        if (remainingAvailableBaseUnit == 0L || remainingPendingBaseUnit.compareTo(BigInteger.ZERO) == 0) {
                            break;
                        }
                        // Nothing left to distribute for this request
                        if (availableBaseUnitsForRequest == 0L || pendingBaseUnitsForRequest.compareTo(BigInteger.ZERO) == 0) {
                            break;
                        }
                        // This change is already fully satisfied
                        if (change.getAmountReady() >= change.getAmount()) {
                            continue;
                        }
                        final long pendingChangeBaseUnit = pendingBaseUnitByChange.get(change.getId());
                        // This change is already fully satisfied
                        if (pendingChangeBaseUnit == 0) {
                            continue;
                        }
                        final long appliedChangeBaseUnit = Math.min(pendingChangeBaseUnit, availableBaseUnitsForRequest);
                        final long appliedChangeOriginalUnit = originalUnit.convert(appliedChangeBaseUnit, baseUnit);
                        final long pendingChangeOriginalUnit = pendingOriginalUnitByChange.get(change.getId());
                        availableBaseUnitsForRequest -= appliedChangeBaseUnit;
                        pendingBaseUnitsForRequest = pendingBaseUnitsForRequest.subtract(BigInteger.valueOf(appliedChangeBaseUnit));
                        remainingAvailableBaseUnit -= appliedChangeBaseUnit;
                        remainingPendingBaseUnit = remainingPendingBaseUnit.subtract(BigInteger.valueOf(appliedChangeBaseUnit));
                        remainingAvailableOriginalUnit -= appliedChangeOriginalUnit;
                        remainingPendingOriginalUnit = remainingPendingOriginalUnit.subtract(BigInteger.valueOf(appliedChangeOriginalUnit));
                        pendingBaseUnitByChange.put(change.getId(), pendingChangeBaseUnit - appliedChangeBaseUnit);
                        pendingOriginalUnitByChange.put(change.getId(), pendingChangeOriginalUnit - appliedChangeOriginalUnit);
                        incrementByChange.compute(change.getId(), (k, v) -> v == null ? appliedChangeOriginalUnit : v + appliedChangeOriginalUnit);
                    }
                }
            }
            if (remainderBaseUnits > 0L) {
                // Distribute remainder units, one by one
                final Pair<Long, BigInteger> remainder = distributeUnevenlyIntegerImpl(originalUnit, baseUnit, changesForKey,
                        remainderOriginalUnits, remainderBaseUnits, remainingPendingOriginalUnit, remainingPendingBaseUnit,
                        pendingOriginalUnitByChange, pendingBaseUnitByChange, incrementByChange);
                remainingPendingBaseUnit = BigInteger.ZERO;
                remainingPendingOriginalUnit = BigInteger.ZERO;
                for (final Map.Entry<Long, Long> entry : pendingOriginalUnitByChange.entrySet()) {
                    final long converted = baseUnit.convertInteger(entry.getValue(), originalUnit);
                    remainingPendingBaseUnit = remainingPendingBaseUnit.add(BigInteger.valueOf(converted));
                    remainingPendingOriginalUnit = remainingPendingOriginalUnit.add(BigInteger.valueOf(entry.getValue()));
                }
                final long distributedOriginalUnit = remainderOriginalUnits - remainder.getLeft();
                final long distributedBaseUnit = baseUnit.convert(distributedOriginalUnit, originalUnit);
                remainingAvailableOriginalUnit -= distributedOriginalUnit;
                remainingAvailableBaseUnit -= distributedBaseUnit;
            }
        }
        if (remainingAvailableOriginalUnit > 0 && remainingPendingOriginalUnit.compareTo(BigInteger.ZERO) > 0) {
            // Distribute remaining fractional resources
            remainingAvailableOriginalUnit = distributeUnevenlyImpl(originalUnit, baseUnit, changesForKey,
                    remainingAvailableOriginalUnit, remainingAvailableBaseUnit, remainingPendingOriginalUnit,
                    remainingPendingBaseUnit, pendingOriginalUnitByChange, pendingBaseUnitByChange, incrementByChange);
        }
        buildAccumulatedChanges(allocate, quotaDistributionBuilder, changesForKey, incrementByChange);
        return remainingAvailableOriginalUnit;
    }

    @NotNull
    private BigInteger sumPendingRequests(@NotNull final List<QuotaChangeInRequest> changesForRequest,
                                          @NotNull final Map<Long, Long> pendingBaseUnitByChange) {
        BigInteger sum = BigInteger.ZERO;
        for (final QuotaChangeInRequest change : changesForRequest) {
            sum = sum.add(BigInteger.valueOf(pendingBaseUnitByChange.getOrDefault(change.getId(), 0L)));
        }
        return sum;
    }

    private long countPendingRequests(@NotNull final Set<QuotaChangeInRequest> changesForKey,
                                      @NotNull final Map<Long, Long> pendingBaseUnitByChange) {
        long pendingRequestsCount = 0L;
        final Set<Long> seenRequestIds = new HashSet<>();
        for (final QuotaChangeInRequest change : changesForKey) {
            if (pendingBaseUnitByChange.get(change.getId()) == 0L) {
                continue;
            }
            if (seenRequestIds.contains(change.getRequestId())) {
                continue;
            }
            seenRequestIds.add(change.getRequestId());
            pendingRequestsCount++;
        }
        return pendingRequestsCount;
    }

    private long distributeUnevenly(final boolean allocate,
                                    @NotNull final DiUnit originalUnit, @NotNull final DiUnit baseUnit,
                                    @NotNull final Set<QuotaChangeInRequest> changesForKey,
                                    final long available,
                                    final long availableBaseUnit,
                                    @NotNull final BigInteger pending,
                                    @NotNull final BigInteger pendingBaseUnit,
                                    @NotNull final QuotaDistribution.Builder quotaDistributionBuilder) {
        // Remaining amount of base units of pending resource per change
        final Map<Long, Long> pendingOriginalUnitByChange = new HashMap<>();
        // Remaining amount of base units of pending resource per change
        final Map<Long, Long> pendingBaseUnitByChange = new HashMap<>();
        // Group pending amounts per change
        changesForKey.forEach(change -> {
            final long pendingOriginalUnit = change.getAmount() > change.getAmountReady()
                    ? change.getAmount() - change.getAmountReady() : 0L;
            final long converted = baseUnit.convertInteger(pendingOriginalUnit, originalUnit);
            pendingOriginalUnitByChange.put(change.getId(), pendingOriginalUnit);
            pendingBaseUnitByChange.put(change.getId(), converted);
        });
        // Collect increments for each change in original units
        final Map<Long, Long> incrementByChange = new HashMap<>();
        final long remainingAvailable = distributeUnevenlyImpl(originalUnit, baseUnit, changesForKey, available,
                availableBaseUnit, pending, pendingBaseUnit, pendingOriginalUnitByChange,
                pendingBaseUnitByChange, incrementByChange);
        buildAccumulatedChanges(allocate, quotaDistributionBuilder, changesForKey, incrementByChange);
        return remainingAvailable;
    }

    private void buildAccumulatedChanges(final boolean allocate,
                                         @NotNull final QuotaDistribution.Builder quotaDistributionBuilder,
                                         @NotNull final Set<QuotaChangeInRequest> changesForKey,
                                         @NotNull final Map<Long, Long> incrementByChange) {
        // Use accumulated changes to prepare updates
        changesForKey.forEach(change -> {
            if (change.getAmountReady() >= change.getAmount()) {
                return;
            }
            if (!incrementByChange.containsKey(change.getId())) {
                return;
            }
            final long increment = incrementByChange.get(change.getId());
            if (increment == 0L) {
                return;
            }
            final QuotaDistribution.Change.Builder distributionChangeBuilder = QuotaDistribution.Change.builder();
            distributionChangeBuilder
                    .id(change.getId())
                    .requestId(change.getRequestId())
                    .resource(change.getResource())
                    .segments(change.getSegments())
                    .amount(change.getAmount())
                    .amountReady(change.getAmountReady())
                    .amountAllocated(change.getAmountAllocated())
                    .amountAllocating(change.getAmountAllocating());
            distributionChangeBuilder.amountReadyIncrement(increment);
            final long incrementedAmountReady = change.getAmountReady() + increment;
            if (allocate && incrementedAmountReady > change.getAmountAllocated()) {
                distributionChangeBuilder
                        .amountAllocatedIncrement(incrementedAmountReady - change.getAmountAllocated());
            } else {
                distributionChangeBuilder.amountAllocatedIncrement(0L);
            }
            quotaDistributionBuilder.addChange(distributionChangeBuilder.build());
        });
    }

    private long distributeUnevenlyImpl(@NotNull final DiUnit originalUnit, @NotNull final DiUnit baseUnit,
                                        @NotNull final Set<QuotaChangeInRequest> changesForKey,
                                        final long available,
                                        final long availableBaseUnit,
                                        @NotNull final BigInteger pending,
                                        @NotNull final BigInteger pendingBaseUnit,
                                        @NotNull final Map<Long, Long> pendingOriginalUnitByChange,
                                        @NotNull final Map<Long, Long> pendingBaseUnitByChange,
                                        @NotNull final Map<Long, Long> incrementByChange) {
        final Pair<Long, BigInteger> remaindersInteger = distributeUnevenlyIntegerImpl(originalUnit, baseUnit, changesForKey, available,
                availableBaseUnit, pending, pendingBaseUnit, pendingOriginalUnitByChange, pendingBaseUnitByChange, incrementByChange);
        final Pair<Long, BigInteger> remaindersFractional = distributeUnevenlyFractionalImpl(changesForKey,
                remaindersInteger.getLeft(), remaindersInteger.getRight(), pendingOriginalUnitByChange, incrementByChange);
        return remaindersFractional.getLeft();
    }

    @NotNull
    private Pair<Long, BigInteger> distributeUnevenlyFractionalImpl(@NotNull final Set<QuotaChangeInRequest> changesForKey,
                                                                    final long available,
                                                                    @NotNull final BigInteger pending,
                                                                    @NotNull final Map<Long, Long> pendingOriginalUnitByChange,
                                                                    @NotNull final Map<Long, Long> incrementByChange) {
        long remainingAvailableOriginalUnit = available;
        BigInteger remainingPendingOriginalUnit = pending;
        final List<QuotaChangeInRequest> sortedChanges = changesForKey.stream()
                .sorted(Comparator.comparing(QuotaChangeInRequest::getId)).collect(Collectors.toList());
        for (final QuotaChangeInRequest change : sortedChanges) {
            // Break the loop when the distribution is finished
            if (remainingAvailableOriginalUnit == 0L || remainingPendingOriginalUnit.compareTo(BigInteger.ZERO) == 0) {
                break;
            }
            // This change is already fully satisfied
            if (change.getAmountReady() >= change.getAmount()) {
                continue;
            }
            final long pendingChangeOriginalUnit = pendingOriginalUnitByChange.get(change.getId());
            // This change is already fully satisfied
            if (pendingChangeOriginalUnit == 0L) {
                continue;
            }
            // Distribute parts smaller than one base unit to the first change that can take them
            final long appliedOriginalUnit = Math.min(pendingChangeOriginalUnit, remainingAvailableOriginalUnit);
            remainingAvailableOriginalUnit -= appliedOriginalUnit;
            remainingPendingOriginalUnit = remainingPendingOriginalUnit.subtract(BigInteger.valueOf(appliedOriginalUnit));
            pendingOriginalUnitByChange.put(change.getId(), pendingChangeOriginalUnit - appliedOriginalUnit);
            incrementByChange.compute(change.getId(), (k, v) -> v == null ? appliedOriginalUnit : v + appliedOriginalUnit);
        }
        return Pair.of(remainingAvailableOriginalUnit, remainingPendingOriginalUnit);
    }

    @NotNull
    private Pair<Long, BigInteger> distributeUnevenlyIntegerImpl(@NotNull final DiUnit originalUnit, @NotNull final DiUnit baseUnit,
                                                                 @NotNull final Set<QuotaChangeInRequest> changesForKey,
                                                                 final long available,
                                                                 final long availableBaseUnit,
                                                                 @NotNull final BigInteger pending,
                                                                 @NotNull final BigInteger pendingBaseUnit,
                                                                 @NotNull final Map<Long, Long> pendingOriginalUnitByChange,
                                                                 @NotNull final Map<Long, Long> pendingBaseUnitByChange,
                                                                 @NotNull final Map<Long, Long> incrementByChange) {
        // Changes by project id and request id
        final Map<Long, Map<Long, Set<QuotaChangeInRequest>>> groupedChanges = new HashMap<>();
        // Group changes
        changesForKey.forEach(change -> {
            groupedChanges.computeIfAbsent(change.getProject().getId(), k -> new HashMap<>())
                    .computeIfAbsent(change.getRequestId(), k -> new HashSet<>()).add(change);
        });
        // Always process projects in the same order
        final List<Long> sortedProjectIds = groupedChanges.keySet().stream().sorted().collect(Collectors.toList());
        // Sorted request ids by project id
        final Map<Long, List<Long>> requestIdsByProject = new HashMap<>();
        // Last request id index by project id
        final Map<Long, Integer> lastRequestIdIndexByProject = new HashMap<>();
        groupedChanges.forEach((k, v) -> {
            // Always process request ids in the same order
            requestIdsByProject.put(k, v.keySet().stream().sorted().collect(Collectors.toList()));
            lastRequestIdIndexByProject.put(k, 0);
        });
        // Remaining amount of available resource in original units
        long remainingAvailableOriginalUnit = available;
        // Remaining amount of base units of available resource
        long remainingAvailableBaseUnit = availableBaseUnit;
        // Remaining amount of pending resource in original units
        BigInteger remainingPendingOriginalUnit = pending;
        // Remaining amount of base units of pending resource
        BigInteger remainingPendingBaseUnit = pendingBaseUnit;
        // Distribute integer base units of the resource first
        while (remainingAvailableBaseUnit > 0L && remainingPendingBaseUnit.compareTo(BigInteger.ZERO) > 0) {
            // Distribute until either no more resources are available or every request is satisfied
            // Iterate over projects on each iteration
            for (final Long projectId : sortedProjectIds) {
                // Break the loop when the distribution is finished
                if (remainingAvailableBaseUnit == 0L || remainingPendingBaseUnit.compareTo(BigInteger.ZERO) == 0) {
                    break;
                }
                // All request ids for this project
                final List<Long> sortedRequestIds = requestIdsByProject.get(projectId);
                // Current request id index for this project
                final int requestIdIndex = lastRequestIdIndexByProject.get(projectId);
                // Current request id for this project
                final long requestId = sortedRequestIds.get(requestIdIndex);
                // Changes for current request
                final Set<QuotaChangeInRequest> changes = groupedChanges.get(projectId).get(requestId);
                // Use the same order when iterating over changes
                final List<QuotaChangeInRequest> sortedChanges = changes.stream()
                        .sorted(Comparator.comparing(QuotaChangeInRequest::getId)).collect(Collectors.toList());
                // Iterate over request changes until the first unit of quota is distributed or until there are no more changes
                for (final QuotaChangeInRequest change : sortedChanges) {
                    // This change is already fully satisfied
                    if (change.getAmountReady() >= change.getAmount()) {
                        continue;
                    }
                    final long pendingChange = pendingBaseUnitByChange.get(change.getId());
                    // This change is already fully satisfied
                    if (pendingChange == 0L) {
                        continue;
                    }
                    // Distribute exactly one base unit
                    final long appliedOriginalUnit = originalUnit.convert(1L, baseUnit);
                    final long pendingChangeOriginalUnit = pendingOriginalUnitByChange.get(change.getId());
                    remainingAvailableBaseUnit -= 1L;
                    remainingAvailableOriginalUnit -= appliedOriginalUnit;
                    remainingPendingBaseUnit = remainingPendingBaseUnit.subtract(BigInteger.ONE);
                    remainingPendingOriginalUnit = remainingPendingOriginalUnit.subtract(BigInteger.valueOf(appliedOriginalUnit));
                    pendingBaseUnitByChange.put(change.getId(), pendingChange - 1L);
                    pendingOriginalUnitByChange.put(change.getId(), pendingChangeOriginalUnit - appliedOriginalUnit);
                    incrementByChange.compute(change.getId(), (k, v) -> v == null ? appliedOriginalUnit : v + appliedOriginalUnit);
                    // Advance to the next project if quota was distributed
                    break;
                }
                // Either advance to the next request in this project in the following iteration or rewind back to the beginning of the list
                if (requestIdIndex + 1 >= sortedRequestIds.size()) {
                    lastRequestIdIndexByProject.put(projectId, 0);
                } else {
                    lastRequestIdIndexByProject.put(projectId, requestIdIndex + 1);
                }
            }
        }
        return Pair.of(remainingAvailableOriginalUnit, remainingPendingOriginalUnit);
    }

    @NotNull
    private Map<DistributionKey, DiUnit> getBaseUnits(@NotNull final DistributeQuotaParams distributeQuotaParams,
                                                      @NotNull final List<QuotaChangeInRequest> quotaRequestsChanges) {
        final Set<DistributionKey> sourceKeys = distributeQuotaParams.getChanges().stream()
                .map(DistributionKey::new).collect(Collectors.toSet());
        final Set<DistributionKey> destinationKeys = quotaRequestsChanges.stream()
                .map(DistributionKey::new).collect(Collectors.toSet());
        final Set<DistributionKey> keys = Sets.union(sourceKeys, destinationKeys);
        return keys.stream().collect(Collectors.toMap(Function.identity(),
                key -> getBaseUnit(key.getResource(), distributeQuotaParams.getService())));
    }

    @NotNull
    private Result<QuotaDistribution, ErrorCollection<String, TypedError<String>>> prepareRemaindersError(
            @NotNull final Map<DistributionKey, Long> remainders,
            @NotNull final QuotaDistribution quotaDistribution,
            @NotNull final DistributeQuotaParams distributeQuotaParams) {
        final ErrorCollection.Builder<String, TypedError<String>> errorBuilder = ErrorCollection.builder();
        errorBuilder.addError(TypedError.invalid("Resources can not be distributed without remainders, see details."));
        final QuotaDistributionRemainder.Builder remainderBuilder = QuotaDistributionRemainder.builder();
        remainders.entrySet().stream().filter(e -> e.getValue() > 0L).forEach(e -> {
            remainderBuilder.addChange(QuotaDistributionRemainder.Change.builder()
                    .resourceKey(e.getKey().getResource().getPublicKey())
                    .segmentKeys(e.getKey().getSegments().stream().map(Segment::getPublicKey)
                            .collect(Collectors.toSet()))
                    .amountReady(DiAmount.of(e.getValue(), e.getKey().getResource().getType().getBaseUnit()))
                    .build());
        });
        errorBuilder.addDetail("quotaRemainders", remainderBuilder.build());
        final DistributedQuota distributedQuota = prepareDistributedQuota(quotaDistribution, distributeQuotaParams);
        if (!distributedQuota.getUpdates().isEmpty()) {
            errorBuilder.addDetail("quotaDistribution", distributedQuota);
        }
        final DistributedQuotaDeltas distributedQuotaDeltas = prepareDistributedQuotaDeltas(quotaDistribution, distributeQuotaParams);
        if (!distributedQuotaDeltas.getUpdates().isEmpty()) {
            errorBuilder.addDetail("quotaDistributionDeltas", distributedQuotaDeltas);
        }
        final DistributableQuota distributableQuota = prepareDistributableQuota(quotaDistribution, distributeQuotaParams);
        if (!distributableQuota.getChanges().isEmpty()) {
            errorBuilder.addDetail("distributableQuota", distributableQuota);
        }
        return Result.failure(errorBuilder.build());
    }

    @NotNull
    private QuotaDistributionPlan prepareQuotaDistributionPlan(@NotNull final QuotaDistribution quotaDistribution,
                                                               @NotNull final DistributeQuotaParams distributeQuotaParams) {
        final DistributedQuota distributedQuota = prepareDistributedQuota(quotaDistribution, distributeQuotaParams);
        final DistributedQuotaDeltas distributedQuotaDeltas = prepareDistributedQuotaDeltas(quotaDistribution, distributeQuotaParams);
        return QuotaDistributionPlan.builder()
                .distributedQuota(distributedQuota)
                .distributedQuotaDeltas(distributedQuotaDeltas)
                .build();
    }

    @NotNull
    private DistributedQuota prepareDistributedQuota(@NotNull final QuotaDistribution quotaDistribution,
                                                     @NotNull final DistributeQuotaParams distributeQuotaParams) {
        final DistributedQuota.Builder builder = DistributedQuota.builder();
        final Map<Long, List<QuotaDistribution.Change>> distributionByRequestId = quotaDistribution.getChanges()
                .stream().collect(Collectors.groupingBy(QuotaDistribution.Change::getRequestId));
        distributionByRequestId.forEach((requestId, changes) -> {
            final DistributedQuota.Update.Builder updateBuilder = DistributedQuota.Update.builder();
            updateBuilder
                    .requestId(requestId)
                    .comment(distributeQuotaParams.getComment());
            changes.forEach(change -> {
                final DistributedQuota.Update.Change.Builder changeBuilder = DistributedQuota.Update.Change.builder();
                changeBuilder
                        .resourceKey(change.getResource().getPublicKey())
                        .segmentKeys(change.getSegments().stream().map(Segment::getPublicKey).collect(Collectors.toSet()));
                if (change.getAmountReadyIncrement() > 0L) {
                    final long updatedAmountReady = change.getAmountReady() + change.getAmountReadyIncrement();
                    changeBuilder.amountReady(DiAmount.of(updatedAmountReady,
                            change.getResource().getType().getBaseUnit()));
                }
                if (change.getAmountAllocatedIncrement() > 0L) {
                    final long updatedAmountAllocated = change.getAmountAllocated() + change.getAmountAllocatedIncrement();
                    changeBuilder.amountAllocated(DiAmount.of(updatedAmountAllocated,
                            change.getResource().getType().getBaseUnit()));
                }
                if (change.getAmountReadyIncrement() > 0L || change.getAmountAllocatedIncrement() > 0L) {
                    updateBuilder.addChange(changeBuilder.build());
                }
            });
            if (!updateBuilder.getChanges().isEmpty()) {
                builder.addUpdate(updateBuilder.build());
            }
        });
        return builder.build();
    }

    @NotNull
    private DistributedQuotaDeltas prepareDistributedQuotaDeltas(@NotNull final QuotaDistribution quotaDistribution,
                                                                 @NotNull final DistributeQuotaParams distributeQuotaParams) {
        final DistributedQuotaDeltas.Builder builder = DistributedQuotaDeltas.builder();
        final Map<Long, List<QuotaDistribution.Change>> distributionByRequestId = quotaDistribution.getChanges()
                .stream().collect(Collectors.groupingBy(QuotaDistribution.Change::getRequestId));
        distributionByRequestId.forEach((requestId, changes) -> {
            final DistributedQuotaDeltas.Update.Builder updateBuilder = DistributedQuotaDeltas.Update.builder();
            updateBuilder
                    .requestId(requestId)
                    .comment(distributeQuotaParams.getComment());
            changes.forEach(change -> {
                final DistributedQuotaDeltas.Update.Change.Builder changeBuilder = DistributedQuotaDeltas.Update.Change.builder();
                changeBuilder
                        .resourceKey(change.getResource().getPublicKey())
                        .segmentKeys(change.getSegments().stream().map(Segment::getPublicKey).collect(Collectors.toSet()));
                if (change.getAmountReadyIncrement() > 0L) {
                    changeBuilder.amountReady(DiAmount.of(change.getAmountReadyIncrement(),
                            change.getResource().getType().getBaseUnit()));
                }
                if (change.getAmountAllocatedIncrement() > 0L) {
                    changeBuilder.amountAllocated(DiAmount.of(change.getAmountAllocatedIncrement(),
                            change.getResource().getType().getBaseUnit()));
                }
                if (change.getAmountReadyIncrement() > 0L || change.getAmountAllocatedIncrement() > 0L) {
                    updateBuilder.addChange(changeBuilder.build());
                }
            });
            if (!updateBuilder.getChanges().isEmpty()) {
                builder.addUpdate(updateBuilder.build());
            }
        });
        return builder.build();
    }

    @NotNull
    private DistributableQuota prepareDistributableQuota(@NotNull final QuotaDistribution quotaDistribution,
                                                         @NotNull final DistributeQuotaParams distributeQuotaParams) {
        final DistributableQuota.Builder builder = DistributableQuota.builder();
        builder.algorithm(distributeQuotaParams.getAlgorithm())
                .serviceKey(distributeQuotaParams.getService().getKey())
                .orderId(distributeQuotaParams.getBigOrder().getId())
                .campaignId(distributeQuotaParams.getCampaign().getId())
                .comment(distributeQuotaParams.getComment())
                .allocate(distributeQuotaParams.isAllocate());
        final Map<DistributionKey, List<QuotaDistribution.Change>> changesByKey = quotaDistribution.getChanges()
                .stream().collect(Collectors.groupingBy(DistributionKey::new));
        changesByKey.forEach((key, changes) -> {
            final DistributableQuota.Change.Builder changeBuilder = DistributableQuota.Change.builder();
            long amountReadyIncrement = 0L;
            for (final QuotaDistribution.Change change : changes) {
                amountReadyIncrement += change.getAmountReadyIncrement();
            }
            if (amountReadyIncrement > 0L) {
                changeBuilder
                        .resourceKey(key.getResource().getPublicKey())
                        .segmentKeys(key.getSegments().stream().map(Segment::getPublicKey).collect(Collectors.toSet()))
                        .amountReady(DiAmount.of(amountReadyIncrement, key.getResource().getType().getBaseUnit()));
                builder.addChange(changeBuilder.build());
            }
        });
        return builder.build();
    }

    @NotNull
    private List<QuotaChangeInRequest> findRequestsForDistribution(
            @NotNull final DistributeQuotaParams distributeQuotaParams, final boolean lock) {
        final Set<ResourceSegments> resourceSegments = distributeQuotaParams.getChanges().stream()
                .filter(c -> !c.getSegments().isEmpty()).map(c -> new ResourceSegments(c.getResource().getId(),
                        c.getSegments().stream().map(LongIndexBase::getId).collect(Collectors.toSet())))
                .collect(Collectors.toSet());
        final Set<Long> resourceIds = distributeQuotaParams.getChanges().stream().filter(c -> c.getSegments().isEmpty())
                .map(c -> c.getResource().getId()).collect(Collectors.toSet());
        final Set<QuotaChangeRequest.Status> statuses;
        if (distributeQuotaParams.isAllocate()) {
            statuses = Collections.singleton(QuotaChangeRequest.Status.CONFIRMED);
        } else {
            statuses = ImmutableSet.of(QuotaChangeRequest.Status.NEW, QuotaChangeRequest.Status.READY_FOR_REVIEW,
                    QuotaChangeRequest.Status.CONFIRMED);
        }
        final List<QuotaChangeInRequest> segmentedChanges = handleLockingFailures(() -> quotaChangeRequestDao
                .selectSegmentedChangesForQuotaDistributionUpdate(distributeQuotaParams.getCampaign().getId(),
                        distributeQuotaParams.getBigOrder().getId(), distributeQuotaParams.getService().getId(),
                        QuotaChangeRequest.Type.RESOURCE_PREORDER, statuses, resourceSegments, lock));
        final List<QuotaChangeInRequest> nonSegmentedChanges = handleLockingFailures(() -> quotaChangeRequestDao
                .selectNonSegmentedChangesForQuotaDistributionUpdate(distributeQuotaParams.getCampaign().getId(),
                        distributeQuotaParams.getBigOrder().getId(), distributeQuotaParams.getService().getId(),
                        QuotaChangeRequest.Type.RESOURCE_PREORDER, statuses, resourceIds, lock));
        final List<QuotaChangeInRequest> result = new ArrayList<>(segmentedChanges);
        result.addAll(nonSegmentedChanges);
        return result;
    }

    @NotNull
    private DiUnit getBaseUnit(@NotNull final Resource resource, @NotNull final Service service) {
        if (SERVICE_BASE_UNITS.containsKey(service.getKey()) && SERVICE_BASE_UNITS.get(service.getKey()).containsKey(resource.getType())) {
            return SERVICE_BASE_UNITS.get(service.getKey()).get(resource.getType());
        }
        if (BASE_UNITS.containsKey(resource.getType())) {
            return BASE_UNITS.get(resource.getType());
        }
        return resource.getType().getBaseUnit();
    }

    private <T> T handleLockingFailures(@NotNull final Supplier<T> supplier) {
        try {
            return supplier.get();
        } catch (CannotAcquireLockException e) {
            LOG.warn("Failed to acquire lock", e);
            throw new TooManyRequestsException(e);
        }
    }

    private void handleLockingFailures(@NotNull final Runnable runnable) {
        try {
            runnable.run();
        } catch (CannotAcquireLockException e) {
            LOG.warn("Failed to acquire lock", e);
            throw new TooManyRequestsException(e);
        }
    }

    private static final class DistributionKey {

        @NotNull
        private final Resource resource;
        @NotNull
        private final Set<Segment> segments;

        private DistributionKey(@NotNull final Resource resource, @NotNull final Set<Segment> segments) {
            this.resource = resource;
            this.segments = segments;
        }

        private DistributionKey(@NotNull final QuotaChangeInRequest request) {
            this.resource = request.getResource();
            this.segments = request.getSegments();
        }

        private DistributionKey(@NotNull final DistributeQuotaParams.Change change) {
            this.resource = change.getResource();
            this.segments = change.getSegments();
        }

        private DistributionKey(@NotNull final QuotaDistribution.Change change) {
            this.resource = change.getResource();
            this.segments = change.getSegments();
        }

        private DistributionKey(@NotNull final QuotaChangeRequest.Change change) {
            this.resource = change.getResource();
            this.segments = change.getSegments();
        }

        @NotNull
        public Resource getResource() {
            return resource;
        }

        @NotNull
        public Set<Segment> getSegments() {
            return segments;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final DistributionKey that = (DistributionKey) o;
            return resource.equals(that.resource) &&
                    segments.equals(that.segments);
        }

        @Override
        public int hashCode() {
            return Objects.hash(resource, segments);
        }

        @Override
        public String toString() {
            return "DistributionKey{" +
                    "resource=" + resource +
                    ", segments=" + segments +
                    '}';
        }

    }

}
