package ru.yandex.qe.dispenser.ws.api.impl;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.inject.Inject;

import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.ws.api.domain.distribution.DistributeQuotaParams;
import ru.yandex.qe.dispenser.ws.bot.BigOrderManager;
import ru.yandex.qe.dispenser.ws.common.domain.errors.ErrorCollection;
import ru.yandex.qe.dispenser.ws.common.domain.errors.TypedError;
import ru.yandex.qe.dispenser.ws.common.domain.result.Result;
import ru.yandex.qe.dispenser.ws.quota.request.workflow.context.PerformerContext;
import ru.yandex.qe.dispenser.ws.reqbody.DistributeQuotaBody;

@Component
public class QuotaDistributionValidationManager {

    @NotNull
    private final HierarchySupplier hierarchySupplier;
    @NotNull
    private final BigOrderManager bigOrderManager;
    @NotNull
    private final CampaignDao campaignDao;

    @Inject
    public QuotaDistributionValidationManager(@NotNull final HierarchySupplier hierarchySupplier,
                                              @NotNull final BigOrderManager bigOrderManager,
                                              @NotNull final CampaignDao campaignDao) {
        this.hierarchySupplier = hierarchySupplier;
        this.bigOrderManager = bigOrderManager;
        this.campaignDao = campaignDao;
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<DistributeQuotaParams, ErrorCollection<String, TypedError<String>>> validateDistributionParams(
            @NotNull final DistributeQuotaBody params,
            @NotNull final PerformerContext performerContext) {
        final Hierarchy hierarchy = hierarchySupplier.get();
        final ErrorCollection.Builder<String, TypedError<String>> errorsBuilder = ErrorCollection.builder();
        final DistributeQuotaParams.Builder distributeQuotaBuilder = DistributeQuotaParams.builder();
        if (params.getServiceKey() != null) {
            final Service service = hierarchy.getServiceReader().readOrNull(params.getServiceKey());
            if (service != null) {
                distributeQuotaBuilder.service(service);
            } else {
                errorsBuilder.addError("service", TypedError.invalid("Service with key ["
                        + params.getServiceKey() + "] is not found."));
            }
        } else {
            errorsBuilder.addError("serviceKey", TypedError.invalid("Service key is required."));
        }
        if (distributeQuotaBuilder.getService() == null) {
            return Result.failure(errorsBuilder.build());
        }
        final Set<Person> serviceTrustees = hierarchy.getServiceReader().getTrustees(distributeQuotaBuilder.getService());
        final Set<Person> serviceAdmins = hierarchy.getServiceReader().getAdmins(distributeQuotaBuilder.getService());
        final Set<Person> dispenserAdmins = hierarchy.getDispenserAdminsReader().getDispenserAdmins();
        if (!serviceTrustees.contains(performerContext.getPerson()) && !serviceAdmins.contains(performerContext.getPerson())
                && !dispenserAdmins.contains(performerContext.getPerson())) {
            errorsBuilder.addError(TypedError.forbidden("Only service trustees and admins may use this endpoint."));
            return Result.failure(errorsBuilder.build());
        }
        if (params.getAlgorithm() != null) {
            distributeQuotaBuilder.algorithm(params.getAlgorithm());
        } else {
            errorsBuilder.addError("algorithm", TypedError.invalid("Algorithm is required."));
        }
        if (params.getOrderId() != null) {
            final BigOrder bigOrder = bigOrderManager.getByIdCached(params.getOrderId());
            if (bigOrder != null) {
                distributeQuotaBuilder.bigOrder(bigOrder);
            } else {
                errorsBuilder.addError("orderId", TypedError.invalid("Big-order with id ["
                        + params.getOrderId() + "] is not found."));
            }
        } else {
            errorsBuilder.addError("orderId", TypedError.invalid("Big-order id is required."));
        }
        if (params.getCampaignId() != null) {
            final Optional<Campaign> campaign = campaignDao.readOptional(params.getCampaignId());
            if (campaign.isPresent()) {
                distributeQuotaBuilder.campaign(campaign.get());
            } else {
                errorsBuilder.addError("campaignId", TypedError.invalid("Campaign with id ["
                        + params.getCampaignId() + "] is not found."));
            }
        } else {
            errorsBuilder.addError("campaignId", TypedError.invalid("Campaign is required."));
        }
        if (distributeQuotaBuilder.getCampaign() != null && distributeQuotaBuilder.getBigOrder() != null) {
            final Set<Long> campaignBigOrderIds = distributeQuotaBuilder.getCampaign().getBigOrders().stream()
                    .map(Campaign.BigOrder::getBigOrderId).collect(Collectors.toSet());
            if (!campaignBigOrderIds.contains(distributeQuotaBuilder.getBigOrder().getId())) {
                errorsBuilder.addError("orderId", TypedError.invalid("Big-order is not in the campaign."));
            }
        }
        if (distributeQuotaBuilder.getCampaign() != null
                && distributeQuotaBuilder.getCampaign().getType() != Campaign.Type.AGGREGATED) {
            errorsBuilder.addError("campaignId", TypedError.invalid("Campaign type is not AGGREGATED."));
        }
        distributeQuotaBuilder.comment(params.getComment());
        distributeQuotaBuilder.allocate(params.getAllocate() != null ? params.getAllocate() : false);
        if (params.getChanges() != null) {
            if (params.getChanges().isEmpty()) {
                errorsBuilder.addError("changes", TypedError.invalid("Changes are required."));
            }
            long changeIndex = 0L;
            for (final DistributeQuotaBody.Change change : params.getChanges()) {
                if (change != null) {
                    validateChange(change, hierarchy, distributeQuotaBuilder, errorsBuilder, changeIndex);
                } else {
                    errorsBuilder.addError("changes[" + changeIndex + "]",
                            TypedError.invalid("Change may not be null."));
                }
                changeIndex++;
            }
        } else {
            errorsBuilder.addError("changes", TypedError.invalid("Changes are required."));
        }
        final Map<DistributionKey, Long> distributionKeysCount = new HashMap<>();
        distributeQuotaBuilder.getChanges().forEach(change -> {
            final DistributionKey key = new DistributionKey(change.getResource(), change.getSegments());
            distributionKeysCount.compute(key, (k, v) -> v == null ? 1L : v + 1L);
        });
        if (distributionKeysCount.values().stream().anyMatch(v -> v > 1L)) {
            errorsBuilder.addError("changes", TypedError.invalid("Duplicate quota distribution keys."));
        }
        if (distributeQuotaBuilder.isAllocate() && distributeQuotaBuilder.getService().getSettings().isManualQuotaAllocation()) {
            errorsBuilder.addError("allocate", TypedError.invalid("Auto allocation conflicts with manual allocation."));
        }
        if (errorsBuilder.hasAnyErrors()) {
            return Result.failure(errorsBuilder.build());
        }
        return Result.success(distributeQuotaBuilder.build());
    }

    private void validateChange(@NotNull final DistributeQuotaBody.Change changeBody, @NotNull final Hierarchy hierarchy,
                                @NotNull final DistributeQuotaParams.Builder distributeQuotaBuilder,
                                @NotNull final ErrorCollection.Builder<String, TypedError<String>> errorsBuilder,
                                final long changeIndex) {
        final DistributeQuotaParams.Change.Builder changeBuilder = DistributeQuotaParams.Change.builder();
        final ErrorCollection.Builder<String, TypedError<String>> changeErrorsBuilder = ErrorCollection.builder();
        if (changeBody.getResourceKey() != null) {
            if (distributeQuotaBuilder.getService() != null) {
                final Resource resource = hierarchy.getResourceReader().readOrNull(new Resource.Key(changeBody
                        .getResourceKey(), distributeQuotaBuilder.getService()));
                if (resource != null) {
                    changeBuilder.resource(resource);
                } else {
                    changeErrorsBuilder.addError("changes[" + changeIndex + "].resourceKey",
                            TypedError.invalid("Resource with key ["
                                    + changeBody.getResourceKey() + "] is not found."));
                }
            }
        } else {
            changeErrorsBuilder.addError("changes[" + changeIndex + "].resourceKey",
                    TypedError.invalid("Resource key is required."));
        }
        validateSegments(changeBody, hierarchy, changeBuilder, changeErrorsBuilder, changeIndex);
        if (changeBody.getAmountReady() != null) {
            if (changeBuilder.getResource() != null) {
                if (changeBuilder.getResource().getType().getBaseUnit().isConvertible(changeBody.getAmountReady())) {
                    final long amountReady = changeBuilder.getResource().getType().getBaseUnit()
                            .convert(changeBody.getAmountReady());
                    if (changeBody.getAmountReady().getValue() <= 0L || amountReady <= 0L) {
                        changeErrorsBuilder.addError("changes[" + changeIndex + "].amountReady",
                                TypedError.invalid("Amount ready value must be positive."));
                    } else {
                        changeBuilder.amountReady(amountReady);
                    }
                } else {
                    changeErrorsBuilder.addError("changes[" + changeIndex + "].amountReady",
                            TypedError.invalid("Amount ready units does not match resource units."));
                }
            }
        } else {
            changeErrorsBuilder.addError("changes[" + changeIndex + "].amountReady",
                    TypedError.invalid("Amount ready is required."));
        }
        if (changeBuilder.getSegments().stream().anyMatch(Segment::isAggregationSegment)) {
            changeErrorsBuilder.addError("changes[" + changeIndex + "].segmentKeys",
                    TypedError.invalid("Aggregation segments are not allowed."));
        }
        if (changeErrorsBuilder.hasAnyErrors()) {
            errorsBuilder.add(changeErrorsBuilder);
            return;
        }
        distributeQuotaBuilder.addChange(changeBuilder.build());
    }

    private void validateSegments(@NotNull final DistributeQuotaBody.Change changeBody, @NotNull final Hierarchy hierarchy,
                                  @NotNull final DistributeQuotaParams.Change.Builder changeBuilder,
                                  @NotNull final ErrorCollection.Builder<String, TypedError<String>> changeErrorsBuilder,
                                  final long changeIndex) {
        final ErrorCollection.Builder<String, TypedError<String>> segmentsErrorsBuilder = ErrorCollection.builder();
        if (changeBuilder.getResource() == null) {
            return;
        }
        final Set<Segmentation> resourceSegmentations =  hierarchy.getResourceSegmentationReader()
                .getSegmentations(changeBuilder.getResource());
        final List<String> segmentKeys = changeBody.getSegmentKeys() != null
                ? changeBody.getSegmentKeys() : Collections.emptyList();
        if (!segmentKeys.isEmpty() && resourceSegmentations.isEmpty()) {
            segmentsErrorsBuilder.addError("changes[" + changeIndex + "].segmentKeys",
                    TypedError.invalid("Resource is not segmented."));
            changeErrorsBuilder.add(segmentsErrorsBuilder);
            return;
        }
        if (segmentKeys.stream().distinct().count() != segmentKeys.size()) {
            segmentsErrorsBuilder.addError("changes[" + changeIndex + "].segmentKeys",
                    TypedError.invalid("Duplicated segment keys."));
        }
        final List<Segment> inputSegments = new ArrayList<>();
        long segmentKeyIndex = 0L;
        for (final String segmentKey : segmentKeys) {
            if (segmentKey != null) {
                validateSegment(segmentKey, inputSegments, resourceSegmentations, hierarchy, segmentsErrorsBuilder,
                        changeIndex, segmentKeyIndex);
            } else {
                segmentsErrorsBuilder.addError("changes[" + changeIndex + "].segmentKeys[" + segmentKeyIndex + "]",
                        TypedError.invalid("Segment key may not be null."));
            }
            segmentKeyIndex++;
        }
        final Set<Segmentation> inputSegmentations = inputSegments.stream().map(Segment::getSegmentation)
                .collect(Collectors.toSet());
        final Set<Segment> fullSegmentSet = new HashSet<>(inputSegments);
        resourceSegmentations.forEach(segmentation -> {
            if (!inputSegmentations.contains(segmentation)) {
                fullSegmentSet.add(Segment.totalOf(segmentation));
            }
        });
        final Map<Segmentation, List<Segment>> segmentationsWithMultipleSegments = fullSegmentSet.stream()
                .collect(Collectors.groupingBy(Segment::getSegmentation)).entrySet().stream()
                .filter(e -> e.getValue().size() > 1).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
        if (!segmentationsWithMultipleSegments.isEmpty()) {
            final String badSegmentations = segmentationsWithMultipleSegments.entrySet().stream()
                    .map(e -> e.getKey().getKey().getPublicKey() + ": " + e.getValue().stream()
                            .map(Segment::getPublicKey).collect(Collectors.joining(", ")))
                    .collect(Collectors.joining("; "));
            segmentsErrorsBuilder.addError("changes[" + changeIndex + "].segmentKeys",
                    TypedError.invalid("There are multiple segments for the same segmentation: "
                            + badSegmentations + "."));
        }
        if (segmentsErrorsBuilder.hasAnyErrors()) {
            changeErrorsBuilder.add(segmentsErrorsBuilder);
            return;
        }
        changeBuilder.segments(fullSegmentSet);
    }

    private void validateSegment(@NotNull final String segmentKey, @NotNull final List<Segment> segments,
                                 @NotNull final Set<Segmentation> resourceSegmentations, @NotNull final Hierarchy hierarchy,
                                 @NotNull final ErrorCollection.Builder<String, TypedError<String>> segmentsErrorsBuilder,
                                 final long changeIndex, final long segmentKeyIndex) {
        final Segment segment = hierarchy.getSegmentReader().readOrNull(segmentKey);
        if (segment != null) {
            if (resourceSegmentations.contains(segment.getSegmentation())) {
                segments.add(segment);
            } else {
                segmentsErrorsBuilder.addError("changes[" + changeIndex + "].segmentKeys[" + segmentKeyIndex + "]",
                        TypedError.invalid("Resource does not have a segment [" + segmentKey + "]."));
            }
        } else {
            segmentsErrorsBuilder.addError("changes[" + changeIndex + "].segmentKeys[" + segmentKeyIndex + "]",
                    TypedError.invalid("Segment with key [" + segmentKey + "] is not found."));
        }
    }

    private static final class DistributionKey {

        private final Resource resource;
        private final Set<Segment> segments;

        private DistributionKey(final Resource resource, final Set<Segment> segments) {
            this.resource = resource;
            this.segments = segments;
        }

        public Resource getResource() {
            return resource;
        }

        public Set<Segment> getSegments() {
            return segments;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final DistributionKey that = (DistributionKey) o;
            return Objects.equals(resource, that.resource) &&
                    Objects.equals(segments, that.segments);
        }

        @Override
        public int hashCode() {
            return Objects.hash(resource, segments);
        }

        @Override
        public String toString() {
            return "DistributionKey{" +
                    "resource=" + resource +
                    ", segments=" + segments +
                    '}';
        }

    }

}
