package ru.yandex.qe.dispenser.ws.api.model.distribution;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.v1.DiAmount;

@JsonInclude(JsonInclude.Include.NON_EMPTY)
public final class DistributableQuota {

    @NotNull
    private final ResourceDistributionAlgorithm algorithm;
    @NotNull
    private final String serviceKey;
    private final long orderId;
    private final long campaignId;
    @Nullable
    private final String comment;
    private final boolean allocate;
    @NotNull
    private final List<Change> changes;

    private DistributableQuota(
            @NotNull final ResourceDistributionAlgorithm algorithm,
            @NotNull final String serviceKey,
            final long orderId,
            final long campaignId,
            @Nullable final String comment,
            final boolean allocate,
            @NotNull final List<Change> changes) {
        this.algorithm = algorithm;
        this.serviceKey = serviceKey;
        this.orderId = orderId;
        this.campaignId = campaignId;
        this.comment = comment;
        this.allocate = allocate;
        this.changes = changes;
    }

    public static Builder builder() {
        return new Builder();
    }

    @NotNull
    public ResourceDistributionAlgorithm getAlgorithm() {
        return algorithm;
    }

    @NotNull
    public String getServiceKey() {
        return serviceKey;
    }

    public long getOrderId() {
        return orderId;
    }

    public long getCampaignId() {
        return campaignId;
    }

    @Nullable
    public String getComment() {
        return comment;
    }

    public boolean getAllocate() {
        return allocate;
    }

    @NotNull
    public List<Change> getChanges() {
        return changes;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final DistributableQuota that = (DistributableQuota) o;
        return orderId == that.orderId &&
                campaignId == that.campaignId &&
                allocate == that.allocate &&
                algorithm == that.algorithm &&
                serviceKey.equals(that.serviceKey) &&
                Objects.equals(comment, that.comment) &&
                changes.equals(that.changes);
    }

    @Override
    public int hashCode() {
        return Objects.hash(algorithm, serviceKey, orderId, campaignId, comment, allocate, changes);
    }

    @Override
    public String toString() {
        return "DistributableQuota{" +
                "algorithm=" + algorithm +
                ", serviceKey='" + serviceKey + '\'' +
                ", orderId=" + orderId +
                ", campaignId=" + campaignId +
                ", comment='" + comment + '\'' +
                ", allocate=" + allocate +
                ", changes=" + changes +
                '}';
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public static final class Change {

        @NotNull
        private final String resourceKey;
        @NotNull
        private final Set<String> segmentKeys;
        @NotNull
        private final DiAmount amountReady;

        private Change(
                @NotNull final String resourceKey,
                @NotNull final Set<String> segmentKeys,
                @NotNull final DiAmount amountReady) {
            this.resourceKey = resourceKey;
            this.segmentKeys = segmentKeys;
            this.amountReady = amountReady;
        }

        public static Builder builder() {
            return new Builder();
        }

        @NotNull
        public String getResourceKey() {
            return resourceKey;
        }

        @NotNull
        public Set<String> getSegmentKeys() {
            return segmentKeys;
        }

        @NotNull
        @JsonSerialize(using = DiAmount.CompactSerializer.class)
        public DiAmount getAmountReady() {
            return amountReady;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final Change change = (Change) o;
            return Objects.equals(resourceKey, change.resourceKey) &&
                    Objects.equals(segmentKeys, change.segmentKeys) &&
                    Objects.equals(amountReady, change.amountReady);
        }

        @Override
        public int hashCode() {
            return Objects.hash(resourceKey, segmentKeys, amountReady);
        }

        @Override
        public String toString() {
            return "Change{" +
                    "resourceKey='" + resourceKey + '\'' +
                    ", segmentKeys=" + segmentKeys +
                    ", amountReady=" + amountReady +
                    '}';
        }

        public static final class Builder {

            @Nullable
            private String resourceKey;
            @NotNull
            private Set<String> segmentKeys = new HashSet<>();
            @Nullable
            private DiAmount amountReady;

            private Builder() {
            }

            @NotNull
            public Builder resourceKey(@NotNull final String resourceKey) {
                Objects.requireNonNull(resourceKey, "Resource key is required.");
                this.resourceKey = resourceKey;
                return this;
            }

            @NotNull
            public Builder segmentKeys(@NotNull final Set<String> segmentKeys) {
                Objects.requireNonNull(segmentKeys, "Segment keys are required.");
                this.segmentKeys = segmentKeys;
                return this;
            }

            @NotNull
            public Builder amountReady(@NotNull final DiAmount amountReady) {
                Objects.requireNonNull(amountReady, "Amount ready is required.");
                this.amountReady = amountReady;
                return this;
            }

            @Nullable
            public String getResourceKey() {
                return resourceKey;
            }

            @NotNull
            public Set<String> getSegmentKeys() {
                return segmentKeys;
            }

            @Nullable
            public DiAmount getAmountReady() {
                return amountReady;
            }

            @NotNull
            public Change build() {
                Objects.requireNonNull(resourceKey, "Resource key is required.");
                Objects.requireNonNull(amountReady, "Amount ready is required.");
                return new Change(resourceKey, segmentKeys, amountReady);
            }

            @Override
            public boolean equals(final Object o) {
                if (this == o) {
                    return true;
                }
                if (o == null || getClass() != o.getClass()) {
                    return false;
                }
                final Builder builder = (Builder) o;
                return Objects.equals(resourceKey, builder.resourceKey) &&
                        segmentKeys.equals(builder.segmentKeys) &&
                        Objects.equals(amountReady, builder.amountReady);
            }

            @Override
            public int hashCode() {
                return Objects.hash(resourceKey, segmentKeys, amountReady);
            }

            @Override
            public String toString() {
                return "Builder{" +
                        "resourceKey='" + resourceKey + '\'' +
                        ", segmentKeys=" + segmentKeys +
                        ", amountReady=" + amountReady +
                        '}';
            }

        }

    }

    public static final class Builder {

        @Nullable
        private ResourceDistributionAlgorithm algorithm;
        @Nullable
        private String serviceKey;
        @Nullable
        private Long orderId;
        @Nullable
        private Long campaignId;
        @Nullable
        private String comment;
        @Nullable
        private Boolean allocate;
        @NotNull
        private final List<Change> changes = new ArrayList<>();

        private Builder() {
        }

        @NotNull
        public Builder algorithm(@NotNull final ResourceDistributionAlgorithm algorithm) {
            Objects.requireNonNull(algorithm, "Algorithm is required.");
            this.algorithm = algorithm;
            return this;
        }

        @NotNull
        public Builder serviceKey(@NotNull final String serviceKey) {
            Objects.requireNonNull(serviceKey, "Service key is required.");
            this.serviceKey = serviceKey;
            return this;
        }

        @NotNull
        public Builder orderId(final long orderId) {
            this.orderId = orderId;
            return this;
        }

        @NotNull
        public Builder campaignId(final long campaignId) {
            this.campaignId = campaignId;
            return this;
        }

        @NotNull
        public Builder comment(@Nullable final String comment) {
            this.comment = comment;
            return this;
        }

        @NotNull
        public Builder allocate(final boolean allocate) {
            this.allocate = allocate;
            return this;
        }

        @NotNull
        public Builder addChange(@NotNull final Change change) {
            Objects.requireNonNull(change, "Change is required.");
            this.changes.add(change);
            return this;
        }

        @Nullable
        public ResourceDistributionAlgorithm getAlgorithm() {
            return algorithm;
        }

        @Nullable
        public String getServiceKey() {
            return serviceKey;
        }

        @Nullable
        public Long getOrderId() {
            return orderId;
        }

        @Nullable
        public Long getCampaignId() {
            return campaignId;
        }

        @Nullable
        public String getComment() {
            return comment;
        }

        @Nullable
        public Boolean getAllocate() {
            return allocate;
        }

        @NotNull
        public List<Change> getChanges() {
            return changes;
        }

        public DistributableQuota build() {
            Objects.requireNonNull(algorithm, "Algorithm is required.");
            Objects.requireNonNull(serviceKey, "Service key is required.");
            Objects.requireNonNull(orderId, "Order id is required.");
            Objects.requireNonNull(campaignId, "Campaign id is required.");
            Objects.requireNonNull(allocate, "Allocate is required.");
            return new DistributableQuota(algorithm, serviceKey, orderId, campaignId, comment, allocate, changes);
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final Builder builder = (Builder) o;
            return algorithm == builder.algorithm &&
                    Objects.equals(serviceKey, builder.serviceKey) &&
                    Objects.equals(orderId, builder.orderId) &&
                    Objects.equals(campaignId, builder.campaignId) &&
                    Objects.equals(comment, builder.comment) &&
                    Objects.equals(allocate, builder.allocate) &&
                    changes.equals(builder.changes);
        }

        @Override
        public int hashCode() {
            return Objects.hash(algorithm, serviceKey, orderId, campaignId, comment, allocate, changes);
        }

        @Override
        public String toString() {
            return "Builder{" +
                    "algorithm=" + algorithm +
                    ", serviceKey='" + serviceKey + '\'' +
                    ", orderId=" + orderId +
                    ", campaignId=" + campaignId +
                    ", comment='" + comment + '\'' +
                    ", allocate=" + allocate +
                    ", changes=" + changes +
                    '}';
        }

    }

}
