package ru.yandex.qe.dispenser.ws.base_resources;

import java.io.UncheckedIOException;
import java.util.Set;

import javax.inject.Inject;
import javax.ws.rs.DELETE;
import javax.ws.rs.GET;
import javax.ws.rs.PATCH;
import javax.ws.rs.POST;
import javax.ws.rs.PUT;
import javax.ws.rs.Path;
import javax.ws.rs.PathParam;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import com.fasterxml.jackson.module.kotlin.KotlinModule;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import io.swagger.annotations.Authorization;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.stereotype.Service;

import ru.yandex.qe.dispenser.domain.hierarchy.Session;
import ru.yandex.qe.dispenser.swagger.DispenserSecurityDefinition;
import ru.yandex.qe.dispenser.swagger.SwaggerTags;
import ru.yandex.qe.dispenser.ws.Access;
import ru.yandex.qe.dispenser.ws.ServiceBase;
import ru.yandex.qe.dispenser.ws.base_resources.impl.BaseResourceLimitsManager;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitAmountDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitAmountInputDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitKeyDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitsPageDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitsReportDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitsReportInputDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.CreateBaseResourceLimitDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.ExpandBaseResourceLimit;
import ru.yandex.qe.dispenser.ws.base_resources.model.SingleBaseResourceLimitDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.UpdateBaseResourceLimitDto;
import ru.yandex.qe.dispenser.ws.common.domain.errors.ErrorCollection;
import ru.yandex.qe.dispenser.ws.common.domain.errors.TypedError;
import ru.yandex.qe.dispenser.ws.common.domain.result.Result;
import ru.yandex.qe.dispenser.ws.common.impl.ResultToResponse;
import ru.yandex.qe.dispenser.ws.common.model.response.ErrorResponse;

@Path("/v1/base-resource-limits")
@Produces(ServiceBase.APPLICATION_JSON_UTF_8)
@Service("base-resource-limits")
@Api(tags = {SwaggerTags.DISPENSER_API}, authorizations = {@Authorization(value = DispenserSecurityDefinition.AUTHORIZATION_SCHEME_NAME)})
public class BaseResourceLimitsService {

    private final BaseResourceLimitsManager baseResourceLimitsManager;
    private final ObjectWriter prettyReportWriter = prettyReportWriter();
    private final ObjectWriter prettySingleReportWriter = prettySingleReportWriter();

    @Inject
    public BaseResourceLimitsService(BaseResourceLimitsManager baseResourceLimitsManager) {
        this.baseResourceLimitsManager = baseResourceLimitsManager;
    }

    @GET
    @Access
    @NotNull
    @ApiOperation(value = "Get base resource limits page")
    @ApiResponses({@ApiResponse(code = 200, message = "Successful response", response = BaseResourceLimitsPageDto.class),
            @ApiResponse(code = 400, message = "Illegal argument response", response = ErrorResponse.class)})
    public Response getPage(@Nullable @QueryParam("from") Long from,
                            @Nullable @QueryParam("limit") Integer limit,
                            @Nullable @QueryParam("expand") Set<ExpandBaseResourceLimit> expand) {
        Result<BaseResourceLimitsPageDto, ErrorCollection<String, TypedError<String>>> result
                = baseResourceLimitsManager.getPage(from, limit, Session.USER_LOCALE.get(), expand);
        return ResultToResponse.toResponse(result);
    }

    @GET
    @Access
    @NotNull
    @Path("/{id}")
    @ApiOperation(value = "Get base resource limit")
    @ApiResponses({@ApiResponse(code = 200, message = "Successful response", response = SingleBaseResourceLimitDto.class),
            @ApiResponse(code = 404, message = "'Not found' response", response = ErrorResponse.class)})
    public Response getById(@PathParam("id") long id,
                            @Nullable @QueryParam("expand") Set<ExpandBaseResourceLimit> expand) {
        Result<SingleBaseResourceLimitDto, ErrorCollection<String, TypedError<String>>> result
                = baseResourceLimitsManager.getById(id, Session.USER_LOCALE.get(), expand);
        return ResultToResponse.toResponse(result);
    }

    @POST
    @Access
    @NotNull
    @Path("/_findByKey")
    @ApiOperation(value = "Find base resource limit by key")
    @ApiResponses({@ApiResponse(code = 200, message = "Successful response", response = SingleBaseResourceLimitDto.class),
            @ApiResponse(code = 404, message = "'Not found' response", response = ErrorResponse.class)})
    public Response findByKey(@Nullable @QueryParam("expand") Set<ExpandBaseResourceLimit> expand,
                              BaseResourceLimitKeyDto key) {
        Result<SingleBaseResourceLimitDto, ErrorCollection<String, TypedError<String>>> result
                = baseResourceLimitsManager.getByKey(key, Session.USER_LOCALE.get(), expand);
        return ResultToResponse.toResponse(result);
    }

    @POST
    @Access
    @NotNull
    @ApiOperation(value = "Create base resource limit")
    @ApiResponses({@ApiResponse(code = 200, message = "Successful response", response = SingleBaseResourceLimitDto.class),
            @ApiResponse(code = 400, message = "Illegal argument response", response = ErrorResponse.class)})
    public Response create(@Nullable @QueryParam("expand") Set<ExpandBaseResourceLimit> expand,
                           CreateBaseResourceLimitDto body) {
        Result<SingleBaseResourceLimitDto, ErrorCollection<String, TypedError<String>>> result
                = baseResourceLimitsManager.create(body, Session.WHOAMI.get(), Session.USER_LOCALE.get(), expand);
        return ResultToResponse.toResponse(result);
    }

    @PATCH
    @Access
    @NotNull
    @Path("/{id}")
    @ApiOperation(value = "Update base resource limit")
    @ApiResponses({@ApiResponse(code = 200, message = "Successful response", response = SingleBaseResourceLimitDto.class),
            @ApiResponse(code = 404, message = "'Not found' response", response = ErrorResponse.class),
            @ApiResponse(code = 400, message = "Illegal argument response", response = ErrorResponse.class)})
    public Response update(@PathParam("id") long id,
                           @Nullable @QueryParam("expand") Set<ExpandBaseResourceLimit> expand,
                           UpdateBaseResourceLimitDto body) {
        Result<SingleBaseResourceLimitDto, ErrorCollection<String, TypedError<String>>> result
                = baseResourceLimitsManager.update(id, body, Session.WHOAMI.get(), Session.USER_LOCALE.get(), expand);
        return ResultToResponse.toResponse(result);
    }

    @DELETE
    @Access
    @NotNull
    @Path("/{id}")
    @ApiOperation(value = "Delete base resource limit")
    @ApiResponses({@ApiResponse(code = 204, message = "Successful response"),
            @ApiResponse(code = 404, message = "'Not found' response", response = ErrorResponse.class)})
    public Response delete(@PathParam("id") long id) {
        Result<Void, ErrorCollection<String, TypedError<String>>> result
                = baseResourceLimitsManager.delete(id, Session.WHOAMI.get(), Session.USER_LOCALE.get());
        return ResultToResponse.toResponse(result, r -> Response.noContent().build(), true);
    }

    @GET
    @Access
    @NotNull
    @Path("/human-readable/_report")
    @ApiOperation(value = "Get base resource limits report for campaign and provider")
    @ApiResponses({@ApiResponse(code = 200, message = "Successful response", response = BaseResourceLimitsReportDto.class),
            @ApiResponse(code = 400, message = "Illegal argument response", response = ErrorResponse.class)})
    public Response getReport(@Nullable @QueryParam("campaignKey") String campaignKey,
                              @Nullable @QueryParam("providerKey") String providerKey) {
        Result<BaseResourceLimitsReportDto, ErrorCollection<String, TypedError<String>>> result
                = baseResourceLimitsManager.getReport(campaignKey, providerKey, Session.USER_LOCALE.get());
        return ResultToResponse.toResponse(result, r -> Response.ok(serializeToString(r, prettyReportWriter),
                MediaType.APPLICATION_JSON_TYPE).build(), true);
    }

    @PUT
    @Access
    @NotNull
    @Path("/human-readable/_report")
    @ApiOperation(value = "Put base resource limits report for campaign and provider")
    @ApiResponses({@ApiResponse(code = 200, message = "Successful response", response = BaseResourceLimitsReportDto.class),
            @ApiResponse(code = 400, message = "Illegal argument response", response = ErrorResponse.class)})
    public Response putReport(@Nullable @QueryParam("campaignKey") String campaignKey,
                              @Nullable @QueryParam("providerKey") String providerKey,
                              BaseResourceLimitsReportInputDto body) {
        Result<BaseResourceLimitsReportDto, ErrorCollection<String, TypedError<String>>> result
                = baseResourceLimitsManager.putReport(campaignKey, providerKey, body, Session.WHOAMI.get(),
                        Session.USER_LOCALE.get());
        return ResultToResponse.toResponse(result, r -> Response.ok(serializeToString(r, prettyReportWriter),
                MediaType.APPLICATION_JSON_TYPE).build(), true);
    }

    @GET
    @Access
    @NotNull
    @Path("/human-readable/_reportOne")
    @ApiOperation(value = "Get single base resource limit report for campaign and provider")
    @ApiResponses({@ApiResponse(code = 200, message = "Successful response", response = BaseResourceLimitAmountDto.class),
            @ApiResponse(code = 400, message = "Illegal argument response", response = ErrorResponse.class)})
    public Response getSingleReport(@Nullable @QueryParam("campaignKey") String campaignKey,
                                    @Nullable @QueryParam("baseResourceKey") String baseResourceKey) {
        Result<BaseResourceLimitAmountDto, ErrorCollection<String, TypedError<String>>> result
                = baseResourceLimitsManager.getSingleReport(campaignKey, baseResourceKey, Session.USER_LOCALE.get());
        return ResultToResponse.toResponse(result, r -> Response.ok(serializeToString(r, prettySingleReportWriter),
                MediaType.APPLICATION_JSON_TYPE).build(), true);
    }

    @PUT
    @Access
    @NotNull
    @Path("/human-readable/_reportOne")
    @ApiOperation(value = "Put single base resource limit report for campaign and provider")
    @ApiResponses({@ApiResponse(code = 200, message = "Successful response", response = BaseResourceLimitAmountDto.class),
            @ApiResponse(code = 400, message = "Illegal argument response", response = ErrorResponse.class)})
    public Response putSingleReport(@Nullable @QueryParam("campaignKey") String campaignKey,
                                    @Nullable @QueryParam("baseResourceKey") String baseResourceKey,
                                    BaseResourceLimitAmountInputDto body) {
        Result<BaseResourceLimitAmountDto, ErrorCollection<String, TypedError<String>>> result
                = baseResourceLimitsManager.putSingleReport(campaignKey, baseResourceKey, body, Session.WHOAMI.get(),
                Session.USER_LOCALE.get());
        return ResultToResponse.toResponse(result, r -> Response.ok(serializeToString(r, prettySingleReportWriter),
                MediaType.APPLICATION_JSON_TYPE).build(), true);
    }

    private static ObjectWriter prettyReportWriter() {
        ObjectMapper prettyMapper = new ObjectMapper();
        prettyMapper.registerModule(new Jdk8Module());
        prettyMapper.registerModule(new JavaTimeModule());
        prettyMapper.registerModule(new KotlinModule.Builder().build());
        return prettyMapper.writerFor(BaseResourceLimitsReportDto.class).withDefaultPrettyPrinter();
    }

    private static ObjectWriter prettySingleReportWriter() {
        ObjectMapper prettyMapper = new ObjectMapper();
        prettyMapper.registerModule(new Jdk8Module());
        prettyMapper.registerModule(new JavaTimeModule());
        prettyMapper.registerModule(new KotlinModule.Builder().build());
        return prettyMapper.writerFor(BaseResourceLimitAmountDto.class).withDefaultPrettyPrinter();
    }

    private <T> String serializeToString(T value, ObjectWriter writer) {
        try {
            return writer.writeValueAsString(value);
        } catch (JsonProcessingException e) {
            throw new UncheckedIOException(e);
        }
    }

}
