package ru.yandex.qe.dispenser.ws.base_resources.impl;

import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.inject.Inject;

import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiCampaign;
import ru.yandex.qe.dispenser.api.v1.DiResourceType;
import ru.yandex.qe.dispenser.api.v1.DiSegment;
import ru.yandex.qe.dispenser.api.v1.DiService;
import ru.yandex.qe.dispenser.api.v1.DiUnit;
import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResource;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceLimit;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceType;
import ru.yandex.qe.dispenser.domain.dao.base_resources.BaseResourceCache;
import ru.yandex.qe.dispenser.domain.dao.base_resources.BaseResourceDao;
import ru.yandex.qe.dispenser.domain.dao.base_resources.BaseResourceLimitDao;
import ru.yandex.qe.dispenser.domain.dao.base_resources.BaseResourceMappingDao;
import ru.yandex.qe.dispenser.domain.dao.base_resources.BaseResourceTypeCache;
import ru.yandex.qe.dispenser.domain.dao.base_resources.BaseResourceTypeDao;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignCache;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.dao.project.ProjectReader;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentReader;
import ru.yandex.qe.dispenser.domain.dao.service.ServiceDao;
import ru.yandex.qe.dispenser.domain.dao.service.ServiceReader;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.hierarchy.Role;
import ru.yandex.qe.dispenser.domain.i18n.LocalizableString;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.util.LocalizationUtils;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitAmountDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitAmountInputDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitKeyDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitReportDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitReportInputDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitsPageDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitsReportDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceLimitsReportInputDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceTypeDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.CreateBaseResourceLimitDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.ExpandBaseResourceLimit;
import ru.yandex.qe.dispenser.ws.base_resources.model.SingleBaseResourceLimitDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.UpdateBaseResourceLimitDto;
import ru.yandex.qe.dispenser.ws.common.domain.errors.ErrorCollection;
import ru.yandex.qe.dispenser.ws.common.domain.errors.TypedError;
import ru.yandex.qe.dispenser.ws.common.domain.result.Result;

@Component
public class BaseResourceLimitsManager {

    private final MessageSource errorMessageSource;
    private final BaseResourceLimitDao baseResourceLimitDao;
    private final BaseResourceCache baseResourceCache;
    private final BaseResourceTypeCache baseResourceTypeCache;
    private final HierarchySupplier hierarchySupplier;
    private final CampaignCache campaignCache;
    private final BaseResourceDao baseResourceDao;
    private final CampaignDao campaignDao;
    private final ServiceDao serviceDao;
    private final BaseResourceTypeDao baseResourceTypeDao;
    private final BaseResourceMappingDao baseResourceMappingDao;

    @Inject
    public BaseResourceLimitsManager(@Qualifier("errorMessageSource") MessageSource errorMessageSource,
                                     BaseResourceLimitDao baseResourceLimitDao,
                                     BaseResourceCache baseResourceCache,
                                     BaseResourceTypeCache baseResourceTypeCache,
                                     HierarchySupplier hierarchySupplier,
                                     CampaignCache campaignCache,
                                     BaseResourceDao baseResourceDao,
                                     CampaignDao campaignDao,
                                     ServiceDao serviceDao,
                                     BaseResourceTypeDao baseResourceTypeDao,
                                     BaseResourceMappingDao baseResourceMappingDao) {
        this.errorMessageSource = errorMessageSource;
        this.baseResourceLimitDao = baseResourceLimitDao;
        this.baseResourceCache = baseResourceCache;
        this.baseResourceTypeCache = baseResourceTypeCache;
        this.hierarchySupplier = hierarchySupplier;
        this.campaignCache = campaignCache;
        this.baseResourceDao = baseResourceDao;
        this.campaignDao = campaignDao;
        this.serviceDao = serviceDao;
        this.baseResourceTypeDao = baseResourceTypeDao;
        this.baseResourceMappingDao = baseResourceMappingDao;
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<BaseResourceLimitsPageDto, ErrorCollection<String, TypedError<String>>> getPage(
            Long from, Integer limit, Locale locale, Set<ExpandBaseResourceLimit> expand) {
        if (limit != null && (limit < 0 || limit > 100)) {
            return Result.failure(ErrorCollection.typedStringBuilder().addError("limit",
                    TypedError.badRequest(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                            LocalizableString.of("invalid.limit.value"), locale))).build());
        }
        int actualLimit = limit == null ? 100 : limit;
        Set<ExpandBaseResourceLimit> actualExpand = expand != null ? expand : Set.of();
        List<BaseResourceLimit> baseResourceLimits = baseResourceLimitDao.getPage(from, actualLimit);
        BaseResourceLimitsPageDto result = preparePageDto(baseResourceLimits, actualExpand);
        return Result.success(result);
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<SingleBaseResourceLimitDto, ErrorCollection<String, TypedError<String>>> getById(
            long id, Locale locale, Set<ExpandBaseResourceLimit> expand) {
        Optional<BaseResourceLimit> baseResourceLimitO = baseResourceLimitDao.getById(id);
        if (baseResourceLimitO.isEmpty()) {
            return Result.failure(ErrorCollection.typedStringBuilder().addError(
                    TypedError.notFound(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                            LocalizableString.of("base.resource.limit.not.found"), locale))).build());
        }
        Set<ExpandBaseResourceLimit> actualExpand = expand != null ? expand : Set.of();
        BaseResourceLimit baseResourceLimit = baseResourceLimitO.get();
        SingleBaseResourceLimitDto result = prepareSingleDto(baseResourceLimit, actualExpand);
        return Result.success(result);
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<SingleBaseResourceLimitDto, ErrorCollection<String, TypedError<String>>> getByKey(
            BaseResourceLimitKeyDto key, Locale locale, Set<ExpandBaseResourceLimit> expand) {
        if (key == null || key.getBaseResourceId().isEmpty() || key.getCampaignId().isEmpty()) {
            return Result.failure(ErrorCollection.typedStringBuilder().addError(
                    TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                            LocalizableString.of("invalid.base.resource.limit.key"), locale))).build());
        }
        BaseResourceLimit.Key keyToLoad = new BaseResourceLimit.Key(key.getBaseResourceId().get(),
                key.getCampaignId().get());
        Optional<BaseResourceLimit> baseResourceLimitO = baseResourceLimitDao.getByKey(keyToLoad);
        if (baseResourceLimitO.isEmpty()) {
            return Result.failure(ErrorCollection.typedStringBuilder().addError(
                    TypedError.notFound(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                            LocalizableString.of("base.resource.limit.not.found"), locale))).build());
        }
        Set<ExpandBaseResourceLimit> actualExpand = expand != null ? expand : Set.of();
        BaseResourceLimit baseResourceLimit = baseResourceLimitO.get();
        SingleBaseResourceLimitDto result = prepareSingleDto(baseResourceLimit, actualExpand);
        return Result.success(result);
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<SingleBaseResourceLimitDto, ErrorCollection<String, TypedError<String>>> create(
            CreateBaseResourceLimitDto body, Person author, Locale locale, Set<ExpandBaseResourceLimit> expand) {
        Result<BaseResourceLimit.Builder, ErrorCollection<String, TypedError<String>>> validated = validateCreate(
                body, author, locale);
        return validated.apply(builder -> {
            BaseResourceLimit baseResourceLimit = baseResourceLimitDao.create(builder);
            Set<ExpandBaseResourceLimit> actualExpand = expand != null ? expand : Set.of();
            return prepareSingleDto(baseResourceLimit, actualExpand);
        });
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<SingleBaseResourceLimitDto, ErrorCollection<String, TypedError<String>>> update(
            long id, UpdateBaseResourceLimitDto body, Person author, Locale locale,
            Set<ExpandBaseResourceLimit> expand) {
        Optional<BaseResourceLimit> baseResourceLimitO = baseResourceLimitDao.getById(id);
        if (baseResourceLimitO.isEmpty()) {
            return Result.failure(ErrorCollection.typedStringBuilder().addError(
                    TypedError.notFound(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                            LocalizableString.of("base.resource.limit.not.found"), locale))).build());
        }
        Result<BaseResourceLimit.Update, ErrorCollection<String, TypedError<String>>> validated = validateUpdate(
                baseResourceLimitO.get(), body, author, locale);
        return validated.andThen(update -> {
            Optional<BaseResourceLimit> updatedBaseResourceLimit = baseResourceLimitDao.update(update);
            if (updatedBaseResourceLimit.isEmpty()) {
                return Result.failure(ErrorCollection.typedStringBuilder().addError(
                        TypedError.notFound(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                                LocalizableString.of("base.resource.limit.not.found"), locale))).build());
            }
            Set<ExpandBaseResourceLimit> actualExpand = expand != null ? expand : Set.of();
            return Result.success(prepareSingleDto(updatedBaseResourceLimit.orElse(baseResourceLimitO.get()),
                    actualExpand));
        });
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<Void, ErrorCollection<String, TypedError<String>>> delete(long id, Person author, Locale locale) {
        Optional<BaseResourceLimit> baseResourceLimitO = baseResourceLimitDao.getById(id);
        if (baseResourceLimitO.isEmpty()) {
            return Result.failure(ErrorCollection.typedStringBuilder().addError(
                    TypedError.notFound(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                            LocalizableString.of("base.resource.limit.not.found"), locale))).build());
        }
        Result<Void, ErrorCollection<String, TypedError<String>>> validated = validateDelete(baseResourceLimitO.get(),
                author, locale);
        return validated.andThen(v -> {
            Optional<BaseResourceLimit> deletedBaseResourceLimit = baseResourceLimitDao.deleteById(id);
            if (deletedBaseResourceLimit.isEmpty()) {
                return Result.failure(ErrorCollection.typedStringBuilder().addError(
                        TypedError.notFound(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                                LocalizableString.of("base.resource.limit.not.found"), locale))).build());
            }
            return Result.success(v);
        });
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<BaseResourceLimitsReportDto, ErrorCollection<String, TypedError<String>>> getReport(
            String campaignKey, String providerKey, Locale locale) {
        ErrorCollection.Builder<String, TypedError<String>> errors = ErrorCollection.builder();
        Optional<Campaign> campaign = validateCampaign(campaignKey, locale, errors);
        Optional<Service> service = validateService(providerKey, locale, errors);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(prepareLimitsReport(campaign.orElseThrow(), service.orElseThrow()));
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<BaseResourceLimitsReportDto, ErrorCollection<String, TypedError<String>>> putReport(
            String campaignKey, String providerKey, BaseResourceLimitsReportInputDto body, Person author, Locale locale) {
        ErrorCollection.Builder<String, TypedError<String>> errors = ErrorCollection.builder();
        Optional<Campaign> campaign = validateCampaign(campaignKey, locale, errors);
        Optional<Service> service = validateService(providerKey, locale, errors);
        if (service.isPresent() && !hasPermissions(service.get(), author)) {
            errors.addError(TypedError.forbidden(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("not.enough.permissions"), locale)));
        }
        Set<ValidatedPutLimit> validatedLimits = new HashSet<>();
        validatePutBody(service.orElse(null), body, errors, validatedLimits, locale);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return doPutReport(campaign.orElseThrow(), service.orElseThrow(), validatedLimits, locale);
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<BaseResourceLimitAmountDto, ErrorCollection<String, TypedError<String>>> getSingleReport(
            String campaignKey, String baseResourceKey, Locale locale) {
        ErrorCollection.Builder<String, TypedError<String>> errors = ErrorCollection.builder();
        Optional<Campaign> campaign = validateCampaign(campaignKey, locale, errors);
        Optional<BaseResource> baseResource = validateBaseResource(baseResourceKey, locale, errors);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        BaseResourceType baseResourceType = baseResourceTypeDao
                .getById(baseResource.orElseThrow().getBaseResourceTypeId()).orElseThrow();
        Optional<BaseResourceLimit> limitO = baseResourceLimitDao
                .getByKey(new BaseResourceLimit.Key(baseResource.orElseThrow().getId(), campaign.orElseThrow().getId()));
        BaseResourceLimitAmountDto result = limitO.map(baseResourceLimit -> new BaseResourceLimitAmountDto(
                        limitToAmount(baseResourceLimit.getLimit(), baseResourceType)))
                .orElseGet(() -> new BaseResourceLimitAmountDto(limitToAmount(0L, baseResourceType)));
        return Result.success(result);
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<BaseResourceLimitAmountDto, ErrorCollection<String, TypedError<String>>> putSingleReport(
            String campaignKey, String baseResourceKey, BaseResourceLimitAmountInputDto body, Person author,
            Locale locale) {
        ErrorCollection.Builder<String, TypedError<String>> errors = ErrorCollection.builder();
        Optional<Campaign> campaign = validateCampaign(campaignKey, locale, errors);
        Optional<BaseResource> baseResource = validateBaseResource(baseResourceKey, locale, errors);
        Optional<ValidatedPutLimit> validated = validatePutSingleReport(baseResource.orElse(null), errors,
                body, author, locale);
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        Optional<BaseResourceLimit> existingLimit = baseResourceLimitDao.getByKey(
                new BaseResourceLimit.Key(baseResource.orElseThrow().getId(), campaign.orElseThrow().getId()));
        if (existingLimit.isPresent()) {
            BaseResourceLimit.Update update = BaseResourceLimit.update(existingLimit.get())
                    .limit(validated.orElseThrow().getValue());
            Optional<BaseResourceLimit> updated = baseResourceLimitDao.update(update);
            return Result.success(new BaseResourceLimitAmountDto(limitToAmount(updated.orElseThrow().getLimit(),
                    validated.orElseThrow().getBaseResourceType())));
        } else {
            BaseResourceLimit.Builder builder = BaseResourceLimit.builder()
                    .baseResourceId(validated.orElseThrow().getBaseResource().getId())
                    .campaignId(campaign.orElseThrow().getId())
                    .limit(validated.orElseThrow().getValue());
            BaseResourceLimit created = baseResourceLimitDao.create(builder);
            return Result.success(new BaseResourceLimitAmountDto(limitToAmount(created.getLimit(),
                    validated.orElseThrow().getBaseResourceType())));
        }
    }

    private Optional<Campaign> validateCampaign(String campaignKey, Locale locale,
                                                ErrorCollection.Builder<String, TypedError<String>> errors) {
        if (campaignKey == null) {
            errors.addError("campaignKey", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("field.is.required"), locale)));
            return Optional.empty();
        } else {
            Optional<Campaign> campaignO = campaignDao.getByKey(campaignKey);
            if (campaignO.isEmpty()) {
                errors.addError("campaignKey", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                        errorMessageSource, LocalizableString.of("campaign.not.found"), locale)));
            }
            return campaignO;
        }
    }

    private Optional<Service> validateService(String providerKey, Locale locale,
                                              ErrorCollection.Builder<String, TypedError<String>> errors) {
        if (providerKey == null) {
            errors.addError("providerKey", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("field.is.required"), locale)));
            return Optional.empty();
        } else {
            Service service = serviceDao.readOrNull(providerKey);
            if (service == null) {
                errors.addError("providerKey", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                        errorMessageSource, LocalizableString.of("resource.provider.not.found"), locale)));
            }
            return Optional.ofNullable(service);
        }
    }

    private Optional<BaseResource> validateBaseResource(String baseResourceKey, Locale locale,
                                                        ErrorCollection.Builder<String, TypedError<String>> errors) {
        if (baseResourceKey == null) {
            errors.addError("baseResourceKey", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("field.is.required"), locale)));
            return Optional.empty();
        } else {
            Optional<BaseResource> baseResourceO = baseResourceDao.getByKey(baseResourceKey);
            if (baseResourceO.isEmpty()) {
                errors.addError("baseResourceKey", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                        errorMessageSource, LocalizableString.of("base.resource.not.found"), locale)));
            }
            return baseResourceO;
        }
    }

    private Optional<ValidatedPutLimit> validatePutSingleReport(
            BaseResource baseResource, ErrorCollection.Builder<String, TypedError<String>> errors,
            BaseResourceLimitAmountInputDto body, Person author, Locale locale) {
        if (body == null || body.getLimit().isEmpty()) {
            errors.addError("body", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("field.is.required"), locale)));
        }
        if (baseResource == null) {
            return Optional.empty();
        }
        BaseResourceType baseResourceType = baseResourceTypeDao
                .getByIdForUpdate(baseResource.getBaseResourceTypeId()).orElseThrow();
        Service service = serviceDao.read(baseResourceType.getServiceId());
        if (!hasPermissions(service, author)) {
            errors.addError(TypedError.forbidden(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                    LocalizableString.of("not.enough.permissions"), locale)));
        }
        if (body == null || body.getLimit().isEmpty()) {
            return Optional.empty();
        }
        DiAmount amount = body.getLimit().get();
        if (amount.getValue() < 0) {
            errors.addError(TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("negative.limit.value.for.base.resource.0",
                            baseResource.getKey()), locale)));
            return Optional.empty();
        }
        if (!baseResourceType.getResourceType().getBaseUnit().isConvertible(amount)) {
            errors.addError(TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("invalid.unit.for.base.resource.0",
                            baseResource.getKey()), locale)));
            return Optional.empty();
        }
        BigInteger converted = baseResourceType.getResourceType().getBaseUnit()
                .convertInteger(BigInteger.valueOf(amount.getValue()), amount.getUnit());
        if (converted.compareTo(BigInteger.valueOf(Long.MAX_VALUE)) > 0) {
            errors.addError(TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                    LocalizableString.of("limit.value.is.out.of.range.for.base.resource.0",
                            baseResource.getKey()), locale)));
            return Optional.empty();
        }
        return Optional.of(new ValidatedPutLimit(baseResource, baseResourceType,
                converted.longValueExact()));
    }

    private Result<BaseResourceLimitsReportDto, ErrorCollection<String, TypedError<String>>> doPutReport(
            Campaign campaign, Service service, Set<ValidatedPutLimit> validatedLimits, Locale locale) {
        Map<Long, ValidatedPutLimit> validatedLimitsByBaseResourceId = validatedLimits.stream()
                .collect(Collectors.toMap(l -> l.getBaseResource().getId(), Function.identity()));
        Set<BaseResourceType> serviceBaseResourceTypes = baseResourceTypeDao.getByServiceIdForUpdate(service.getId());
        Map<Long, BaseResourceType> baseResourceTypesById = serviceBaseResourceTypes.stream()
                .collect(Collectors.toMap(BaseResourceType::getId, Function.identity()));
        List<Long> baseResourceTypeIds = serviceBaseResourceTypes.stream().map(BaseResourceType::getId)
                .distinct().collect(Collectors.toList());
        Set<BaseResource> serviceBaseResources = splitLoad(baseResourceTypeIds,
                baseResourceDao::getByBaseResourceTypeIds);
        Map<Long, BaseResource> baseResourceById = serviceBaseResources.stream()
                .collect(Collectors.toMap(BaseResource::getId, Function.identity()));
        List<Long> baseResourceIds = serviceBaseResources.stream().map(BaseResource::getId)
                .distinct().collect(Collectors.toList());
        Set<Long> mappedBaseResourceIds = splitLoad(baseResourceIds,
                page -> baseResourceMappingDao.getMappedBaseResourceIds(campaign.getId(), page));
        Set<BaseResourceLimit> existingLimits = splitLoad(baseResourceIds,
                page -> baseResourceLimitDao.getByCampaignAndBaseResources(campaign.getId(), page));
        Map<Long, BaseResourceLimit> existingLimitsByBaseResourceId = existingLimits.stream()
                .collect(Collectors.toMap(BaseResourceLimit::getBaseResourceId, Function.identity()));
        List<String> unmappedBaseResourceKeys = validatedLimits.stream()
                .filter(l -> !mappedBaseResourceIds.contains(l.getBaseResource().getId())
                        && !existingLimitsByBaseResourceId.containsKey(l.getBaseResource().getId()))
                .sorted(Comparator.comparing(l -> l.getBaseResource().getId()))
                .map(l -> l.getBaseResource().getKey()).collect(Collectors.toList());
        if (!unmappedBaseResourceKeys.isEmpty()) {
            return Result.failure(ErrorCollection.typedStringBuilder().addError(TypedError
                    .invalid(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                            LocalizableString.of("unmapped.base.resources.0",
                                    String.join(", ", unmappedBaseResourceKeys)), locale))).build());
        }
        List<Long> limitIdsToDelete = new ArrayList<>();
        existingLimits.forEach(limit -> {
            if (!validatedLimitsByBaseResourceId.containsKey(limit.getBaseResourceId())) {
                limitIdsToDelete.add(limit.getId());
            }
        });
        List<BaseResourceLimit.Update> updates = new ArrayList<>();
        List<BaseResourceLimit.Builder> inserts = new ArrayList<>();
        validatedLimits.forEach(limit -> {
            if (existingLimitsByBaseResourceId.containsKey(limit.getBaseResource().getId())) {
                BaseResourceLimit existingLimit = existingLimitsByBaseResourceId.get(limit.getBaseResource().getId());
                updates.add(BaseResourceLimit.update(existingLimit).limit(limit.getValue()));
            } else {
                inserts.add(BaseResourceLimit.builder()
                        .baseResourceId(limit.getBaseResource().getId())
                        .campaignId(campaign.getId())
                        .limit(limit.getValue()));
            }
        });
        splitLoad(limitIdsToDelete, baseResourceLimitDao::deleteByIds);
        List<BaseResourceLimit> createdLimits = Lists.partition(inserts, 100).stream()
                .flatMap(page -> baseResourceLimitDao.create(page).stream()).collect(Collectors.toList());
        List<BaseResourceLimit> updatedLimits = Lists.partition(updates, 100).stream()
                .flatMap(page -> baseResourceLimitDao.update(page).stream()).collect(Collectors.toList());
        List<BaseResourceLimitReportDto> limitsResult = new ArrayList<>();
        List<BaseResourceLimit> remainingLimits = Stream.concat(createdLimits.stream(), updatedLimits.stream())
                .collect(Collectors.toList());
        Map<Long, BaseResourceLimit> remainingLimitsByBaseResourceId = remainingLimits.stream()
                .collect(Collectors.toMap(BaseResourceLimit::getBaseResourceId, Function.identity()));
        Set<Long> allMatchingBaseResourceIds = Sets.union(remainingLimitsByBaseResourceId.keySet(),
                mappedBaseResourceIds);
        List<Long> sortedBaseResourceIds = allMatchingBaseResourceIds.stream().sorted().collect(Collectors.toList());
        sortedBaseResourceIds.forEach(baseResourceId -> {
            BaseResource baseResource = baseResourceById.get(baseResourceId);
            BaseResourceType baseResourceType = baseResourceTypesById.get(baseResource.getBaseResourceTypeId());
            if (remainingLimitsByBaseResourceId.containsKey(baseResourceId)) {
                BaseResourceLimit existingLimit = remainingLimitsByBaseResourceId.get(baseResourceId);
                BaseResourceLimitReportDto reportLimit = new BaseResourceLimitReportDto(baseResource.getKey(),
                        limitToAmount(existingLimit.getLimit(), baseResourceType));
                limitsResult.add(reportLimit);
            } else {
                BaseResourceLimitReportDto reportLimit = new BaseResourceLimitReportDto(baseResource.getKey(),
                        limitToAmount(0L, baseResourceType));
                limitsResult.add(reportLimit);
            }
        });
        return Result.success(new BaseResourceLimitsReportDto(limitsResult));
    }

    private void validatePutBody(Service service,
                                 BaseResourceLimitsReportInputDto body,
                                 ErrorCollection.Builder<String, TypedError<String>> errors,
                                 Set<ValidatedPutLimit> validatedLimits,
                                 Locale locale) {
        if (body == null || body.getLimits().isEmpty()) {
            errors.addError("limits", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("field.is.required"), locale)));
            return;
        }
        List<BaseResourceLimitReportInputDto> limitsToCheckFurther = new ArrayList<>();
        for (int i = 0; i < body.getLimits().get().size(); i++) {
            BaseResourceLimitReportInputDto limit = body.getLimits().get().get(i);
            if (limit == null) {
                errors.addError("limits." + i, TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                        errorMessageSource, LocalizableString.of("field.is.required"), locale)));
                continue;
            }
            boolean checkFurther = true;
            if (limit.getLimit().isEmpty()) {
                errors.addError("limits." + i + ".limit", TypedError.invalid(LocalizationUtils
                        .resolveWithDefaultAsKey(errorMessageSource, LocalizableString.of("field.is.required"),
                                locale)));
                checkFurther = false;
            }
            if (limit.getBaseResourceKey().isEmpty() || limit.getBaseResourceKey().get().isEmpty()) {
                errors.addError("limits." + i + ".baseResourceKey", TypedError.invalid(LocalizationUtils
                        .resolveWithDefaultAsKey(errorMessageSource, LocalizableString.of("field.is.required"),
                                locale)));
                checkFurther = false;
            }
            if (checkFurther) {
                limitsToCheckFurther.add(limit);
            }
        }
        List<String> baseResourceKeys = limitsToCheckFurther.stream().map(l -> l.getBaseResourceKey().orElseThrow())
                .collect(Collectors.toList());
        List<String> uniqueBaseResourceKeys = baseResourceKeys.stream().distinct().collect(Collectors.toList());
        if (baseResourceKeys.size() != uniqueBaseResourceKeys.size()) {
            errors.addError(TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                    LocalizableString.of("duplicated.base.resources.not.allowed"), locale)));
            return;
        }
        if (service == null) {
            return;
        }
        Set<BaseResource> baseResources = splitLoadKeys(uniqueBaseResourceKeys, baseResourceDao::getByKeys);
        List<Long> baseResourceTypeIds = baseResources.stream().map(BaseResource::getBaseResourceTypeId)
                .distinct().collect(Collectors.toList());
        Set<BaseResourceType> baseResourceTypes = splitLoad(baseResourceTypeIds, baseResourceTypeDao::getByIds);
        Map<String, BaseResource> baseResourcesByKey = baseResources.stream()
                .collect(Collectors.toMap(BaseResource::getKey, Function.identity()));
        Map<Long, BaseResourceType> baseResourceTypesById = baseResourceTypes.stream()
                .collect(Collectors.toMap(BaseResourceType::getId, Function.identity()));
        for (BaseResourceLimitReportInputDto limit : limitsToCheckFurther) {
            String baseResourceKey = limit.getBaseResourceKey().orElseThrow();
            DiAmount amount = limit.getLimit().orElseThrow();
            if (!baseResourcesByKey.containsKey(baseResourceKey)) {
                errors.addError(TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                        LocalizableString.of("base.resource.0.not.found", baseResourceKey), locale)));
                continue;
            }
            BaseResource baseResource = baseResourcesByKey.get(baseResourceKey);
            BaseResourceType baseResourceType = baseResourceTypesById.get(baseResource.getBaseResourceTypeId());
            if (baseResourceType.getServiceId() != service.getId()) {
                errors.addError(TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                        LocalizableString.of("base.resource.0.not.found", baseResourceKey), locale)));
                continue;
            }
            if (amount.getValue() < 0) {
                errors.addError(TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                        errorMessageSource, LocalizableString.of("negative.limit.value.for.base.resource.0",
                                baseResourceKey), locale)));
            } else {
                if (!baseResourceType.getResourceType().getBaseUnit().isConvertible(amount)) {
                    errors.addError(TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                            errorMessageSource, LocalizableString.of("invalid.unit.for.base.resource.0",
                                    baseResourceKey), locale)));
                } else {
                    BigInteger converted = baseResourceType.getResourceType().getBaseUnit()
                            .convertInteger(BigInteger.valueOf(amount.getValue()), amount.getUnit());
                    if (converted.compareTo(BigInteger.valueOf(Long.MAX_VALUE)) > 0) {
                        errors.addError(TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                                LocalizableString.of("limit.value.is.out.of.range.for.base.resource.0",
                                        baseResourceKey), locale)));
                    } else {
                        validatedLimits.add(new ValidatedPutLimit(baseResource, baseResourceType,
                                converted.longValueExact()));
                    }
                }
            }
        }
    }

    private BaseResourceLimitsReportDto prepareLimitsReport(Campaign campaign, Service service) {
        List<BaseResourceLimitReportDto> limitsResult = new ArrayList<>();
        Set<BaseResourceType> serviceBaseResourceTypes = baseResourceTypeDao.getByServiceId(service.getId());
        Map<Long, BaseResourceType> baseResourceTypesById = serviceBaseResourceTypes.stream()
                .collect(Collectors.toMap(BaseResourceType::getId, Function.identity()));
        List<Long> baseResourceTypeIds = serviceBaseResourceTypes.stream().map(BaseResourceType::getId)
                .distinct().collect(Collectors.toList());
        Set<BaseResource> serviceBaseResources = splitLoad(baseResourceTypeIds,
                baseResourceDao::getByBaseResourceTypeIds);
        Map<Long, BaseResource> baseResourceById = serviceBaseResources.stream()
                .collect(Collectors.toMap(BaseResource::getId, Function.identity()));
        List<Long> baseResourceIds = serviceBaseResources.stream().map(BaseResource::getId)
                .distinct().collect(Collectors.toList());
        Set<BaseResourceLimit> existingLimits = splitLoad(baseResourceIds,
                page -> baseResourceLimitDao.getByCampaignAndBaseResources(campaign.getId(), page));
        Map<Long, BaseResourceLimit> existingLimitsByBaseResourceId = existingLimits.stream()
                .collect(Collectors.toMap(BaseResourceLimit::getBaseResourceId, Function.identity()));
        Set<Long> mappedBaseResourceIds = splitLoad(baseResourceIds,
                page -> baseResourceMappingDao.getMappedBaseResourceIds(campaign.getId(), page));
        Set<Long> allMatchingBaseResourceIds = Sets.union(existingLimitsByBaseResourceId.keySet(),
                mappedBaseResourceIds);
        List<Long> sortedBaseResourceIds = allMatchingBaseResourceIds.stream().sorted().collect(Collectors.toList());
        sortedBaseResourceIds.forEach(baseResourceId -> {
            BaseResource baseResource = baseResourceById.get(baseResourceId);
            BaseResourceType baseResourceType = baseResourceTypesById.get(baseResource.getBaseResourceTypeId());
            if (existingLimitsByBaseResourceId.containsKey(baseResourceId)) {
                BaseResourceLimit existingLimit = existingLimitsByBaseResourceId.get(baseResourceId);
                BaseResourceLimitReportDto reportLimit = new BaseResourceLimitReportDto(baseResource.getKey(),
                        limitToAmount(existingLimit.getLimit(), baseResourceType));
                limitsResult.add(reportLimit);
            } else {
                BaseResourceLimitReportDto reportLimit = new BaseResourceLimitReportDto(baseResource.getKey(),
                        limitToAmount(0L, baseResourceType));
                limitsResult.add(reportLimit);
            }
        });
        return new BaseResourceLimitsReportDto(limitsResult);
    }

    private DiAmount limitToAmount(long limitValue, BaseResourceType baseResourceType)  {
        return DiUnit.largestIntegerAmount(limitValue, baseResourceType.getResourceType().getBaseUnit());
    }

    private <T> Set<T> splitLoad(List<Long> ids, Function<Collection<? extends Long>, Set<T>> loader) {
        return Lists.partition(ids, 500).stream()
                .flatMap(page -> loader.apply(page).stream()).collect(Collectors.toSet());
    }

    private <T> Set<T> splitLoadKeys(List<String> keys, Function<Collection<? extends String>, Set<T>> loader) {
        return Lists.partition(keys, 500).stream()
                .flatMap(page -> loader.apply(page).stream()).collect(Collectors.toSet());
    }

    private Result<BaseResourceLimit.Builder, ErrorCollection<String, TypedError<String>>> validateCreate(
            CreateBaseResourceLimitDto body, Person author, Locale locale) {
        BaseResourceLimit.Builder builder = BaseResourceLimit.builder();
        ErrorCollection.Builder<String, TypedError<String>> errors = ErrorCollection.builder();
        Optional<BaseResource> baseResourceO = Optional.empty();
        Optional<BaseResourceType> baseResourceTypeO = Optional.empty();
        if (body.getBaseResourceId().isEmpty()) {
            errors.addError("baseResourceId", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("field.is.required"), locale)));
        } else {
            baseResourceO = baseResourceDao.getById(body.getBaseResourceId().get());
            if (baseResourceO.isEmpty()) {
                errors.addError("baseResourceId", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                        errorMessageSource, LocalizableString.of("base.resource.not.found"), locale)));
            } else {
                builder.baseResourceId(baseResourceO.get().getId());
                baseResourceTypeO = baseResourceTypeDao.getByIdForUpdate(baseResourceO.get().getBaseResourceTypeId());
            }
        }
        Optional<Campaign> campaignO = Optional.empty();
        if (body.getCampaignId().isEmpty()) {
            errors.addError("campaignId", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("field.is.required"), locale)));
        } else {
            campaignO = campaignDao.readOptional(body.getCampaignId().get());
            if (campaignO.isEmpty()) {
                errors.addError("campaignId", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                        errorMessageSource, LocalizableString.of("campaign.not.found"), locale)));
            } else {
                builder.campaignId(campaignO.get().getId());
            }
        }
        if (body.getLimit().isEmpty()) {
            errors.addError("limit", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("field.is.required"), locale)));
        } else {
            if (body.getLimit().get().getValue() < 0) {
                errors.addError("limit", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                        errorMessageSource, LocalizableString.of("value.out.of.range"), locale)));
            } else if (baseResourceTypeO.isPresent()) {
                if (!baseResourceTypeO.get().getResourceType().getBaseUnit().isConvertible(body.getLimit().get())) {
                    errors.addError("limit", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                            errorMessageSource, LocalizableString.of("value.out.of.range"), locale)));
                } else {
                    BigInteger converted = baseResourceTypeO.get().getResourceType().getBaseUnit()
                            .convertInteger(BigInteger.valueOf(body.getLimit().get().getValue()),
                                    body.getLimit().get().getUnit());
                    if (converted.compareTo(BigInteger.valueOf(Long.MAX_VALUE)) > 0) {
                        errors.addError("limit", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                                errorMessageSource, LocalizableString.of("value.out.of.range"), locale)));
                    } else {
                        builder.limit(converted.longValueExact());
                    }
                }
            }
        }
        if (baseResourceO.isPresent() && campaignO.isPresent()) {
            BaseResourceLimit.Key key = new BaseResourceLimit.Key(baseResourceO.get().getId(), campaignO.get().getId());
            Optional<BaseResourceLimit> existingO = baseResourceLimitDao.getByKey(key);
            if (existingO.isPresent()) {
                errors.addError(TypedError.conflict(LocalizationUtils.resolveWithDefaultAsKey(
                        errorMessageSource, LocalizableString.of("conflicting.base.resource.limit.already.exists"),
                        locale)));
            }
        }
        if (baseResourceTypeO.isPresent()) {
            Service service = serviceDao.read(baseResourceTypeO.get().getServiceId());
            if (!hasPermissions(service, author)) {
                errors.addError(TypedError.forbidden(LocalizationUtils.resolveWithDefaultAsKey(
                        errorMessageSource, LocalizableString.of("not.enough.permissions"), locale)));
            }
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(builder);
    }

    private Result<BaseResourceLimit.Update, ErrorCollection<String, TypedError<String>>> validateUpdate(
            BaseResourceLimit currentLimit, UpdateBaseResourceLimitDto body, Person author, Locale locale) {
        BaseResourceLimit.Update update = BaseResourceLimit.update(currentLimit);
        ErrorCollection.Builder<String, TypedError<String>> errors = ErrorCollection.builder();
        BaseResource baseResource = baseResourceDao.getById(currentLimit.getBaseResourceId()).orElseThrow();
        BaseResourceType baseResourceType = baseResourceTypeDao.getByIdForUpdate(baseResource.getBaseResourceTypeId())
                .orElseThrow();
        Service service = serviceDao.read(baseResourceType.getServiceId());
        if (body.getLimit().isEmpty()) {
            errors.addError("limit", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("field.is.required"), locale)));
        } else {
            if (body.getLimit().get().getValue() < 0) {
                errors.addError("limit", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                        errorMessageSource, LocalizableString.of("value.out.of.range"), locale)));
            } else {
                if (!baseResourceType.getResourceType().getBaseUnit().isConvertible(body.getLimit().get())) {
                    errors.addError("limit", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                            errorMessageSource, LocalizableString.of("value.out.of.range"), locale)));
                } else {
                    BigInteger converted = baseResourceType.getResourceType().getBaseUnit()
                            .convertInteger(BigInteger.valueOf(body.getLimit().get().getValue()),
                                    body.getLimit().get().getUnit());
                    if (converted.compareTo(BigInteger.valueOf(Long.MAX_VALUE)) > 0) {
                        errors.addError("limit", TypedError.invalid(LocalizationUtils.resolveWithDefaultAsKey(
                                errorMessageSource, LocalizableString.of("value.out.of.range"), locale)));
                    } else {
                        update.limit(converted.longValueExact());
                    }
                }
            }
        }
        if (!hasPermissions(service, author)) {
            errors.addError(TypedError.forbidden(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("not.enough.permissions"), locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(update);
    }

    private Result<Void, ErrorCollection<String, TypedError<String>>> validateDelete(BaseResourceLimit currentLimit,
                                                                                     Person author, Locale locale) {
        ErrorCollection.Builder<String, TypedError<String>> errors = ErrorCollection.builder();
        BaseResource baseResource = baseResourceDao.getById(currentLimit.getBaseResourceId()).orElseThrow();
        BaseResourceType baseResourceType = baseResourceTypeDao.getByIdForUpdate(baseResource.getBaseResourceTypeId())
                .orElseThrow();
        Service service = serviceDao.read(baseResourceType.getServiceId());
        if (!hasPermissions(service, author)) {
            errors.addError(TypedError.forbidden(LocalizationUtils.resolveWithDefaultAsKey(
                    errorMessageSource, LocalizableString.of("not.enough.permissions"), locale)));
        }
        if (errors.hasAnyErrors()) {
            return Result.failure(errors.build());
        }
        return Result.success(null);
    }

    private boolean hasPermissions(Service service, Person author) {
        Hierarchy hierarchy = hierarchySupplier.get();
        if (hierarchy.getDispenserAdminsReader().getDispenserAdmins().contains(author)) {
            return true;
        }
        Set<Long> adminServiceIds = hierarchy.getServiceReader().getAdminServices(author).stream()
                .map(LongIndexBase::getId).collect(Collectors.toSet());
        if (adminServiceIds.contains(service.getId())) {
            return true;
        }
        ProjectReader projectReader = hierarchy.getProjectReader();
        Project root = projectReader.getRoot();
        if (projectReader.hasRole(author, root, Role.PROCESS_RESPONSIBLE)) {
            return true;
        }
        return false;
    }

    private BaseResourceLimitsPageDto preparePageDto(List<BaseResourceLimit> baseResourceLimits,
                                                     Set<ExpandBaseResourceLimit> expand) {
        Set<Long> baseResourceIds = baseResourceLimits.stream().map(BaseResourceLimit::getBaseResourceId)
                .collect(Collectors.toSet());
        boolean expandBaseResources = expand.contains(ExpandBaseResourceLimit.BASE_RESOURCES);
        Set<BaseResource> loadedBaseResources = baseResourceCache.getByIds(baseResourceIds);
        Map<Long, BaseResource> loadedBaseResourceById = loadedBaseResources.stream()
                .collect(Collectors.toMap(BaseResource::getId, Function.identity()));
        Set<BaseResource> baseResources = expandBaseResources ? loadedBaseResources : Set.of();
        Set<Long> baseResourceTypeIds = loadedBaseResources.stream().map(BaseResource::getBaseResourceTypeId)
                .collect(Collectors.toSet());
        boolean expandTypes = expand.contains(ExpandBaseResourceLimit.BASE_RESOURCE_TYPES);
        Set<BaseResourceType> loadedBaseResourceTypes = baseResourceTypeCache.getByIds(baseResourceTypeIds);
        Map<Long, BaseResourceType> loadedBaseResourceTypesById = loadedBaseResourceTypes.stream()
                .collect(Collectors.toMap(BaseResourceType::getId, Function.identity()));
        Set<BaseResourceType> baseResourceTypes = expandTypes && expandBaseResources
                ? loadedBaseResourceTypes : Set.of();
        Map<Long, DiResourceType> resourceTypeByBaseResourceId = new HashMap<>();
        loadedBaseResourceById.forEach((k, v) -> resourceTypeByBaseResourceId
                .put(k, loadedBaseResourceTypesById.get(v.getBaseResourceTypeId()).getResourceType()));
        Set<Long> campaignIds = baseResourceLimits.stream().map(BaseResourceLimit::getCampaignId)
                .collect(Collectors.toSet());
        boolean expandCampaigns = expand.contains(ExpandBaseResourceLimit.CAMPAIGNS);
        Set<Campaign> campaigns = expandCampaigns ? campaignCache.getByIds(campaignIds) : Set.of();
        Hierarchy hierarchy = hierarchySupplier.get();
        ServiceReader serviceReader = hierarchy.getServiceReader();
        SegmentReader segmentReader = hierarchy.getSegmentReader();
        Set<Long> segmentIds = baseResources.stream().flatMap(r -> r.getSegmentIds().stream())
                .collect(Collectors.toSet());
        Set<Long> serviceIds = baseResourceTypes.stream().map(BaseResourceType::getServiceId)
                .collect(Collectors.toSet());
        List<BaseResourceLimitDto> baseResourceLimitDto = baseResourceLimits.stream()
                .map(r -> new BaseResourceLimitDto(r.getId(), r.getBaseResourceId(), r.getCampaignId(),
                        findServiceId(r, loadedBaseResourceById, loadedBaseResourceTypesById),
                        prepareAmount(r, resourceTypeByBaseResourceId)))
                .collect(Collectors.toList());
        boolean expandServices = expand.contains(ExpandBaseResourceLimit.PROVIDERS);
        boolean expandSegments = expand.contains(ExpandBaseResourceLimit.SEGMENTS);
        Set<Service> services = expandServices ? serviceReader.readByIds(serviceIds) : Set.of();
        Set<Segment> segments = expandSegments ? segmentReader.readByIds(segmentIds) : Set.of();
        Map<Long, BaseResourceDto> baseResourceById = baseResources.stream().collect(Collectors
                .toMap(BaseResource::getId, r -> new BaseResourceDto(r.getId(), r.getKey(), r.getName(),
                        r.getBaseResourceTypeId(), r.getSegmentIds())));
        Map<Long, DiService> servicesById = services.stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Service::toView));
        Map<Long, DiSegment> segmentsById = segments.stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Segment::toView));
        Map<Long, BaseResourceTypeDto> baseResourceTypesById = baseResourceTypes.stream()
                .collect(Collectors.toMap(BaseResourceType::getId, t -> new BaseResourceTypeDto(t.getId(), t.getKey(),
                        t.getName(), t.getServiceId(), t.getResourceType(), t.getResourceType().getBaseUnit())));
        Map<Long, DiCampaign> campaignById = campaigns.stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Campaign::toView));
        return new BaseResourceLimitsPageDto(baseResourceLimitDto, baseResourceById, baseResourceTypesById,
                servicesById, campaignById, segmentsById,
                baseResourceLimits.isEmpty() ? null : baseResourceLimits.get(baseResourceLimits.size() - 1).getId());
    }

    private long findServiceId(BaseResourceLimit limit, Map<Long, BaseResource> baseResourceById,
                               Map<Long, BaseResourceType> baseResourceTypesById) {
        return baseResourceTypesById.get(baseResourceById.get(limit.getBaseResourceId()).getBaseResourceTypeId())
                .getServiceId();
    }

    private SingleBaseResourceLimitDto prepareSingleDto(BaseResourceLimit baseResourceLimit,
                                                        Set<ExpandBaseResourceLimit> expand) {
        boolean expandBaseResources = expand.contains(ExpandBaseResourceLimit.BASE_RESOURCES);
        BaseResource loadedBaseResource = baseResourceCache.getById(baseResourceLimit.getBaseResourceId())
                .orElseThrow(() -> new RuntimeException("Base resource "
                        + baseResourceLimit.getBaseResourceId() + " not found"));
        Optional<BaseResource> baseResourceO = expandBaseResources ? Optional.of(loadedBaseResource) : Optional.empty();
        boolean expandTypes = expand.contains(ExpandBaseResourceLimit.BASE_RESOURCE_TYPES);
        BaseResourceType loadedBaseResourceType = baseResourceTypeCache
                .getById(loadedBaseResource.getBaseResourceTypeId())
                        .orElseThrow(() -> new RuntimeException("Base resource type "
                                + loadedBaseResource.getBaseResourceTypeId() + " not found"));
        Optional<BaseResourceType> baseResourceTypeO = expandTypes && expandBaseResources
                ? Optional.of(loadedBaseResourceType) : Optional.empty();
        boolean expandCampaigns = expand.contains(ExpandBaseResourceLimit.CAMPAIGNS);
        Optional<Campaign> campaignO = expandCampaigns
                ? Optional.of(campaignCache.getById(baseResourceLimit.getCampaignId())
                .orElseThrow(() -> new RuntimeException("Campaign "
                        + baseResourceLimit.getCampaignId() + " not found")))
                : Optional.empty();
        Hierarchy hierarchy = hierarchySupplier.get();
        ServiceReader serviceReader = hierarchy.getServiceReader();
        SegmentReader segmentReader = hierarchy.getSegmentReader();
        boolean expandServices = expand.contains(ExpandBaseResourceLimit.PROVIDERS);
        boolean expandSegments = expand.contains(ExpandBaseResourceLimit.SEGMENTS);
        Set<Long> segmentIds = baseResourceO.map(BaseResource::getSegmentIds).orElse(Set.of());
        Set<Segment> segments = expandSegments ? segmentReader.readByIds(segmentIds) : Set.of();
        Set<Long> serviceIds = baseResourceTypeO.map(BaseResourceType::getServiceId).stream()
                .collect(Collectors.toSet());
        Set<Service> services = expandServices ? serviceReader.readByIds(serviceIds) : Set.of();
        Map<Long, DiSegment> segmentsById = segments.stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Segment::toView));
        BaseResourceDto baseResourceDto = baseResourceO.map(baseResource -> new BaseResourceDto(baseResource.getId(),
                baseResource.getKey(), baseResource.getName(), baseResource.getBaseResourceTypeId(),
                baseResource.getSegmentIds())).orElse(null);
        BaseResourceTypeDto baseResourceTypeDto = baseResourceTypeO.map(baseResourceType ->
                new BaseResourceTypeDto(baseResourceType.getId(), baseResourceType.getKey(),
                        baseResourceType.getName(), baseResourceType.getServiceId(),
                        baseResourceType.getResourceType(), baseResourceType.getResourceType().getBaseUnit()))
                .orElse(null);
        BaseResourceLimitDto baseResourceLimitDto = new BaseResourceLimitDto(baseResourceLimit.getId(),
                baseResourceLimit.getBaseResourceId(), baseResourceLimit.getCampaignId(),
                loadedBaseResourceType.getServiceId(),
                prepareAmount(baseResourceLimit, loadedBaseResourceType.getResourceType()));
        DiCampaign campaignDto = campaignO.map(Campaign::toView).orElse(null);
        Map<Long, DiService> servicesById = services.stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Service::toView));
        return new SingleBaseResourceLimitDto(baseResourceLimitDto, baseResourceDto, baseResourceTypeDto,
                servicesById, campaignDto, segmentsById);
    }

    private DiAmount prepareAmount(BaseResourceLimit limit, Map<Long, DiResourceType> resourceTypeByBaseResourceId) {
        return DiAmount.of(limit.getLimit(), resourceTypeByBaseResourceId.get(limit.getBaseResourceId()).getBaseUnit());
    }

    private DiAmount prepareAmount(BaseResourceLimit limit, DiResourceType resourceType) {
        return DiAmount.of(limit.getLimit(), resourceType.getBaseUnit());
    }

    private static final class ValidatedPutLimit {

        private final BaseResource baseResource;
        private final BaseResourceType baseResourceType;
        private final long value;

        private ValidatedPutLimit(BaseResource baseResource, BaseResourceType baseResourceType, long value) {
            this.baseResource = baseResource;
            this.baseResourceType = baseResourceType;
            this.value = value;
        }

        public BaseResource getBaseResource() {
            return baseResource;
        }

        public BaseResourceType getBaseResourceType() {
            return baseResourceType;
        }

        public long getValue() {
            return value;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            ValidatedPutLimit that = (ValidatedPutLimit) o;
            return value == that.value &&
                    Objects.equals(baseResource, that.baseResource) &&
                    Objects.equals(baseResourceType, that.baseResourceType);
        }

        @Override
        public int hashCode() {
            return Objects.hash(baseResource, baseResourceType, value);
        }

        @Override
        public String toString() {
            return "ValidatedPutLimit{" +
                    "baseResource=" + baseResource +
                    ", baseResourceType=" + baseResourceType +
                    ", value=" + value +
                    '}';
        }

    }

}
