package ru.yandex.qe.dispenser.ws.base_resources.impl;

import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.inject.Inject;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.MessageSource;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.api.v1.DiSegment;
import ru.yandex.qe.dispenser.api.v1.DiService;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResource;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceType;
import ru.yandex.qe.dispenser.domain.dao.base_resources.BaseResourceDao;
import ru.yandex.qe.dispenser.domain.dao.base_resources.BaseResourceTypeCache;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentReader;
import ru.yandex.qe.dispenser.domain.dao.service.ServiceReader;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.i18n.LocalizableString;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.util.LocalizationUtils;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourceTypeDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.BaseResourcesPageDto;
import ru.yandex.qe.dispenser.ws.base_resources.model.ExpandBaseResource;
import ru.yandex.qe.dispenser.ws.base_resources.model.SingleBaseResourceDto;
import ru.yandex.qe.dispenser.ws.common.domain.errors.ErrorCollection;
import ru.yandex.qe.dispenser.ws.common.domain.errors.TypedError;
import ru.yandex.qe.dispenser.ws.common.domain.result.Result;

@Component
public class BaseResourcesManager {

    private final MessageSource errorMessageSource;
    private final BaseResourceDao baseResourceDao;
    private final BaseResourceTypeCache baseResourceTypeCache;
    private final HierarchySupplier hierarchySupplier;

    @Inject
    public BaseResourcesManager(@Qualifier("errorMessageSource") MessageSource errorMessageSource,
                                BaseResourceDao baseResourceDao,
                                BaseResourceTypeCache baseResourceTypeCache,
                                HierarchySupplier hierarchySupplier) {
        this.errorMessageSource = errorMessageSource;
        this.baseResourceDao = baseResourceDao;
        this.baseResourceTypeCache = baseResourceTypeCache;
        this.hierarchySupplier = hierarchySupplier;
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<BaseResourcesPageDto, ErrorCollection<String, TypedError<String>>> getPage(
            Long from, Integer limit, Locale locale, Set<ExpandBaseResource> expand) {
        if (limit != null && (limit < 0 || limit > 100)) {
            return Result.failure(ErrorCollection.typedStringBuilder().addError("limit",
                    TypedError.badRequest(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                            LocalizableString.of("invalid.limit.value"), locale))).build());
        }
        int actualLimit = limit == null ? 100 : limit;
        Set<ExpandBaseResource> actualExpand = expand != null ? expand : Set.of();
        List<BaseResource> baseResources = baseResourceDao.getPage(from, actualLimit);
        BaseResourcesPageDto result = preparePageDto(baseResources, actualExpand);
        return Result.success(result);
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public Result<SingleBaseResourceDto, ErrorCollection<String, TypedError<String>>> getById(
            long id, Locale locale, Set<ExpandBaseResource> expand) {
        Optional<BaseResource> baseResourceO = baseResourceDao.getById(id);
        if (baseResourceO.isEmpty()) {
            return Result.failure(ErrorCollection.typedStringBuilder().addError(
                    TypedError.notFound(LocalizationUtils.resolveWithDefaultAsKey(errorMessageSource,
                            LocalizableString.of("base.resource.not.found"), locale))).build());
        }
        Set<ExpandBaseResource> actualExpand = expand != null ? expand : Set.of();
        BaseResource baseResource = baseResourceO.get();
        SingleBaseResourceDto result = prepareSingleDto(baseResource, actualExpand);
        return Result.success(result);
    }

    private BaseResourcesPageDto preparePageDto(List<BaseResource> baseResources,
                                                Set<ExpandBaseResource> expand) {
        Set<Long> baseResourceTypeIds = baseResources.stream().map(BaseResource::getBaseResourceTypeId)
                .collect(Collectors.toSet());
        boolean expandTypes = expand.contains(ExpandBaseResource.BASE_RESOURCE_TYPES);
        Set<BaseResourceType> baseResourceTypes = expandTypes
                ? baseResourceTypeCache.getByIds(baseResourceTypeIds) : Set.of();
        Hierarchy hierarchy = hierarchySupplier.get();
        ServiceReader serviceReader = hierarchy.getServiceReader();
        SegmentReader segmentReader = hierarchy.getSegmentReader();
        Set<Long> segmentIds = baseResources.stream().flatMap(r -> r.getSegmentIds().stream())
                .collect(Collectors.toSet());
        Set<Long> serviceIds = baseResourceTypes.stream().map(BaseResourceType::getServiceId)
                .collect(Collectors.toSet());
        List<BaseResourceDto> baseResourceDto = baseResources.stream().map(r -> new BaseResourceDto(r.getId(),
                r.getKey(), r.getName(), r.getBaseResourceTypeId(), r.getSegmentIds())).collect(Collectors.toList());
        boolean expandServices = expand.contains(ExpandBaseResource.PROVIDERS);
        boolean expandSegments = expand.contains(ExpandBaseResource.SEGMENTS);
        Set<Service> services = expandServices ? serviceReader.readByIds(serviceIds) : Set.of();
        Set<Segment> segments = expandSegments ? segmentReader.readByIds(segmentIds) : Set.of();
        Map<Long, DiService> servicesById = services.stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Service::toView));
        Map<Long, DiSegment> segmentsById = segments.stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Segment::toView));
        Map<Long, BaseResourceTypeDto> baseResourceTypesById = baseResourceTypes.stream()
                .collect(Collectors.toMap(BaseResourceType::getId, t -> new BaseResourceTypeDto(t.getId(), t.getKey(),
                        t.getName(), t.getServiceId(), t.getResourceType(), t.getResourceType().getBaseUnit())));
        return new BaseResourcesPageDto(baseResourceDto, baseResourceTypesById, segmentsById,
                servicesById, baseResources.isEmpty() ? null : baseResources.get(baseResources.size() - 1).getId());
    }

    private SingleBaseResourceDto prepareSingleDto(BaseResource baseResource, Set<ExpandBaseResource> expand) {
        boolean expandTypes = expand.contains(ExpandBaseResource.BASE_RESOURCE_TYPES);
        Optional<BaseResourceType> baseResourceTypeO = expandTypes
                ? Optional.of(baseResourceTypeCache.getById(baseResource.getBaseResourceTypeId())
                        .orElseThrow(() -> new RuntimeException("Base resource type "
                                + baseResource.getBaseResourceTypeId() + " not found")))
                : Optional.empty();
        Hierarchy hierarchy = hierarchySupplier.get();
        ServiceReader serviceReader = hierarchy.getServiceReader();
        SegmentReader segmentReader = hierarchy.getSegmentReader();
        boolean expandServices = expand.contains(ExpandBaseResource.PROVIDERS);
        boolean expandSegments = expand.contains(ExpandBaseResource.SEGMENTS);
        Optional<Service> serviceO = expandServices
                ? baseResourceTypeO.map(baseResourceType -> serviceReader.read(baseResourceType.getServiceId()))
                : Optional.empty();
        DiService serviceDto = serviceO.map(Service::toView).orElse(null);
        Set<Segment> segments = expandSegments ? segmentReader.readByIds(baseResource.getSegmentIds()) : Set.of();
        Map<Long, DiSegment> segmentsById = segments.stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Segment::toView));
        BaseResourceDto baseResourceDto = new BaseResourceDto(baseResource.getId(), baseResource.getKey(),
                baseResource.getName(), baseResource.getBaseResourceTypeId(), baseResource.getSegmentIds());
        BaseResourceTypeDto baseResourceTypeDto = baseResourceTypeO.map(baseResourceType ->
                new BaseResourceTypeDto(baseResourceType.getId(), baseResourceType.getKey(),
                        baseResourceType.getName(), baseResourceType.getServiceId(),
                        baseResourceType.getResourceType(), baseResourceType.getResourceType().getBaseUnit()))
                .orElse(null);
        return new SingleBaseResourceDto(baseResourceDto, baseResourceTypeDto, segmentsById, serviceDto);
    }

}
