package ru.yandex.qe.dispenser.ws.base_resources.impl;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.google.common.collect.Sets;
import org.springframework.stereotype.Component;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResource;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceChange;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceChangeByService;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceMapping;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceType;
import ru.yandex.qe.dispenser.domain.base_resources.BigOrderMappings;
import ru.yandex.qe.dispenser.domain.base_resources.CampaignMappings;
import ru.yandex.qe.dispenser.domain.base_resources.ResourceAndSegmentsId;
import ru.yandex.qe.dispenser.domain.base_resources.ServiceBaseResourceChange;
import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.domain.dao.base_resources.BaseResourceCache;
import ru.yandex.qe.dispenser.domain.dao.base_resources.BaseResourceTypeCache;
import ru.yandex.qe.dispenser.domain.dao.base_resources.PerCampaignMappingsCache;
import ru.yandex.qe.dispenser.domain.dao.segment.SegmentReader;
import ru.yandex.qe.dispenser.domain.dao.service.ServiceReader;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.hierarchy.HierarchySupplier;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.ws.bot.BigOrderManager;

@Component
public class BaseResourcesMapper {

    private final PerCampaignMappingsCache perCampaignMappingsCache;
    private final BaseResourceCache baseResourceCache;
    private final BaseResourceTypeCache baseResourceTypeCache;
    private final BigOrderManager bigOrderManager;
    private final HierarchySupplier hierarchySupplier;
    private final BaseResourceRelationRemoteEvaluator remoteEvaluator;

    @Inject
    public BaseResourcesMapper(PerCampaignMappingsCache perCampaignMappingsCache,
                               BaseResourceCache baseResourceCache,
                               BaseResourceTypeCache baseResourceTypeCache,
                               BigOrderManager bigOrderManager,
                               HierarchySupplier hierarchySupplier,
                               BaseResourceRelationRemoteEvaluator remoteEvaluator) {
        this.perCampaignMappingsCache = perCampaignMappingsCache;
        this.baseResourceCache = baseResourceCache;
        this.baseResourceTypeCache = baseResourceTypeCache;
        this.bigOrderManager = bigOrderManager;
        this.hierarchySupplier = hierarchySupplier;
        this.remoteEvaluator = remoteEvaluator;
    }

    public List<BaseResourceChange.Builder> mapQuotaChangeRequestsAsBaseChanges(
            Collection<? extends QuotaChangeRequest> requests, boolean skipRemoteErrors) {
        List<BaseResourceChange.Builder> result = new ArrayList<>();
        if (requests.isEmpty()) {
            return result;
        }
        List<QuotaChangeRequest> requestsWithCampaign = requests.stream().filter(r -> r.getCampaignId() != null)
                .collect(Collectors.toList());
        Map<Long, QuotaChangeRequest> requestsById = requestsWithCampaign.stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Function.identity()));
        Map<Long, Map<Long, Set<QuotaChangeRequest.Change>>> perRequestPerBigOrderChanges = new HashMap<>();
        requestsWithCampaign.forEach(request -> {
            perRequestPerBigOrderChanges.put(request.getId(), request.getChanges().stream()
                    .filter(change -> change.getBigOrder() != null)
                    .collect(Collectors.groupingBy(c -> c.getBigOrder().getId(), Collectors.toSet())));
        });
        Map<Long, Map<Long, Map<Long, Map<ResourceAndSegmentsId, Long>>>> perRequestPerBigOrderMappingParameters
                = new HashMap<>();
        Map<Long, Map<Long, Map<Long, Map<ResourceAndSegmentsId, QuotaChangeRequest.Change>>>>
                perRequestPerBigOrderMappingChanges = new HashMap<>();
        perRequestPerBigOrderChanges.forEach((requestId, perBigOrderChanges) -> {
            QuotaChangeRequest request = requestsById.get(requestId);
            CampaignMappings campaignMappings = perCampaignMappingsCache.getByCampaignId(request.getCampaignId());
            Map<Long, BigOrderMappings> perBigOrderMappings =  campaignMappings.getPerBigOrderMappings();
            perBigOrderChanges.forEach((bigOrderId, changes) -> {
                BigOrderMappings mappings = perBigOrderMappings.get(bigOrderId);
                if (mappings == null) {
                    return;
                }
                changes.forEach(change -> {
                    ResourceAndSegmentsId resourceKey = new ResourceAndSegmentsId(change.getResource().getId(),
                            change.getSegments().stream().map(LongIndexBase::getId).collect(Collectors.toSet()));
                    Set<Long> matchingMappings = mappings.getMappingIdsByResourceId()
                            .getOrDefault(resourceKey, Set.of());
                    matchingMappings.forEach(matchingMappingId -> {
                        perRequestPerBigOrderMappingParameters
                                .computeIfAbsent(requestId,k -> new HashMap<>())
                                .computeIfAbsent(bigOrderId, k -> new HashMap<>())
                                .computeIfAbsent(matchingMappingId, k -> new HashMap<>())
                                .put(resourceKey, change.getAmount());
                        perRequestPerBigOrderMappingChanges
                                .computeIfAbsent(requestId,k -> new HashMap<>())
                                .computeIfAbsent(bigOrderId, k -> new HashMap<>())
                                .computeIfAbsent(matchingMappingId, k -> new HashMap<>())
                                .put(resourceKey, change);
                    });
                });
            });
        });
        Map<Long, Map<Long, Map<Long, BigDecimal>>> perRequestPerBigOrderAppliedMappings = new HashMap<>();
        Map<Long, Map<Long, Map<Long, Map<ResourceAndSegmentsId, QuotaChangeRequest.Change>>>>
                perRequestPerBigOrderMappingChangesRemote = new HashMap<>();
        perRequestPerBigOrderMappingParameters.forEach((requestId, perBigOrderMappingParameters) -> {
            QuotaChangeRequest request = requestsById.get(requestId);
            CampaignMappings campaignMappings = perCampaignMappingsCache.getByCampaignId(request.getCampaignId());
            perBigOrderMappingParameters.forEach((bigOrderId, mappingParameters) -> {
                BigOrderMappings bigOrderMappings = campaignMappings.getPerBigOrderMappings().get(bigOrderId);
                if (bigOrderMappings == null) {
                    return;
                }
                mappingParameters.forEach((mappingId, parameters) -> {
                    BaseResourceMapping mapping = bigOrderMappings.getMappingsById().get(mappingId);
                    if (mapping == null) {
                        return;
                    }
                    if (!mapping.getRelation().isLocalEvaluation()) {
                        perRequestPerBigOrderMappingChangesRemote
                                .computeIfAbsent(requestId, k -> new HashMap<>())
                                .computeIfAbsent(bigOrderId, k -> new HashMap<>())
                                .put(mappingId, perRequestPerBigOrderMappingChanges
                                        .get(requestId).get(bigOrderId).get(mappingId));
                        return;
                    }
                    BigDecimal applicationResult = mapping.getRelation().evaluateLocally(parameters);
                    if (applicationResult.compareTo(BigDecimal.ZERO) != 0) {
                        perRequestPerBigOrderAppliedMappings
                                .computeIfAbsent(requestId, k -> new HashMap<>())
                                .computeIfAbsent(bigOrderId, k -> new HashMap<>())
                                .put(mappingId, applicationResult);
                    }
                });
            });
        });
        Map<Long, Map<Long, Map<Long, BigDecimal>>> perRequestPerBigOrderRemoteAppliedMappings = remoteEvaluator
                .evaluateRemote(requestsById, perRequestPerBigOrderMappingChangesRemote,
                        perCampaignMappingsCache::getByCampaignId, skipRemoteErrors);
        perRequestPerBigOrderRemoteAppliedMappings
                .forEach((requestId, perBigOrderRemoteAppliedMappings) -> perBigOrderRemoteAppliedMappings
                        .forEach((bigOrderId, appliedMappings) -> perRequestPerBigOrderAppliedMappings
                                .computeIfAbsent(requestId, k -> new HashMap<>())
                                .computeIfAbsent(bigOrderId, k -> new HashMap<>())
                                .putAll(appliedMappings)));
        perRequestPerBigOrderAppliedMappings.forEach((requestId, perBigOrderAppliedMappings) -> {
            QuotaChangeRequest request = requestsById.get(requestId);
            CampaignMappings campaignMappings = perCampaignMappingsCache.getByCampaignId(request.getCampaignId());
            perBigOrderAppliedMappings.forEach((bigOrderId, appliedMappings) -> {
                BigOrderMappings bigOrderMappings = campaignMappings.getPerBigOrderMappings().get(bigOrderId);
                if (bigOrderMappings == null) {
                    return;
                }
                Map<Long, Map<Long, BigDecimal>> perBaseResourceAppliedMappings = new HashMap<>();
                appliedMappings.forEach((mappingId, applicationResult) -> {
                    BaseResourceMapping mapping = bigOrderMappings.getMappingsById().get(mappingId);
                    if (mapping == null) {
                        return;
                    }
                    perBaseResourceAppliedMappings.computeIfAbsent(mapping.getBaseResourceId(), k -> new HashMap<>())
                            .put(mappingId, applicationResult);
                });
                perBaseResourceAppliedMappings.forEach((baseResourceId, mappings) -> {
                    BigDecimal totalAmount = BigDecimal.ZERO;
                    Map<Long, Map<Long, BigDecimal>> perServiceAppliedMappings = new HashMap<>();
                    for (Map.Entry<Long, BigDecimal> appliedMapping : mappings.entrySet()) {
                        BaseResourceMapping mapping = bigOrderMappings.getMappingsById().get(appliedMapping.getKey());
                        if (mapping == null || appliedMapping.getValue().compareTo(BigDecimal.ZERO) == 0) {
                            continue;
                        }
                        perServiceAppliedMappings.computeIfAbsent(mapping.getServiceId(), k -> new HashMap<>())
                                .put(appliedMapping.getKey(), appliedMapping.getValue());
                        totalAmount = totalAmount.add(appliedMapping.getValue());
                    }
                    if (totalAmount.compareTo(BigDecimal.ZERO) == 0) {
                        return;
                    }
                    Set<ServiceBaseResourceChange> perServiceChanges = new HashSet<>();
                    perServiceAppliedMappings.forEach((serviceId, appliedByMappingId) -> {
                        Set<Long> mappingIds = new HashSet<>();
                        BigDecimal sum = BigDecimal.ZERO;
                        for (Map.Entry<Long, BigDecimal> appliedMapping : appliedByMappingId.entrySet()) {
                            BaseResourceMapping mapping = bigOrderMappings.getMappingsById().get(appliedMapping.getKey());
                            if (mapping == null || appliedMapping.getValue().compareTo(BigDecimal.ZERO) == 0) {
                                continue;
                            }
                            sum = sum.add(appliedMapping.getValue());
                            mappingIds.add(appliedMapping.getKey());
                        }
                        if (sum.compareTo(BigDecimal.ZERO) == 0) {
                            return;
                        }
                        ServiceBaseResourceChange perServiceChange = ServiceBaseResourceChange.builder()
                                .serviceId(serviceId)
                                .amount(sum.longValueExact())
                                .addMappingIds(mappingIds)
                                .build();
                        perServiceChanges.add(perServiceChange);
                    });
                    BaseResourceChange.Builder baseChange = BaseResourceChange.builder()
                            .quotaRequestId(request.getId())
                            .bigOrderId(bigOrderId)
                            .baseResourceId(baseResourceId)
                            .amount(totalAmount.longValueExact())
                            .amountByService(BaseResourceChangeByService.builder()
                                    .addChanges(perServiceChanges)
                                    .build());
                    result.add(baseChange);
                });
            });
        });
        return result;
    }

    public List<BaseResourceChange.Builder> mapQuotaChangeRequestAsBaseChanges(QuotaChangeRequest request,
                                                                               boolean skipRemoteErrors) {
        List<BaseResourceChange.Builder> result = new ArrayList<>();
        if (request.getCampaignId() == null) {
            return result;
        }
        long campaignId = request.getCampaignId();
        CampaignMappings campaignMappings = perCampaignMappingsCache.getByCampaignId(campaignId);
        Map<Long, BigOrderMappings> perBigOrderMappings =  campaignMappings.getPerBigOrderMappings();
        Map<Long, Set<QuotaChangeRequest.Change>> perBigOrderChanges = request.getChanges().stream()
                .filter(change -> change.getBigOrder() != null)
                .collect(Collectors.groupingBy(c -> c.getBigOrder().getId(), Collectors.toSet()));
        Map<Long, Map<Long, Map<ResourceAndSegmentsId, Long>>> perBigOrderMappingParameters = new HashMap<>();
        Map<Long, Map<Long, Map<ResourceAndSegmentsId, QuotaChangeRequest.Change>>> perBigOrderMappingChanges
                = new HashMap<>();
        perBigOrderChanges.forEach((bigOrderId, changes) -> {
            BigOrderMappings mappings = perBigOrderMappings.get(bigOrderId);
            if (mappings == null) {
                return;
            }
            changes.forEach(change -> {
                ResourceAndSegmentsId resourceKey = new ResourceAndSegmentsId(change.getResource().getId(),
                        change.getSegments().stream().map(LongIndexBase::getId).collect(Collectors.toSet()));
                Set<Long> matchingMappings = mappings.getMappingIdsByResourceId().getOrDefault(resourceKey, Set.of());
                matchingMappings.forEach(matchingMappingId -> {
                    perBigOrderMappingParameters
                            .computeIfAbsent(bigOrderId, k -> new HashMap<>())
                            .computeIfAbsent(matchingMappingId, k -> new HashMap<>())
                            .put(resourceKey, change.getAmount());
                    perBigOrderMappingChanges
                            .computeIfAbsent(bigOrderId, k -> new HashMap<>())
                            .computeIfAbsent(matchingMappingId, k -> new HashMap<>())
                            .put(resourceKey, change);
                });
            });
        });
        Map<Long, Map<Long, BigDecimal>> perBigOrderAppliedMappings = new HashMap<>();
        Map<Long, Map<Long, Map<ResourceAndSegmentsId, QuotaChangeRequest.Change>>> perBigOrderMappingChangesRemote
                = new HashMap<>();
        perBigOrderMappingParameters.forEach((bigOrderId, mappingParameters) -> {
            BigOrderMappings bigOrderMappings = campaignMappings.getPerBigOrderMappings().get(bigOrderId);
            if (bigOrderMappings == null) {
                return;
            }
            mappingParameters.forEach((mappingId, parameters) -> {
                BaseResourceMapping mapping = bigOrderMappings.getMappingsById().get(mappingId);
                if (mapping == null) {
                    return;
                }
                if (!mapping.getRelation().isLocalEvaluation()) {
                    perBigOrderMappingChangesRemote.computeIfAbsent(bigOrderId, k -> new HashMap<>())
                            .put(mappingId, perBigOrderMappingChanges.get(bigOrderId).get(mappingId));
                    return;
                }
                BigDecimal applicationResult = mapping.getRelation().evaluateLocally(parameters);
                if (applicationResult.compareTo(BigDecimal.ZERO) != 0) {
                    perBigOrderAppliedMappings.computeIfAbsent(bigOrderId, k -> new HashMap<>())
                            .put(mappingId, applicationResult);
                }
            });
        });
        Map<Long, Map<Long, BigDecimal>> perBigOrderRemoteAppliedMappings = remoteEvaluator
                .evaluateRemote(request, perBigOrderMappingChangesRemote, campaignMappings, skipRemoteErrors);
        perBigOrderRemoteAppliedMappings.forEach((bigOrderId, appliedMappings) -> perBigOrderAppliedMappings
                .computeIfAbsent(bigOrderId, k -> new HashMap<>()).putAll(appliedMappings));
        perBigOrderAppliedMappings.forEach((bigOrderId, appliedMappings) -> {
            BigOrderMappings bigOrderMappings = campaignMappings.getPerBigOrderMappings().get(bigOrderId);
            if (bigOrderMappings == null) {
                return;
            }
            Map<Long, Map<Long, BigDecimal>> perBaseResourceAppliedMappings = new HashMap<>();
            appliedMappings.forEach((mappingId, applicationResult) -> {
                BaseResourceMapping mapping = bigOrderMappings.getMappingsById().get(mappingId);
                if (mapping == null) {
                    return;
                }
                perBaseResourceAppliedMappings.computeIfAbsent(mapping.getBaseResourceId(), k -> new HashMap<>())
                        .put(mappingId, applicationResult);
            });
            perBaseResourceAppliedMappings.forEach((baseResourceId, mappings) -> {
                BigDecimal totalAmount = BigDecimal.ZERO;
                Map<Long, Map<Long, BigDecimal>> perServiceAppliedMappings = new HashMap<>();
                for (Map.Entry<Long, BigDecimal> appliedMapping : mappings.entrySet()) {
                    BaseResourceMapping mapping = bigOrderMappings.getMappingsById().get(appliedMapping.getKey());
                    if (mapping == null || appliedMapping.getValue().compareTo(BigDecimal.ZERO) == 0) {
                        continue;
                    }
                    perServiceAppliedMappings.computeIfAbsent(mapping.getServiceId(), k -> new HashMap<>())
                            .put(appliedMapping.getKey(), appliedMapping.getValue());
                    totalAmount = totalAmount.add(appliedMapping.getValue());
                }
                if (totalAmount.compareTo(BigDecimal.ZERO) == 0) {
                    return;
                }
                Set<ServiceBaseResourceChange> perServiceChanges = new HashSet<>();
                perServiceAppliedMappings.forEach((serviceId, appliedByMappingId) -> {
                    Set<Long> mappingIds = new HashSet<>();
                    BigDecimal sum = BigDecimal.ZERO;
                    for (Map.Entry<Long, BigDecimal> appliedMapping : appliedByMappingId.entrySet()) {
                        BaseResourceMapping mapping = bigOrderMappings.getMappingsById().get(appliedMapping.getKey());
                        if (mapping == null || appliedMapping.getValue().compareTo(BigDecimal.ZERO) == 0) {
                            continue;
                        }
                        sum = sum.add(appliedMapping.getValue());
                        mappingIds.add(appliedMapping.getKey());
                    }
                    if (sum.compareTo(BigDecimal.ZERO) == 0) {
                        return;
                    }
                    ServiceBaseResourceChange perServiceChange = ServiceBaseResourceChange.builder()
                            .serviceId(serviceId)
                            .amount(sum.longValueExact())
                            .addMappingIds(mappingIds)
                            .build();
                    perServiceChanges.add(perServiceChange);
                });
                BaseResourceChange.Builder baseChange = BaseResourceChange.builder()
                        .quotaRequestId(request.getId())
                        .bigOrderId(bigOrderId)
                        .baseResourceId(baseResourceId)
                        .amount(totalAmount.longValueExact())
                        .amountByService(BaseResourceChangeByService.builder()
                                .addChanges(perServiceChanges)
                                .build());
                result.add(baseChange);
            });
        });
        return result;
    }

    public QuotaChangeRequestBaseResources expandBaseResourceChanges(Set<BaseResourceChange> baseChanges) {
        Set<BigOrderBaseResourceAmount> amounts = new HashSet<>();
        if (baseChanges.isEmpty()) {
            return new QuotaChangeRequestBaseResources(amounts);
        }
        Set<Long> bigOrderIds = baseChanges.stream().map(BaseResourceChange::getBigOrderId).collect(Collectors.toSet());
        Set<Long> baseResourceIds = baseChanges.stream().map(BaseResourceChange::getBaseResourceId)
                .collect(Collectors.toSet());
        Map<Long, BigOrder> bigOrdersById = bigOrderManager.getByIdsCached(bigOrderIds).stream()
                .collect(Collectors.toMap(BigOrder::getId, Function.identity()));
        Map<Long, BaseResource> baseResourcesById = baseResourceCache.getByIds(baseResourceIds).stream()
                .collect(Collectors.toMap(BaseResource::getId, Function.identity()));
        Set<Long> baseResourceTypeIds = baseResourcesById.values().stream().map(BaseResource::getBaseResourceTypeId)
                .collect(Collectors.toSet());
        Map<Long, BaseResourceType> baseResourceTypesById = baseResourceTypeCache.getByIds(baseResourceTypeIds)
                .stream().collect(Collectors.toMap(BaseResourceType::getId, Function.identity()));
        Hierarchy hierarchy = hierarchySupplier.get();
        ServiceReader serviceReader = hierarchy.getServiceReader();
        SegmentReader segmentReader = hierarchy.getSegmentReader();
        Set<Long> baseResourceSegmentIds = baseResourcesById.values().stream().flatMap(v -> v.getSegmentIds().stream())
                .collect(Collectors.toSet());
        Map<Long, Segment> segmentsById = segmentReader.readByIds(baseResourceSegmentIds).stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Function.identity()));
        Set<Long> serviceIds = new HashSet<>();
        baseResourceTypesById.values().forEach(t -> serviceIds.add(t.getServiceId()));
        baseChanges.forEach(c -> serviceIds.addAll(c.getAmountByService().getChangesByService().keySet()));
        Map<Long, Service> servicesById = serviceReader.readByIds(serviceIds).stream()
                .collect(Collectors.toMap(LongIndexBase::getId, Function.identity()));
        baseChanges.forEach(baseChange -> {
            BaseResource baseResource = baseResourcesById.get(baseChange.getBaseResourceId());
            BaseResourceType baseResourceType = baseResourceTypesById.get(baseResource.getBaseResourceTypeId());
            BigOrder bigOrder = bigOrdersById.get(baseChange.getBigOrderId());
            Set<Segment> baseResourceSegments = baseResource.getSegmentIds().stream().map(segmentsById::get)
                    .collect(Collectors.toSet());
            Service service = servicesById.get(baseResourceType.getServiceId());
            Set<PerServiceBaseResourceAmount> perServiceAmounts = new HashSet<>();
            baseChange.getAmountByService().getChanges().forEach(serviceChange -> {
                Service changeService = servicesById.get(serviceChange.getServiceId());
                perServiceAmounts.add(new PerServiceBaseResourceAmount(changeService, serviceChange.getAmount(),
                        serviceChange.getMappingIds()));
            });
            BigOrderBaseResourceAmount amount = new BigOrderBaseResourceAmount(bigOrder, baseResource, baseResourceType,
                    baseResourceSegments, service, baseChange.getAmount(), perServiceAmounts);
            amounts.add(amount);
        });
        return new QuotaChangeRequestBaseResources(amounts);
    }

    public void splitBaseChangesUpdate(Set<BaseResourceChange> currentBaseChanges,
                                       List<BaseResourceChange.Builder> newBaseChanges,
                                       List<BaseResourceChange.Builder> baseChangesToCreate,
                                       List<BaseResourceChange.Update> baseChangesToUpdate,
                                       Set<BaseResourceChange> unmodifiedBaseChanges,
                                       Set<Long> baseChangesToDelete) {
        Map<BaseResourceChange.Key, BaseResourceChange> currentChangesByKey = currentBaseChanges.stream()
                .collect(Collectors.toMap(BaseResourceChange::toKey, Function.identity()));
        Map<BaseResourceChange.Key, BaseResourceChange.Builder> newChangesByKey = newBaseChanges.stream()
                .collect(Collectors.toMap(BaseResourceChange.Builder::toKey, Function.identity()));
        Set<BaseResourceChange.Key> keysToDelete = Sets.difference(currentChangesByKey.keySet(),
                newChangesByKey.keySet());
        keysToDelete.forEach(k -> baseChangesToDelete.add(currentChangesByKey.get(k).getId()));
        Set<BaseResourceChange.Key> keysToCreate = Sets.difference(newChangesByKey.keySet(),
                currentChangesByKey.keySet());
        keysToCreate.forEach(k -> baseChangesToCreate.add(newChangesByKey.get(k)));
        Set<BaseResourceChange.Key> keysToUpdate = Sets.intersection(currentChangesByKey.keySet(),
                newChangesByKey.keySet());
        keysToUpdate.forEach(k -> {
            BaseResourceChange currentChange = currentChangesByKey.get(k);
            BaseResourceChange updatedChange = newChangesByKey.get(k).build(currentChange.getId());
            if (!currentChange.equals(updatedChange)) {
                baseChangesToUpdate.add(BaseResourceChange.update(currentChange, updatedChange));
            } else {
                unmodifiedBaseChanges.add(currentChange);
            }
        });
    }

}
