package ru.yandex.qe.dispenser.ws.base_resources.model;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import io.swagger.annotations.ApiModel;
import io.swagger.annotations.ApiModelProperty;

import ru.yandex.qe.dispenser.api.v1.DiSegment;
import ru.yandex.qe.dispenser.api.v1.DiService;

@JsonIgnoreProperties(ignoreUnknown = true)
@ApiModel("Single base resource")
public final class SingleBaseResourceDto {

    private final BaseResourceDto resource;
    private final BaseResourceTypeDto resourceType;
    private final Map<Long, DiSegment> segments;
    private final DiService provider;

    @JsonCreator
    public SingleBaseResourceDto(@JsonProperty("resource") BaseResourceDto resource,
                                 @JsonProperty("resourceType") BaseResourceTypeDto resourceType,
                                 @JsonProperty("segments") Map<Long, DiSegment> segments,
                                 @JsonProperty("provider") DiService provider) {
        this.resource = resource;
        this.resourceType = resourceType;
        this.segments = segments;
        this.provider = provider;
    }

    @ApiModelProperty("Base resource")
    public BaseResourceDto getResource() {
        return resource;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @ApiModelProperty("Base resource type referenced by this base resource")
    public Optional<BaseResourceTypeDto> getResourceType() {
        return Optional.ofNullable(resourceType);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @ApiModelProperty("Resource segments referenced by the base resource, indexed by segment id")
    public Map<Long, DiSegment> getSegments() {
        return segments;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    @ApiModelProperty("Provider referenced by base resource type")
    public Optional<DiService> getProvider() {
        return Optional.ofNullable(provider);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        SingleBaseResourceDto that = (SingleBaseResourceDto) o;
        return Objects.equals(resource, that.resource) &&
                Objects.equals(resourceType, that.resourceType) &&
                Objects.equals(segments, that.segments) &&
                Objects.equals(provider, that.provider);
    }

    @Override
    public int hashCode() {
        return Objects.hash(resource, resourceType, segments, provider);
    }

    @Override
    public String toString() {
        return "SingleBaseResourceDto{" +
                "resource=" + resource +
                ", resourceType=" + resourceType +
                ", segments=" + segments +
                ", provider=" + provider +
                '}';
    }

}
