package ru.yandex.qe.dispenser.ws.billing;

import com.google.common.base.Stopwatch;
import io.github.resilience4j.core.IntervalFunction;
import io.github.resilience4j.retry.Retry;
import io.github.resilience4j.retry.RetryConfig;
import org.springframework.context.annotation.Profile;
import org.springframework.stereotype.Component;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.qe.dispenser.solomon.SolomonHolder;
import ru.yandex.qe.dispenser.ws.iam.IAMDestination;

import javax.inject.Inject;
import javax.ws.rs.ServerErrorException;
import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

@Component
@Profile("billing-integration")
public class BillingApiHelper {

    private static final String ALL = "all";
    private static final String RATE = "http_outgoing.requests_rate";
    private static final String TIME = "http_outgoing.requests_time_millis";
    private static final String ENDPOINT = "endpoint";
    private static final String CODE = "code";
    private static final String CODE_NON_2XX = "non_2xx";
    private static final String DESTINATION = "destination";
    private static final String BILLING_API = "billing-api";

    private final BillingApi billingApi;
    private final IAMDestination iamDestination;
    private final Rate totalRate;
    private final Rate totalNon2xxRate;
    private final Histogram totalDuration;
    private final Retry retry;

    @Inject
    public BillingApiHelper(BillingApi billingApi, IAMDestination iamDestination, SolomonHolder solomonHolder) {
        this.billingApi = billingApi;
        this.iamDestination = iamDestination;
        MetricRegistry registry = solomonHolder.getRootRegistry();
        this.totalRate = registry.rate(RATE, Labels.of(CODE, ALL, ENDPOINT, ALL, DESTINATION, BILLING_API));
        this.totalNon2xxRate = registry.rate(RATE, Labels.of(CODE, CODE_NON_2XX, ENDPOINT, ALL, DESTINATION, BILLING_API));
        this.totalDuration = registry.histogramRate(TIME, Labels.of(CODE, ALL, ENDPOINT, ALL, DESTINATION, BILLING_API),
                Histograms.exponential(22, 2.0d, 1.0d));
        this.retry = Retry.of("billingHttpRetry", RetryConfig.custom()
                .maxAttempts(3)
                .intervalFunction(IntervalFunction.ofExponentialRandomBackoff(500, 1.5, 0.5))
                .retryOnException(e -> e instanceof SocketTimeoutException || e instanceof ConnectException
                        || e instanceof ServerErrorException)
                .build());
    }

    public BillingDryRunResponse dryRun(BillingDryRunRequest request) {
        return retry.executeSupplier(() -> iamDestination.runAuthorized(() -> measure(() -> billingApi.dryRun(request))));
    }

    private <T> T measure(Supplier<T> supplier) {
        Stopwatch stopwatch = Stopwatch.createStarted();
        totalRate.inc();
        boolean success = false;
        try {
            T result = supplier.get();
            success = true;
            return result;
        } finally {
            if (!success) {
                totalNon2xxRate.inc();
            }
            stopwatch.stop();
            long elapsedMillis = stopwatch.elapsed(TimeUnit.MILLISECONDS);
            totalDuration.record(elapsedMillis);
        }
    }

}
